/*
 * Projeto: Circus Refine
 * 
 * Autores: Alessandro Gurgel <alessandro87@consiste.dimap.ufrn.br>
 * 			Cristiano Castro  <crisgc@consiste.dimap.ufrn.br>
 */
package circusRefine.core.crules.parseArgument;

import java.io.StringWriter;
import java.util.List;

import net.sourceforge.czt.base.ast.Term;
import net.sourceforge.czt.base.visitor.TermVisitor;
import net.sourceforge.czt.circus.ast.InterleaveAction;
import net.sourceforge.czt.circus.ast.NameSet;
import net.sourceforge.czt.circus.visitor.InterleaveActionVisitor;
import circusRefine.core.util.ChildrenTermExtractor;

/**
 * Um Visitor para retornar um NameSet a partir da AST. Usado no
 * parse de Argumentos
 *  
 * @author Cristiano Gurgel
 */
public class NameSetGetter implements TermVisitor<NameSet>, 
		InterleaveActionVisitor<NameSet> {

	/**
	 * Retorna um NameSet a partir de uma AST, usando, para isso,
	 * uma instncia de {@link NameSetGetter}.
	 * 
	 * @param ast a AST a ser visitada
	 * @return um NameSet contido na AST
	 */
	public static NameSet getANameSet(Term ast) {
		NameSetGetter getter = new NameSetGetter();
		NameSet result = ast.accept(getter);
		return result;
	}
	
	
	/** Log para a documentao */
	private StringWriter log;
	
	/**
	 * Construtor Padro. Somente cria um log para o objeto
	 */
	public NameSetGetter() { 
		this.setLog(new StringWriter());
		this.addMessage("Novo log do NameSetGetter: " + this);
	}
	
	
	/**
	 * Acessa o log da classe
	 * 
	 * @return a informacao da classe
	 */
	public String getInfo() {
		return this.getLog().toString();
	}
	
	private StringWriter getLog() {
		return log;
	}
	
	private void setLog(StringWriter log) {
		this.log = log;
	}
	
	
	/**
	 * Adiciona uma mensagem ao log de unificacao
	 * 
	 * @param msg a mensagem a ser adicionada
	 */
	private void addMessage(String msg) {
		this.getLog().append(msg + "\n");
	}	
	
	/**
	 * Percorre a rvore procurando uma instncia de 
	 * {@link NameSet}.
	 * 
	 * @param arg0 a AST a ser visitada.
	 * @return o {@link NameSet} encontrado na rvore ou 
	 * 	<code>null</code> caso esse no seja encontrado.
	 */
	public NameSet visitTerm(Term arg0) {
		this.addMessage("Visitando termo: " + arg0);
		List<Term> filhos = ChildrenTermExtractor.extrairFilhos(arg0);
		
		for (Term filho : filhos) {
			NameSet result = filho.accept(this);
			if (result != null) {
				return result;
			}
		}
		
		/* No encontrei o NameSet na Ast*/
		return null;
	}

	/**
	 * Caso encontre uma instncia de {@link InterleaveAction}, o seu
	 *  {@link NameSet} esquerdo  retornado.
	 * 
	 * @param arg0 o InterleaveAction a ser visitado.
	 * @return o NameSet esquerdo do {@link InterleaveAction}.
	 */
	public NameSet visitInterleaveAction(InterleaveAction arg0) {
		this.addMessage("Visitando um InterleaveAction: " + arg0);
		return arg0.getLeftNameSet();
	}

}
