/*
 * Projeto: Circus Refine
 */
package circusRefine.core.crules.parseArgument;

import java.io.StringWriter;
import java.util.List;

import circusRefine.core.util.ChildrenTermExtractor;
import net.sourceforge.czt.base.ast.Term;
import net.sourceforge.czt.base.visitor.TermVisitor;
import net.sourceforge.czt.z.ast.NameList;
import net.sourceforge.czt.z.ast.VarDecl;
import net.sourceforge.czt.z.visitor.VarDeclVisitor;

/**
 * Um Visitor para retornar um NameList a partir da AST. Usado no
 * parse de Argumentos
 *  
 * @author Cristiano Gurgel
 */
public class NameListGetter implements TermVisitor<NameList>, 
		VarDeclVisitor<NameList> {

	/**
	 * Retorna um NameList a partir de uma AST, usando, para isso,
	 * uma instncia de {@link NameListGetter}.
	 * 
	 * @param ast a AST a ser visitada
	 * @return um NameList contido na AST
	 */
	public static NameList getANameList(Term ast) {
		NameListGetter getter = new NameListGetter();
		NameList result = ast.accept(getter);
		return result;
	}
	
	
	/** Log para a documentao */
	private StringWriter log;
	
	/**
	 * Construtor Padro. Somente cria um log para o objeto
	 */
	public NameListGetter() { 
		this.setLog(new StringWriter());
		this.addMessage("Novo log do NameListGetter: " + this);
	}
	
	
	/**
	 * Acessa o log da classe
	 * 
	 * @return a informacao da classe
	 */
	public String getInfo() {
		return this.getLog().toString();
	}
	
	private StringWriter getLog() {
		return log;
	}
	
	private void setLog(StringWriter log) {
		this.log = log;
	}
	
	
	/**
	 * Adiciona uma mensagem ao log de unificacao
	 * 
	 * @param msg a mensagem a ser adicionada
	 */
	private void addMessage(String msg) {
		this.getLog().append(msg + "\n");
	}	
	
	/**
	 * Percorre a rvore procurando uma instncia de 
	 * {@link NameList}.
	 * 
	 * @param arg0 a AST a ser visitada.
	 * @return o {@link NameList} encontrado na rvore ou 
	 * 	<code>null</code> caso esse no seja encontrado.
	 */
	public NameList visitTerm(Term arg0) {
		this.addMessage("Visitando termo: " + arg0);
		List<Term> filhos = ChildrenTermExtractor.extrairFilhos(arg0);
		
		for (Term filho : filhos) {
			NameList result = filho.accept(this);
			if (result != null) {
				return result;
			}
		}
		
		/* No encontrei a NameList na Ast*/
		return null;
	}

	/**
	 * Caso encontre uma instncia de VarDecl, o {@link NameList} da
	 * 	mesma  retornado
	 * 
	 * @param arg0 o VarDecl a ser visitado.
	 * @return o NameList encontrado
	 */
	public NameList visitVarDecl(VarDecl arg0) {
		this.addMessage("Visitando um VarDecl: " + arg0);
		return arg0.getNameList();
	}

}
