/*
 * Projeto: Circus Refine
 * 
 * Autores: Alessandro Gurgel <alessandro87@consiste.dimap.ufrn.br>
 * 			Cristiano Castro  <crisgc@consiste.dimap.ufrn.br>
 */
package circusRefine.core.crules.factories;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;


import net.sourceforge.czt.base.ast.Term;
import net.sourceforge.czt.circus.ast.ActionTransformerPred;
import net.sourceforge.czt.circus.ast.CircusNameSet;
import net.sourceforge.czt.circus.ast.GuardedAction;
import net.sourceforge.czt.circus.ast.ParallelAction;
import net.sourceforge.czt.circus.ast.SchExprAction;
import net.sourceforge.czt.circus.ast.SeqAction;
import net.sourceforge.czt.circus.ast.Transformation;
import net.sourceforge.czt.circus.util.CircusUtils;
import net.sourceforge.czt.circuspatt.ast.JokerAction;
import net.sourceforge.czt.circuspatt.ast.JokerChannelSet;
import net.sourceforge.czt.circuspatt.util.CircusLaw;
import net.sourceforge.czt.circuspatt.util.CircusPattUtils;
import net.sourceforge.czt.oz.ast.OzFactory;
import net.sourceforge.czt.oz.ast.PredExpr;
import net.sourceforge.czt.oz.impl.OzFactoryImpl;
import net.sourceforge.czt.z.ast.And;
import net.sourceforge.czt.z.ast.AndExpr;
import net.sourceforge.czt.z.ast.AndPred;
import net.sourceforge.czt.z.ast.ApplExpr;
import net.sourceforge.czt.z.ast.DecorExpr;
import net.sourceforge.czt.z.ast.Expr;
import net.sourceforge.czt.z.ast.ExprList;
import net.sourceforge.czt.z.ast.ExprPred;
import net.sourceforge.czt.z.ast.IffPred;
import net.sourceforge.czt.z.ast.MemPred;
import net.sourceforge.czt.z.ast.OrExpr;
import net.sourceforge.czt.z.ast.PreExpr;
import net.sourceforge.czt.z.ast.Pred;
import net.sourceforge.czt.z.ast.RefExpr;
import net.sourceforge.czt.z.ast.SchExpr;
import net.sourceforge.czt.z.ast.SchText;
import net.sourceforge.czt.z.ast.SetExpr;
import net.sourceforge.czt.z.ast.TupleExpr;
import net.sourceforge.czt.z.ast.ZExprList;
import net.sourceforge.czt.z.ast.ZName;
import net.sourceforge.czt.z.ast.ZSchText;
import net.sourceforge.czt.z.util.ZString;
import net.sourceforge.czt.zpatt.ast.JokerDeclList;
import net.sourceforge.czt.zpatt.ast.JokerExpr;
import net.sourceforge.czt.zpatt.ast.JokerPred;
import circusRefine.core.InternalManager;
import circusRefine.core.astmodifiers.ActionArgumentAnn;
import circusRefine.core.crules.LawDefinitionAnn;
import circusRefine.core.crules.anotations.NormalizeApplAnn;
import circusRefine.core.crules.anotations.ReduceStepAnn;
import circusRefine.core.crules.anotations.SeparateDashedDeclListAnn;
import circusRefine.core.crules.anotations.SeparateDeclListAnn;
import circusRefine.core.crules.anotations.UnifyDeclListAnn;
import circusRefine.core.crules.anotations.XiRuleApplicationAnn;
import circusRefine.core.crules.anotations.XiRuleApplicationLeftAnn;
import circusRefine.core.crules.anotations.op.RearrangeDeclListPOMounter;
import circusRefine.core.crules.utils.LawTypeAnnUtils;
import circusRefine.core.opsdischarge.OPsDischargeUtils;
import circusRefine.util.Internacional;

/**
 * Fabrica para criação de leis que operam em esquemas
 * 
 * @author Cristiano Castro
 * @author alessandro87
 */
public class SchemaLawsFactory extends LawsFactory {

	public SchemaLawsFactory(Internacional inter, InternalManager gerInt) {
		super(inter, gerInt);
	}
	
	@Override
	public List<CircusLaw> createAll() {
		List<CircusLaw> result = new LinkedList<CircusLaw>();
		
		
		/* C.70 */
		result.add(this.createSchemaDisjunctionElimination());
		
		/* C.72 */
		result.add(this.createInitialisationSchemaSequenceIntrodution());
		
		/* C-73 */
		result.add(this.createSchemasParallelismCompositionDistribution());
		
		/* Transforma o predicado de esquemas */
		result.add(this.createLawTransformPredicate());
		
		/* Para a normalização */
		result.add(this.createLawNormalize());
		result.add(this.createLawReduceStep());
		
		/* Tranformar um esquema em sua composição */
		result.add(this.createLawSchemaDecomposition1());
		
		/* Decompor um esquema com uma conjunção como predicado */
		result.add(this.createLawSchemaDecomposition2());
		
		/* Aplicação da definição \Xi de um esquema */
		result.add(this.createRuleXiSchema());
		
		/* Elemento neutro da operação and de esquemas */
		result.add(this.createRuleLandZero());
		
		/* Rearranjar lista de declara��es */
		result.add(this.createRuleRearrangeDeclList());
		
		/* Rearranjar lista de declara��es (sem predicado) */
		result.add(this.createRuleRearrangeDeclListWithoutPred());
		
		/* Lei para tranformar um esquema em uma aplicação de \Xi */
		result.add(this.createRuleXiSchemaLeftDirection());
		
		result.add(this.createSchemaRefinement());
		
		
		return result;
	}
	
	
/**
 * 	
 * @return
 */
public CircusLaw createSchemaRefinement(){
		
		// C.75 
		//SExp1 [- SExpr2
		
		JokerAction right  = this.getFactory().createJokerAction("SExpr1", null);
		JokerAction left = this.getFactory().createJokerAction("SExpr2", null);
		
		ActionTransformerPred trans = factory.createActionTransformerPred(null, 
				Transformation.Refinement,CircusUtils.DEFAULT_REFINEMENT_MODEL,
				Arrays.asList(left, right));
		
		//Ops
		
		List<Pred> ops = new ArrayList<Pred>();
		
		/* Criando a Lei */
		String nome = this.createName("SchemaRefinement");
		CircusLaw result = CircusPattUtils.createCircusLaw(nome, trans, ops);

		LawNumberAnn id = new LawNumberAnn("C.75", nome);
		result.getAnns().add(id);
		LawTypeAnnUtils.insertAnnLawType(result,LawType.ACTION_REFINEMENT_SCHEMA);
		return result;
	}
	

	
	
	
	/**
	 * Retorna o nome composto para a lei circus
	 * 
	 * @param cod o c�digo para o nome da lei
	 * @return uma String com o nome constru�do da lei
	 */
	protected String createName(String cod) {
		return this.retornarMsg(LawType.ACTION_REFINEMENT_SCHEMA, cod);
	}
	
	
	/**
     * Cria lei Schemas Disjunction Elimination (C-70)
     * 
     * @return a nova lei de circus
     */
    public CircusLaw createSchemaDisjunctionElimination() {
    	
    	/* LHS */
    	JokerExpr exp1 = this.getFactory().createJokerExpr("SExp1", 
    			null);
    	JokerExpr exp2 = this.getFactory().createJokerExpr("SExp2", 
    			null);
    	
    	OrExpr orexpr = this.getFactory().createOrExpr(Arrays.asList(exp1,exp2));
    	PreExpr preSexp1 = factory.createPreExpr(exp1);
    	
    	ExprPred exprPred = this.getFactory().createExprPred(preSexp1);
    	SchExprAction sExpr = 
    		this.getFactory().createSchExprAction(orexpr);

    	GuardedAction left = this.factory.createGuardedAction(sExpr, exprPred);
    	
    	/* RHS */
    	
    	SchExprAction action = this.getFactory().createSchExprAction(preSexp1);
    	GuardedAction right = this.getFactory().createGuardedAction(action, exprPred);
    	
    	/* Criando o transformer */
    	ActionTransformerPred trans = factory.createActionTransformerPred(null, 
    			Transformation.Equivalence,
    			CircusUtils.DEFAULT_REFINEMENT_MODEL, 
    			Arrays.asList(left, right));

    	/* Criando as obriga��es de prova */
    	List<Pred> ops = new ArrayList<Pred>();
    	
    	/* Criando a Lei */
    	String nome = this.createName("COD0679");
    	CircusLaw result = CircusPattUtils.createCircusLaw(nome, trans, ops);

    	LawNumberAnn id = new LawNumberAnn("C-70", nome);
    	result.getAnns().add(id);
    	LawTypeAnnUtils.insertAnnLawType(result, LawType.ACTION_REFINEMENT_SCHEMA);
    	
    	return result;
    	
    }
	
	
    
    
	/**
     * Cria lei Initialisation schema/Sequence -- introduction (C-72)
     * 
     * @return a nova lei de circus
     */
    public CircusLaw createInitialisationSchemaSequenceIntrodution() {
    	
    	/* LHS */
    	JokerDeclList listaDecl = 
    		this.getFactory().createJokerDeclList("dl", null);
    	JokerPred p1 = this.getFactory().createJokerPred("p1", null);
    	JokerPred p2 = this.getFactory().createJokerPred("p2", null);
    	AndPred pred = 
    		this.getFactory().createAndPred(Arrays.asList(p1, p2), 
    				And.Wedge);
    	SchText texto = 
    		this.getFactory().createZSchText(listaDecl, pred);
    	SchExpr expr = this.getFactory().createSchExpr(texto);
    	SchExprAction left = this.getFactory().createSchExprAction(expr);
    	
    	/* RHS */
    	JokerDeclList ld1 = 
    		this.getFactory().createJokerDeclList("dl1", null);
    	JokerDeclList ld2 = 
    		this.getFactory().createJokerDeclList("dl2", null);
    	
    	/* Anotação para a aplicação da lei */
    	ld1.getAnns().add(new SeparateDashedDeclListAnn(listaDecl, ld1, ld2));
    	
    	SchText texto2 = this.getFactory().createZSchText(ld1, p1);
    	SchText texto3 = this.getFactory().createZSchText(ld2, p2);
    	SchExpr expr2 = this.getFactory().createSchExpr(texto2);
    	SchExpr expr3 = this.getFactory().createSchExpr(texto3);
    	SchExprAction r1 = this.getFactory().createSchExprAction(expr2);
    	SchExprAction r2 = this.getFactory().createSchExprAction(expr3);
    	SeqAction right = 
    		this.getFactory().createSeqAction(Arrays.asList(r1, r2));
    	
    	/* Criando o transformer */
    	ActionTransformerPred trans = factory.createActionTransformerPred(null, 
    			Transformation.Equivalence,
    			CircusUtils.DEFAULT_REFINEMENT_MODEL, 
    			Arrays.asList(left, right));

    	/* Criando as obriga��es de prova */
    	List<Pred> ops = new ArrayList<Pred>();
    	
    	/* 
    	 * *********************************************
    	 * OP1: \alpha(S_1) \cap \alpha(S_2) = \emptyset
    	 * *********************************************
    	 */
    	
    	/* Monta a refer�ncia � função \alpha */
    	ZName nomeAlpha = this.getFactory().createZName(OPsDischargeUtils.ALPHA,
    			this.getFactory().createZStrokeList(), null);
    	RefExpr funAlpha = this.getFactory().createRefExpr(nomeAlpha, 
    			this.getFactory().createZExprList(), false, false);
    	
    	/* Monta o argumento S1 */
    	ZSchText textoArgS1Stroke = this.getFactory().createZSchText(ld1, 
    			this.getFactory().createTruePred());
    	SchExpr argS1Stroke = this.getFactory().createSchExpr(textoArgS1Stroke);
    	
    	/* Monta a aplicação \alpha(S1) */
    	ApplExpr applAlphaS1 = 
    		this.getFactory().createApplExpr(Arrays.asList(funAlpha, 
    				argS1Stroke), false);
    	
    	/* Monta o argumento S2 */
    	ZSchText textoArgS2Stroke = this.getFactory().createZSchText(ld2, 
    			this.getFactory().createTruePred());
    	SchExpr argS2Stroke = this.getFactory().createSchExpr(textoArgS2Stroke);

    	/* Monta a aplicação \alpha(S2) */
    	ApplExpr applAlphaS2 = 
    		this.getFactory().createApplExpr(Arrays.asList(funAlpha, 
    				argS2Stroke), false);
    	
    	/* Monta � referencia � função interseção */
    	ZName nomeIntersecao = this.getFactory().createZName(
    			OPsDischargeUtils.CAP, this.getFactory().createZStrokeList(), 
    			null);
    	RefExpr funIntersecao = this.getFactory().createRefExpr(nomeIntersecao, 
    			this.getFactory().createZExprList(), false, false);
    	
    	/* Monta a lista de argumentos "(\alpha(S1), \alpha(S2))" */
    	ZExprList listaArgsAlphaS1AlphaS2 = 
    		this.getFactory().createZExprList(Arrays.asList(applAlphaS1, 
    				applAlphaS2));
    	TupleExpr argsAlphaS1AlphaS2 = 
    		this.getFactory().createTupleExpr(listaArgsAlphaS1AlphaS2);
    	
    	/* 
    	 * Monta a aplicação da função Interseção 
    	 * "\alpha(S_1) \cap \alpha(S_2)" 
    	 */
    	ApplExpr applIntersecaoAlphaS1AlphaS2 = 
    		this.getFactory().createApplExpr(Arrays.asList(funIntersecao, 
    				argsAlphaS1AlphaS2), true);
    	
    	/* Monta a refer�ncia ao conjunto vazio */
    	SetExpr conjuntoUnitarioDoConjuntoVazio = 
    		SchemaLawsFactory.montarConjuntoUnitarioDoConjuntoVazio();
    	
    	MemPred op1 = this.getFactory().createMemPred(
    			Arrays.asList(applIntersecaoAlphaS1AlphaS2, 
    					conjuntoUnitarioDoConjuntoVazio), true);
    	
    	/* 
    	 * ************************************* * 
    	 * OP2: DFV_Expr(CS_1) \subseteq \alpha(S'_1) *
    	 * ************************************* *
    	 */
    	
    	/* Montando a refer�ncia � função "DFV_Expr" */
    	ZName nomeDFV = this.getFactory().createZName(OPsDischargeUtils.DFV_Expr, 
    			this.getFactory().createZStrokeList(), null);
    	RefExpr funDFV = this.getFactory().createRefExpr(nomeDFV, 
    			this.getFactory().createZExprList(), false, false);
    	
    	/* Montando o argumento "CS_1" */
    	OzFactory factoryAux = new OzFactoryImpl();
    	PredExpr argCS1 = factoryAux.createPredExpr(p1);
    	
    	/* Monta a aplicação � função "DFV_Expr(CS_1)" */
    	ApplExpr applDFVCS1 = 
    		this.getFactory().createApplExpr(Arrays.asList(funDFV, argCS1), 
    				false);
    	
    	/* Monta a aplicação \alpha(S'_1) */
    	ApplExpr applAlphaS1Stroke = 
    		this.getFactory().createApplExpr(Arrays.asList(funAlpha, 
    				argS1Stroke), false);

    	/* Montando a refer�ncia � função "\subseteq" */
    	ZName nomeSubsetEq = this.getFactory().createZName(
    			OPsDischargeUtils.SUBSETEQ, 
    			this.getFactory().createZStrokeList(), null);
    	RefExpr funSubsetEq = this.getFactory().createRefExpr(nomeSubsetEq, 
    			this.getFactory().createZExprList(), false, false);
    	
    	/* Lista de argumentos "(DFV_Expr(CS_1), \alpha(S'_1))" */
    	ZExprList listaArgsDFVCS1AplhaS1Stroke = 
    		this.getFactory().createZExprList(Arrays.asList(applDFVCS1, 
    				applAlphaS1Stroke));
    	TupleExpr argsDFVCS1AplhaS1Stroke = 
    		this.getFactory().createTupleExpr(listaArgsDFVCS1AplhaS1Stroke);
    	
    	/* Montando a OP2 */
    	MemPred op2 = this.getFactory().createMemPred(
    			Arrays.asList(argsDFVCS1AplhaS1Stroke, funSubsetEq), true);
    	
    	/* 
    	 * ************************************* * 
    	 * OP3: DFV_Expr(CS_2) \sebseteq \alpha(S'_2) *
    	 * ************************************* *
    	 */
    	
    	/* Monta o argumento "CS_2" */
    	PredExpr argCS2 = factoryAux.createPredExpr(p2);
    	
    	/* Monta a aplicação "DFV_Expr(CS_1)" */
    	ApplExpr applDFVCS2 = 
    		this.getFactory().createApplExpr(Arrays.asList(funDFV, argCS2), 
    				false);
    	
    	/* Monta a aplicação "\alpha(S'_2)" */
    	ApplExpr applAlphaS2Stroke = 
    		this.getFactory().createApplExpr(Arrays.asList(funAlpha, 
    				argS2Stroke), false);
    	
    	/* Lista de argumentos "(DFV_Expr(CS_2), \alpha(S'_2))" */
    	ZExprList listaArgsDFVCS2AplhaS2Stroke = 
    		this.getFactory().createZExprList(Arrays.asList(applDFVCS2, 
    				applAlphaS2Stroke));
    	TupleExpr argsDFVCS2AplhaS2Stroke = 
    		this.getFactory().createTupleExpr(listaArgsDFVCS2AplhaS2Stroke);
    	
    	/* Montando a OP3 */
    	MemPred op3 = this.getFactory().createMemPred(
    			Arrays.asList(argsDFVCS2AplhaS2Stroke, funSubsetEq), true);
    	
    	/* **************************************************** */
    	
    	/* Adição das obriga��es de prova */
    	ops.add(op1);
    	ops.add(op2);
    	ops.add(op3);
    	
    	/* Criando a Lei */
    	String nome = this.createName("COD0554");
    	CircusLaw result = CircusPattUtils.createCircusLaw(nome, trans, ops);

    	LawNumberAnn id = new LawNumberAnn("C-72", nome);
    	result.getAnns().add(id);
    	LawTypeAnnUtils.insertAnnLawType(result, LawType.ACTION_REFINEMENT_SCHEMA);
    	
    	return result;
    	
    }
    
    
public CircusLaw createTesteSchemaSequenceIntrodution() {
    	
    	/* LHS */
    	JokerDeclList listaDecl = 
    		this.getFactory().createJokerDeclList("dl", null);
    	JokerPred p1 = this.getFactory().createJokerPred("p1", null);
    	JokerPred p2 = this.getFactory().createJokerPred("p2", null);
    	AndPred pred = 
    		this.getFactory().createAndPred(Arrays.asList(p1, p2), 
    				And.Wedge);
    	SchText texto = 
    		this.getFactory().createZSchText(listaDecl, pred);
    	SchExpr expr = this.getFactory().createSchExpr(texto);
    	SchExprAction left = this.getFactory().createSchExprAction(expr);
    	
    	/* RHS */
    	JokerDeclList ld1 = 
    		this.getFactory().createJokerDeclList("dl1", null);
    	JokerDeclList ld2 = 
    		this.getFactory().createJokerDeclList("dl2", null);
    	
    	/* Anotação para a aplicação da lei */
    	ld1.getAnns().add(new SeparateDashedDeclListAnn(listaDecl, ld1, ld2));
    	
    	SchText texto2 = this.getFactory().createZSchText(ld1, p1);
    	SchText texto3 = this.getFactory().createZSchText(ld2, p2);
    	SchExpr expr2 = this.getFactory().createSchExpr(texto2);
    	SchExpr expr3 = this.getFactory().createSchExpr(texto3);
    	SchExprAction r1 = this.getFactory().createSchExprAction(expr2);
    	SchExprAction r2 = this.getFactory().createSchExprAction(expr3);
    	SeqAction right = 
    		this.getFactory().createSeqAction(Arrays.asList(r1, r2));
    	
    	/* Criando o transformer */
    	ActionTransformerPred trans = factory.createActionTransformerPred(null, 
    			Transformation.Equivalence,
    			CircusUtils.DEFAULT_REFINEMENT_MODEL, 
    			Arrays.asList(left, right));

    	/* Criando as obriga��es de prova */
    	List<Pred> ops = new ArrayList<Pred>();
    	
    	/* 
    	 * *********************************************
    	 * OP1: \alpha(S_1) \cap \alpha(S_2) = \emptyset
    	 * *********************************************
    	 */
    	
    	/* Monta a refer�ncia � função \alpha */
    	ZName nomeAlpha = this.getFactory().createZName(OPsDischargeUtils.ALPHA,
    			this.getFactory().createZStrokeList(), null);
    	RefExpr funAlpha = this.getFactory().createRefExpr(nomeAlpha, 
    			this.getFactory().createZExprList(), false, false);
    	
    	/* Monta o argumento S1 */
    	ZSchText textoArgS1Stroke = this.getFactory().createZSchText(ld1, 
    			this.getFactory().createTruePred());
    	SchExpr argS1Stroke = this.getFactory().createSchExpr(textoArgS1Stroke);
    	
    	/* Monta a aplicação \alpha(S1) */
    	ApplExpr applAlphaS1 = 
    		this.getFactory().createApplExpr(Arrays.asList(funAlpha, 
    				argS1Stroke), false);
    	
    	/* Monta o argumento S2 */
    	ZSchText textoArgS2Stroke = this.getFactory().createZSchText(ld2, 
    			this.getFactory().createTruePred());
    	SchExpr argS2Stroke = this.getFactory().createSchExpr(textoArgS2Stroke);

    	/* Monta a aplicação \alpha(S2) */
    	ApplExpr applAlphaS2 = 
    		this.getFactory().createApplExpr(Arrays.asList(funAlpha, 
    				argS2Stroke), false);
    	
    	/* Monta � referencia � função interseção */
    	ZName nomeIntersecao = this.getFactory().createZName(
    			OPsDischargeUtils.CAP, this.getFactory().createZStrokeList(), 
    			null);
    	RefExpr funIntersecao = this.getFactory().createRefExpr(nomeIntersecao, 
    			this.getFactory().createZExprList(), false, false);
    	
    	/* Monta a lista de argumentos "(\alpha(S1), \alpha(S2))" */
    	ZExprList listaArgsAlphaS1AlphaS2 = 
    		this.getFactory().createZExprList(Arrays.asList(applAlphaS1, 
    				applAlphaS2));
    	TupleExpr argsAlphaS1AlphaS2 = 
    		this.getFactory().createTupleExpr(listaArgsAlphaS1AlphaS2);
    	
    	/* 
    	 * Monta a aplicação da função Interseção 
    	 * "\alpha(S_1) \cap \alpha(S_2)" 
    	 */
    	ApplExpr applIntersecaoAlphaS1AlphaS2 = 
    		this.getFactory().createApplExpr(Arrays.asList(funIntersecao, 
    				argsAlphaS1AlphaS2), true);
    	
    	/* Monta a refer�ncia ao conjunto vazio */
    	SetExpr conjuntoUnitarioDoConjuntoVazio = 
    		SchemaLawsFactory.montarConjuntoUnitarioDoConjuntoVazio();
    	
    	MemPred op1 = this.getFactory().createMemPred(
    			Arrays.asList(applIntersecaoAlphaS1AlphaS2, 
    					conjuntoUnitarioDoConjuntoVazio), true);
    	
    	/* 
    	 * ************************************* * 
    	 * OP2: DFV_Expr(CS_1) \subseteq \alpha(S'_1) *
    	 * ************************************* *
    	 */
    	
    	/* Montando a refer�ncia � função "DFV_Expr" */
    	ZName nomeDFV = this.getFactory().createZName(OPsDischargeUtils.DFV_Expr, 
    			this.getFactory().createZStrokeList(), null);
    	RefExpr funDFV = this.getFactory().createRefExpr(nomeDFV, 
    			this.getFactory().createZExprList(), false, false);
    	
    	/* Montando o argumento "CS_1" */
    	OzFactory factoryAux = new OzFactoryImpl();
    	PredExpr argCS1 = factoryAux.createPredExpr(p1);
    	
    	/* Monta a aplicação � função "DFV_Expr(CS_1)" */
    	ApplExpr applDFVCS1 = 
    		this.getFactory().createApplExpr(Arrays.asList(funDFV, argCS1), 
    				false);
    	
    	/* Monta a aplicação \alpha(S'_1) */
    	ApplExpr applAlphaS1Stroke = 
    		this.getFactory().createApplExpr(Arrays.asList(funAlpha, 
    				argS1Stroke), false);

    	/* Montando a refer�ncia � função "\subseteq" */
    	ZName nomeSubsetEq = this.getFactory().createZName(
    			OPsDischargeUtils.SUBSETEQ, 
    			this.getFactory().createZStrokeList(), null);
    	RefExpr funSubsetEq = this.getFactory().createRefExpr(nomeSubsetEq, 
    			this.getFactory().createZExprList(), false, false);
    	
    	/* Lista de argumentos "(DFV_Expr(CS_1), \alpha(S'_1))" */
    	ZExprList listaArgsDFVCS1AplhaS1Stroke = 
    		this.getFactory().createZExprList(Arrays.asList(applDFVCS1, 
    				applAlphaS1Stroke));
    	TupleExpr argsDFVCS1AplhaS1Stroke = 
    		this.getFactory().createTupleExpr(listaArgsDFVCS1AplhaS1Stroke);
    	
    	/* Montando a OP2 */
    	MemPred op2 = this.getFactory().createMemPred(
    			Arrays.asList(argsDFVCS1AplhaS1Stroke, funSubsetEq), true);
    	
    	/* 
    	 * ************************************* * 
    	 * OP3: DFV_Expr(CS_2) \sebseteq \alpha(S'_2) *
    	 * ************************************* *
    	 */
    	
    	/* Monta o argumento "CS_2" */
    	PredExpr argCS2 = factoryAux.createPredExpr(p2);
    	
    	/* Monta a aplicação "DFV_Expr(CS_1)" */
    	ApplExpr applDFVCS2 = 
    		this.getFactory().createApplExpr(Arrays.asList(funDFV, argCS2), 
    				false);
    	
    	/* Monta a aplicação "\alpha(S'_2)" */
    	ApplExpr applAlphaS2Stroke = 
    		this.getFactory().createApplExpr(Arrays.asList(funAlpha, 
    				argS2Stroke), false);
    	
    	/* Lista de argumentos "(DFV_Expr(CS_2), \alpha(S'_2))" */
    	ZExprList listaArgsDFVCS2AplhaS2Stroke = 
    		this.getFactory().createZExprList(Arrays.asList(applDFVCS2, 
    				applAlphaS2Stroke));
    	TupleExpr argsDFVCS2AplhaS2Stroke = 
    		this.getFactory().createTupleExpr(listaArgsDFVCS2AplhaS2Stroke);
    	
    	/* Montando a OP3 */
    	MemPred op3 = this.getFactory().createMemPred(
    			Arrays.asList(argsDFVCS2AplhaS2Stroke, funSubsetEq), true);
    	
    	/* **************************************************** */
    	
    	/* Adição das obriga��es de prova */
    	ops.add(op1);
    	ops.add(op2);
    	ops.add(op3);
    	
    	/* Criando a Lei */
    	String nome = this.createName("COD0554");
    	CircusLaw result = CircusPattUtils.createCircusLaw(nome, trans, ops);

    	LawNumberAnn id = new LawNumberAnn("C-72", nome);
    	result.getAnns().add(id);
    	LawTypeAnnUtils.insertAnnLawType(result, LawType.ACTION_REFINEMENT_SCHEMA);
    	
    	return result;
    	
    }
    
    /**
     * Cria lei Schemas/Parallelism composition--distribution (C-73)
     * 
     * @return a nova lei de circus
     */
    public CircusLaw createSchemasParallelismCompositionDistribution() {
    	
    	/* LHS */
    	JokerAction exp1 = this.getFactory().createJokerAction("SExp", 
    			null);
    	//SchExprAction sExpr = 
    	//	this.getFactory().createSchExprAction(exp1);
    	JokerAction a1 = this.getFactory().createJokerAction("A1", 
    			null);
    	JokerAction a2 = this.getFactory().createJokerAction("A2", 
    			null);
    	JokerExpr exprNS1 = 
    		this.getFactory().createJokerExpr("nsE1", null);
    	JokerExpr exprNS2 = 
    		this.getFactory().createJokerExpr("nsE2", null);
    	CircusNameSet ns1 = 
    		this.getFactory().createCircusNameSet(exprNS1);
    	CircusNameSet ns2 = 
    		this.getFactory().createCircusNameSet(exprNS2);
    	JokerChannelSet cs = 
    		this.getFactory().createJokerChannelSet("cs", null);
    	/*ParallelAction parAction = 
    		this.getFactory().createParallelAction(
    				Arrays.asList(a1, a2), Arrays.asList(ns1, ns2), cs);
    	
    	SeqAction left = 
    		this.getFactory().createSeqAction(Arrays.asList(sExpr, 
    				parAction));*/
    	
    	SeqAction left = 
    		this.getFactory().createSeqAction(Arrays.asList(exp1, 
    				a1));
    	ParallelAction parAction = 
    		this.getFactory().createParallelAction(
    				Arrays.asList(left, a2), Arrays.asList(ns1, ns2), cs);
    	
    	/* RHS */
/*    	SeqAction seq1 = 
    		this.getFactory().createSeqAction(Arrays.asList(sExpr, 
    				a1));
    	ParallelAction right = 
    		this.getFactory().createParallelAction(
    				Arrays.asList(seq1, a2), Arrays.asList(ns1, ns2), cs);*/
 
    	ParallelAction right = 
    		this.getFactory().createParallelAction(
    				Arrays.asList(a1, a2), Arrays.asList(ns1, ns2), cs);
    	
    	SeqAction seq1 = 
    		this.getFactory().createSeqAction(Arrays.asList(exp1, 
    				right));

    	
    	/* Criando o transformer */
    	ActionTransformerPred trans = factory.createActionTransformerPred(null, 
    			Transformation.Equivalence,
    			CircusUtils.DEFAULT_REFINEMENT_MODEL, 
    			//Arrays.asList(left, right));
    			Arrays.asList(parAction,seq1));

    	/* Criando as obriga��es de prova */
    	List<Pred> ops = new ArrayList<Pred>();
    	
    	/* 
    	 * ****************************** *
    	 * OP1: wrtV(SExp) \subseteq ns_1 *
    	 * ****************************** * 
    	 */

    	/* Montando a refer�ncia � função "wrtV" */
    	ZName nomeWrtV = this.getFactory().createZName("wrtV", this.getFactory().createZStrokeList(), null);
    	RefExpr funWrtV = this.getFactory().createRefExpr(nomeWrtV, 
    			this.getFactory().createZExprList(), false, false);

    	/* Montando o argumento (SExp) */
    	RefExpr argSExpr = this.getFactory().createRefExpr();
    	argSExpr.getAnns().add(new ActionArgumentAnn(exp1));
    	
    	/* Aplicação da função "wrtV(SExpr)" */
    	ApplExpr applWrtVSExpr = 
    		this.getFactory().createApplExpr(Arrays.asList(funWrtV, argSExpr), 
    				false);
    	
    	/* Montando a refer�ncia � função "\subseteq" */
    	ZName nomeSubsetEq = this.getFactory().createZName(ZString.ARG_TOK + 
    			ZString.SUBSETEQ  + ZString.ARG_TOK, 
    			this.getFactory().createZStrokeList(), null);
    	RefExpr funSubsetEq = this.getFactory().createRefExpr(nomeSubsetEq, 
    			this.getFactory().createZExprList(), false, false);

    	/* Monta a lista de argumentos "(wrtV(SExpr), ns_1)" */
    	ZExprList listaArgsWrtVSExprNs1 = 
    		this.getFactory().createZExprList(Arrays.asList(applWrtVSExpr, 
    				exprNS1)); 
    	TupleExpr argsWrtVSExprNs1 = 
    		this.getFactory().createTupleExpr(listaArgsWrtVSExprNs1);

    	/* Aplicação do est� contido */
    	MemPred op1 = 
    		this.getFactory().createMemPred(Arrays.asList(argsWrtVSExprNs1, 
    				funSubsetEq), true);
    	
    	/* Adicionando a OP1 */
    	ops.add(op1);
    	
    	/* 
    	 * ********************************************
    	 * OP2: wrtV(SExpr) \cap usedV(A_2) = \emptyset *
    	 * ******************************************** 
    	 */
    	
    	/* Montando a refer�ncia � função usedV */
    
    	RefExpr funUsedV = 
    		OPsDischargeUtils.refFuncao(OPsDischargeUtils.USED_V);
    	
    	/* Monta a lista de argumentos (A_2) */
    	RefExpr argA2 = this.getFactory().createRefExpr();
    	argA2.getAnns().add(new ActionArgumentAnn(a2));
    	
    	/* Monta a aplicação "usedV(A_2)" */
    	ApplExpr applUsedVA2 = 
    		this.getFactory().createApplExpr(Arrays.asList(funUsedV, argA2), 
    				false);
    	
    	/* Monta � referencia � função "_ \cap _" */
    	ZName nomeIntersecao = this.getFactory().createZName(ZString.ARG_TOK + 
    			ZString.CAP + ZString.ARG_TOK, 
    			this.getFactory().createZStrokeList(), null);
    	RefExpr funIntersecao = this.getFactory().createRefExpr(nomeIntersecao, 
    			this.getFactory().createZExprList(), false, false);
    	
    	/* Monta a lista de argumentos "(wrtV(SExpr), usedV(A_2))" */
    	ExprList listaArgsWrtVSExprUsedVA2 = 
    		this.getFactory().createZExprList(Arrays.asList(applWrtVSExpr, 
    				applUsedVA2));
    	TupleExpr argsWrtVSExprUsedVA2 = 
    		this.getFactory().createTupleExpr(listaArgsWrtVSExprUsedVA2);
    	
    	/* Monta a aplicação "wrtV(SExpr) \cap usedV(A_2)" */
    	ApplExpr applWrtVSExprCapUsedVA2 = 
    		this.getFactory().createApplExpr(Arrays.asList(funIntersecao, 
    				argsWrtVSExprUsedVA2), true);

    	/* Monta a refer�ncia ao conjunto vazio */
    	SetExpr conjuntoUnitarioDoConjuntoVazio = 
    		SchemaLawsFactory.montarConjuntoUnitarioDoConjuntoVazio();
    	
    	/* Igualdade */
    	MemPred op2 = 
    		this.getFactory().createMemPred(
    				Arrays.asList(applWrtVSExprCapUsedVA2, 
    						conjuntoUnitarioDoConjuntoVazio), true);
    	ops.add(op2);
    	
    	/* Criando a Lei */
    	String nome = this.createName("COD0559");
    	CircusLaw result = CircusPattUtils.createCircusLaw(nome, trans, ops);

    	LawNumberAnn id = new LawNumberAnn("C-73", nome);
    	result.getAnns().add(id);
    	LawTypeAnnUtils.insertAnnLawType(result, LawType.ACTION_REFINEMENT_SCHEMA);
    	
    	return result;
    	
    }
    
    
	/**
     * Cria uma lei que transforma o predicado de um esquema em um
     * equivalente definido pelo usu�rio.
     * 
     * @return a nova lei de circus
     */
    public CircusLaw createLawTransformPredicate() {
    	
    	/* LHS */
    	JokerPred pred = this.getFactory().createJokerPred("pred1", 
    			null);
    	JokerDeclList listaDecl = 
    		this.getFactory().createJokerDeclList("dl", null);
    	SchText texto = this.getFactory().createZSchText(listaDecl, pred);
    	SchExpr expr = this.getFactory().createSchExpr(texto);
    	SchExprAction left = this.getFactory().createSchExprAction(expr);
    	
    	/* RHS */
    	JokerPred pred2 = this.getFactory().createJokerPred("pred2", 
    			null);
    	SchText texto2 = 
    		this.getFactory().createZSchText(listaDecl, pred2);
    	SchExpr expr2 = this.getFactory().createSchExpr(texto2);
    	SchExprAction right = 
    		this.getFactory().createSchExprAction(expr2);
    	
    	/* Criando o transformer */
    	ActionTransformerPred trans = factory.createActionTransformerPred(null, 
    			Transformation.Equivalence,
    			CircusUtils.DEFAULT_REFINEMENT_MODEL, 
    			Arrays.asList(left, right));

    	/* Criando as obriga��es de prova */
    	List<Pred> ops = new ArrayList<Pred>();
    	IffPred op1 = this.getFactory().createIffPred(Arrays.asList(pred, 
    			pred2));
    	ops.add(op1);
    	
    	/* Criando a Lei */
    	String nome = this.createName("COD0538");
    	CircusLaw result = CircusPattUtils.createCircusLaw(nome, trans, ops);

    	LawNumberAnn id = new LawNumberAnn("TransformPredicate", nome);
    	result.getAnns().add(id);
    	LawTypeAnnUtils.insertAnnLawType(result, LawType.ACTION_REFINEMENT_SCHEMA);
    	
    	return result;
    	
    }
    
	/**
     * Lei para normalização de um esquema
     * 
     * @return a nova lei de circus
     */
    public CircusLaw createLawNormalize() {
    	
    	/* LHS */
    	JokerExpr sch = this.getFactory().createJokerExpr("sch1", null);
    	SchExprAction left = this.getFactory().createSchExprAction(sch);
    	
    	/* RHS */
    	JokerExpr sch2 = this.getFactory().createJokerExpr("sch2", null);
    	sch2.getAnns().add(new NormalizeApplAnn(sch, sch2));
    	SchExprAction right = 
    		this.getFactory().createSchExprAction(sch2);
    	
    	/* Criando o transformer */
    	ActionTransformerPred trans = factory.createActionTransformerPred(null, 
    			Transformation.Equivalence, 
    			CircusUtils.DEFAULT_REFINEMENT_MODEL, 
    			Arrays.asList(left, right));

    	/* Criando as obrigaa��es de prova */
    	List<Pred> ops = new ArrayList<Pred>();
    	
    	/* Criando a Lei */
    	String nome = this.createName("COD0547");
    	CircusLaw result = CircusPattUtils.createCircusLaw(nome, trans, ops);

    	LawNumberAnn id = new LawNumberAnn("Normalize", nome);
    	result.getAnns().add(id);
    	LawTypeAnnUtils.insertAnnLawType(result, 
    			LawType.ACTION_REFINEMENT_SCHEMA);
    	
    	
    	String[] description = new String[1];
    	description[0] = "It's the same application of zed's normalization";
    	
    	LawDefinitionAnn ann = new LawDefinitionAnn(description,null,null,null);
    	result.getAnns().add(ann);
    	
    	return result;
    	
    }
    
    
    
    /**
     * Lei para normalização de um esquema.
     * Aplica definições de nomes passados como parâmetro de um esquema
     * 
     * @return a nova lei de circus
     */
    public CircusLaw createLawReduceStep() {
    	
    	/* LHS */
    	JokerExpr sch = this.getFactory().createJokerExpr("sch1", null);
    	SchExprAction left = this.getFactory().createSchExprAction(sch);
    	
    	/* RHS */
    	JokerExpr sch2 = this.getFactory().createJokerExpr("sch2", null);
    	sch2.getAnns().add(new ReduceStepAnn(sch, sch2));
    	SchExprAction right = 
    		this.getFactory().createSchExprAction(sch2);
    	
    	/* Criando o transformer */
    	ActionTransformerPred trans = factory.createActionTransformerPred(null, 
    			Transformation.Equivalence, 
    			CircusUtils.DEFAULT_REFINEMENT_MODEL, 
    			Arrays.asList(left, right));

    	/* Criando as obrigaa��es de prova */
    	List<Pred> ops = new ArrayList<Pred>();
    	
    	/* Criando a Lei */
    	String nome = this.createName("ReduceStep");
    	CircusLaw result = CircusPattUtils.createCircusLaw(nome, trans, ops);

    	LawNumberAnn id = new LawNumberAnn("ReduceStep", nome);
    	result.getAnns().add(id);
    	LawTypeAnnUtils.insertAnnLawType(result, 
    			LawType.ACTION_REFINEMENT_SCHEMA);
    	
    	
    	String[] description = new String[1];
    	description[0] = "It substitutes a name param for its definition in a schema";
    	
    	LawDefinitionAnn ann = new LawDefinitionAnn(description,null,null,null);
    	result.getAnns().add(ann);
    	
    	return result;
    	
    }
    

	/**
     * Lei para decomposição de um esquema do tipo:
     * 
     * <p>Ex.: [d1 ; d2 ] = [d1] \land [d2]</p>
     * 
     * @return a nova lei de circus
     */
    public CircusLaw createLawSchemaDecomposition1() {
    	
    	/* LHS */
    	JokerDeclList dl = this.getFactory().createJokerDeclList("dl", null);
    	SchText texto = this.getFactory().createZSchText(dl, null);
    	SchExpr expr = this.getFactory().createSchExpr(texto);
    	SchExprAction left = this.getFactory().createSchExprAction(expr);
    	
    	/* RHS */
    	JokerDeclList dl1 = this.getFactory().createJokerDeclList("dl1", null);
    	JokerDeclList dl2 = this.getFactory().createJokerDeclList("dl2", null);
    	
    	/* Anotação para a aplicação da lei */
    	dl.getAnns().add(new UnifyDeclListAnn(dl, dl1, dl2));
    	dl1.getAnns().add(new SeparateDeclListAnn(dl, dl1, dl2));
    	
    	SchText texto1 = this.getFactory().createZSchText(dl1, null);
    	SchText texto2 = this.getFactory().createZSchText(dl2, null);
    	SchExpr expr1 = this.getFactory().createSchExpr(texto1);
    	SchExpr expr2 = this.getFactory().createSchExpr(texto2);
    	AndExpr conjuncao = 
    		this.getFactory().createAndExpr(Arrays.asList(expr1, expr2));
    	SchExprAction right = this.getFactory().createSchExprAction(conjuncao);
    	
    	/* Criando o transformer */
    	ActionTransformerPred trans = factory.createActionTransformerPred(null, 
    			Transformation.Equivalence, 
    			CircusUtils.DEFAULT_REFINEMENT_MODEL, 
    			Arrays.asList(left, right));

    	/* Criando as obrigaa��es de prova */
    	List<Pred> ops = new ArrayList<Pred>();
    	
    	/* Criando a Lei */
    	String nome = this.createName("COD0591");
    	CircusLaw result = CircusPattUtils.createCircusLaw(nome, trans, ops);

    	LawNumberAnn id = new LawNumberAnn("SchemaDecomposition", nome);
    	result.getAnns().add(id);
    	LawTypeAnnUtils.insertAnnLawType(result, LawType.ACTION_REFINEMENT_SCHEMA);
    	
    	return result;
    	
    }

	/**
     * <p>Lei para normalização de um esquema:</p>
     * 
     * <p>[ d1; d2 | P1 \land P2] = [ d1 | P1 ] \land [d2 | P2]</p>
     * 
     * @return a nova lei de circus
     */
    public CircusLaw createLawSchemaDecomposition2() {
    	
    	/* LHS */
    	JokerDeclList dl = this.getFactory().createJokerDeclList("dl", null);
    	JokerPred p1 = this.getFactory().createJokerPred("P1", null);
    	JokerPred p2 = this.getFactory().createJokerPred("P2", null);
    	SchText texto = this.getFactory().createZSchText(dl, 
    			this.getFactory().createAndPred(Arrays.asList(p1, p2), 
    					And.Wedge));
    	SchExpr expr = this.getFactory().createSchExpr(texto);
    	SchExprAction left = this.getFactory().createSchExprAction(expr);
    	
    	/* RHS */
    	JokerDeclList dl1 = this.getFactory().createJokerDeclList("dl1", null);
    	JokerDeclList dl2 = this.getFactory().createJokerDeclList("dl2", null);
    	
    	/* Anotação para a aplicação da lei */
    	dl.getAnns().add(new UnifyDeclListAnn(dl, dl1, dl2));
    	dl1.getAnns().add(new SeparateDeclListAnn(dl, dl1, dl2));
    	
    	SchText texto1 = this.getFactory().createZSchText(dl1, p1);
    	SchText texto2 = this.getFactory().createZSchText(dl2, p2);
    	SchExpr expr1 = this.getFactory().createSchExpr(texto1);
    	SchExpr expr2 = this.getFactory().createSchExpr(texto2);
    	AndExpr conjuncao = 
    		this.getFactory().createAndExpr(Arrays.asList(expr1, expr2));
    	SchExprAction right = this.getFactory().createSchExprAction(conjuncao);
    	
    	/* Criando o transformer */
    	ActionTransformerPred trans = factory.createActionTransformerPred(null, 
    			Transformation.Equivalence, 
    			CircusUtils.DEFAULT_REFINEMENT_MODEL, 
    			Arrays.asList(left, right));

    	/* Criando as obrigaa��es de prova */
    	List<Pred> ops = new ArrayList<Pred>();
    	
    	/* 
    	 * ********************************
    	 * OP1: FV(P1) \subseteq \alpha(d1)
    	 * ********************************
    	 */
    	
    	/* Criacao de Fv(p1) */
    	OzFactory ozFactory = new OzFactoryImpl();
    	RefExpr fv = OPsDischargeUtils.refFuncao(OPsDischargeUtils.FV_Expr);
    	PredExpr p1Expr = ozFactory.createPredExpr(p1);
    	ApplExpr fvP1 = 
    		this.getFactory().createApplExpr(Arrays.asList(fv, p1Expr), false);

    	/* Aplicação \alpha(d1) */
    	RefExpr alpha = OPsDischargeUtils.refFuncao(OPsDischargeUtils.ALPHA);
    	SchText schDl1 = this.getFactory().createZSchText(dl1, null);
    	SchExpr schDl1Expr = this.getFactory().createSchExpr(schDl1);
    	ApplExpr alphaDl1 = 
    		this.getFactory().createApplExpr(Arrays.asList(alpha, schDl1Expr), 
    				false);
    	
    	/* Fv(p1) \subseteq \alpha(dl1) */
    	RefExpr subseteq = 
    		OPsDischargeUtils.refFuncao(OPsDischargeUtils.SUBSETEQ);
    	ZExprList fvp1Alphadl1 = 
    		this.getFactory().createZExprList(Arrays.asList(fvP1, alphaDl1));
    	TupleExpr argsFvp1Alphadl1 = 
    		this.getFactory().createTupleExpr(fvp1Alphadl1);
    	MemPred fvp1SubseteqAlphadl1 = 
    		this.getFactory().createMemPred(Arrays.asList(argsFvp1Alphadl1, 
    				subseteq), true);
    	
    	/* Adiciona a op a lista de obriga��es de prova */
    	ops.add(fvp1SubseteqAlphadl1);
    	
    	/* 
    	 * ********************************
    	 * OP2: FV(P2) \subseteq \alpha(d2)
    	 * ********************************
    	 */
    	
    	/* Montando FV(P2) */
    	PredExpr p2Expr = ozFactory.createPredExpr(p2);
    	ApplExpr fvp2 = 
    		this.getFactory().createApplExpr(Arrays.asList(fv, p2Expr), false);
    	
    	/* Montando \alpha(dl2) */
    	SchText schDl2 = this.getFactory().createZSchText(dl2, null);
    	SchExpr schDl2Expr = this.getFactory().createSchExpr(schDl2);
    	ApplExpr alphaDl2 = 
    		this.getFactory().createApplExpr(Arrays.asList(alpha, schDl2Expr), 
    				false);
    	
    	/* Montando Fv(p2) \subseteq \alpha(dl2) */
    	ZExprList fvp2Alphadl2 = 
    		this.getFactory().createZExprList(Arrays.asList(fvp2, alphaDl2));
    	TupleExpr argsFvp2Alphadl2 = 
    		this.getFactory().createTupleExpr(fvp2Alphadl2);
    	MemPred fvp2SubseteqAlphadl2 = 
    		this.getFactory().createMemPred(Arrays.asList(argsFvp2Alphadl2, 
    				subseteq), true);
    	
    	/* Adiciona a op a lista de obriga��es de prova */
    	ops.add(fvp2SubseteqAlphadl2);
    	
    	/* Criando a Lei */
    	String nome = this.createName("COD0606");
    	CircusLaw result = CircusPattUtils.createCircusLaw(nome, trans, ops);

    	LawNumberAnn id = new LawNumberAnn("SchemaDecomposition2", nome);
    	result.getAnns().add(id);
    	LawTypeAnnUtils.insertAnnLawType(result, LawType.ACTION_REFINEMENT_SCHEMA);
    	
    	return result;
    	
    }
    
    /**
     * <p>Cria a lei que transforma uma operação do tipo \Xi em sua 
     * definição</p>
     * 
     * <p>\Xi S = [ S, S' | x' = x \land ... \land z' = z],
     * \alpha(S) = {x, ..., z}</p>
     * 
     * @return a lei circus que implementa a regra Z mencionada acima
     */
    public CircusLaw createRuleXiSchema() {
    	
    	/* LHS */
    	RefExpr xi = OPsDischargeUtils.refFuncao(ZString.XI);
    	JokerDeclList d = this.getFactory().createJokerDeclList("d", null);
    	JokerPred p = this.getFactory().createJokerPred("p", null);
    	ZSchText schema = this.getFactory().createZSchText(d, p);
    	SchExpr schemaExpr = this.getFactory().createSchExpr(schema);
    	ApplExpr xiSchema = this.getFactory().createApplExpr(Arrays.asList(xi, 
    			schemaExpr), true);
    	SchExprAction left = this.getFactory().createSchExprAction(xiSchema);
    	
    	/* RHS */
     	JokerDeclList dl = this.getFactory().createJokerDeclList("dl", null);
     	JokerPred pred = this.getFactory().createJokerPred("p1", null);
     	ZSchText sch = this.getFactory().createZSchText(dl, pred);
     	SchExpr schExpr = this.getFactory().createSchExpr(sch);
     	SchExprAction right = this.getFactory().createSchExprAction(schExpr);
     	
     	JokerPred invEquiv = 
     		this.getFactory().createJokerPred("invEquiv", null);
     	
     	/* Anota��es para a aplicação da lei */
     	dl.getAnns().add(new XiRuleApplicationAnn(d, dl, pred, invEquiv));
     	
     	/* Criando o transformer */
    	ActionTransformerPred trans = factory.createActionTransformerPred(null, 
    			Transformation.Refinement, 
    			CircusUtils.DEFAULT_REFINEMENT_MODEL, 
    			Arrays.asList(left, right));

    	/* Criando as obrigaa��es de prova */
    	List<Pred> ops = new ArrayList<Pred>();
    	ops.add(invEquiv);
    	  	
    	/* Criando a Lei */
    	String nome = this.createName("COD0612");
    	CircusLaw result = CircusPattUtils.createCircusLaw(nome, trans, ops);

    	LawNumberAnn id = new LawNumberAnn("XiSchema", nome);
    	result.getAnns().add(id);
    	LawTypeAnnUtils.insertAnnLawType(result, LawType.ACTION_REFINEMENT_SCHEMA);
    	
    	/* Inserindo detalhes da lei*/
    	String[] description = new String[1];
    	String alpha = "\u03B1";
    	description[0] = "P_1 = [" + alpha + "(P); "+ alpha +"(P') |" +
    			" inv(P)" + ZString.AND + " inv(P') " + ZString.AND + " "
    			+ ZString.ALL + "x: x " + ZString.MEM + " " + alpha + "(P) "
    			+ ZString.SPOT + " x' = x]";
    	
    	String[] antes = new String[1];
    	antes[0] = "\\Xi P";
    	String[] depois = new String[1];
    	depois[0] = "P_1";
    	
    	ArrayList<String> opDetails = new ArrayList<String>();
    	String newOp = "Invariant of variables in P and the invariant of " +
    			"variables in P_1 has to be equals";
    	opDetails.add(newOp);
    	
    	LawDefinitionAnn ann = new LawDefinitionAnn(description
    			, opDetails,antes,depois);
    	result.getAnns().add(ann);
    	
    	return result;
    	
    }
    
    /**
     * <p>Cria a lei que transforma uma operação do tipo \Xi em sua 
     * definição</p>
     * 
     * <p>\Xi S = [ S, S' | x' = x \land ... \land z' = z],
     * \alpha(S) = {x, ..., z}</p>
     * 
     * @return a lei circus que implementa a regra Z mencionada acima
     */
    public CircusLaw createRuleXiSchemaLeftDirection() {
    	
    	/* LHS */
     	JokerDeclList dl = this.getFactory().createJokerDeclList("dl", null);
     	JokerPred pred = this.getFactory().createJokerPred("p1", null);
     	ZSchText sch = this.getFactory().createZSchText(dl, pred);
     	SchExpr schExpr = this.getFactory().createSchExpr(sch);
     	SchExprAction left = this.getFactory().createSchExprAction(schExpr);
     	
     	/* RHS */
    	RefExpr xi = OPsDischargeUtils.refFuncao(ZString.XI);
    	JokerExpr nameOfSchema = this.getFactory().createJokerExpr("name", 
    			null);
    	ApplExpr xiSchema = this.getFactory().createApplExpr(Arrays.asList(xi, 
    			nameOfSchema), true);
    	SchExprAction right = this.getFactory().createSchExprAction(xiSchema);
     	
     	JokerPred invEquiv = 
     		this.getFactory().createJokerPred("invEquiv", null);
     	
     	/* Anotações para a aplicação da lei */
     	nameOfSchema.getAnns().add(new XiRuleApplicationLeftAnn(dl, pred, 
     			invEquiv, nameOfSchema));
     	
     	/* Criando o transformer */
    	ActionTransformerPred trans = factory.createActionTransformerPred(null, 
    			Transformation.Refinement, 
    			CircusUtils.DEFAULT_REFINEMENT_MODEL, 
    			Arrays.asList(left, right));

    	/* Criando as obrigaações de prova */
    	List<Pred> ops = new ArrayList<Pred>();
    	ops.add(invEquiv);
    	  	
    	/* Criando a Lei */
    	String nome = this.createName("COD0638");
    	CircusLaw result = CircusPattUtils.createCircusLaw(nome, trans, ops);

    	LawNumberAnn id = new LawNumberAnn("XiSchemaLeft", nome);
    	result.getAnns().add(id);
    	LawTypeAnnUtils.insertAnnLawType(result, LawType.ACTION_REFINEMENT_SCHEMA);
    	
    	
    	
    	/* Inserindo detalhes da lei*/
    	String[] description = new String[2];
    	String alpha = "\u03B1";
    	description[0] =  " P_1 = [" + alpha + "(P); "+ alpha +"(P') |" +
    			" inv(P)" + ZString.AND + " inv(P') " + ZString.AND + " "
    			+ ZString.ALL + "x: x " + ZString.MEM + " " + alpha + "(P) "
    			+ ZString.SPOT + " x' = x]";
    	
    	description[1] = " where Name is the name of a schema inside " +
    			"a process";
    	
    	String[] antes = new String[1];
    	antes[0] = "P_1";
    	String[] depois = new String[1];
    	depois[0] = "\\Xi Name";
    	
    	ArrayList<String> opDetails = new ArrayList<String>();
    	String newOp = "Invariant of variables in P_1 and the invariant of " +
		"variables in Name has to be equals";
    	opDetails.add(newOp);
    	
    	LawDefinitionAnn ann = new LawDefinitionAnn(description
    			, opDetails,antes,depois);
    	result.getAnns().add(ann);
    	
    	
    	
    	
    	return result;
    	
    }
    
    /**
     * Cria a regra S1 \land \Xi S2 = S1
     * 
     * @return a lei do CRefine que implementa a regra acima
     */
    public CircusLaw createRuleLandZero() {

    	/* LHS */
    	JokerExpr s1 = this.getFactory().createJokerExpr("S1", null);

    	RefExpr xi = OPsDischargeUtils.refFuncao(ZString.XI);
    	JokerExpr s2 = this.getFactory().createJokerExpr("S2", null);
    	ApplExpr xiSchema = this.getFactory().createApplExpr(Arrays.asList(xi, 
    			s2), true);
    	AndExpr leftExpr = 
    		this.getFactory().createAndExpr(Arrays.asList(s1, xiSchema));
    	SchExprAction left = this.getFactory().createSchExprAction(leftExpr);
    	
    	/* RHS */
     	SchExprAction right = this.getFactory().createSchExprAction(s1);
     	
     	/* Criando o transformer */
    	ActionTransformerPred trans = factory.createActionTransformerPred(null, 
    			Transformation.Equivalence, 
    			CircusUtils.DEFAULT_REFINEMENT_MODEL, 
    			Arrays.asList(left, right));

    	/* Criando as obrigaações de prova */
    	List<Pred> ops = new ArrayList<Pred>();
    	
    	/*
    	 * **************************************************** 
    	 * OP1: (\alpha(S2) \cup \alpha (S2')) \cap \alpha S1 = 
    	 * 	\emptyset
    	 * ****************************************************
    	 */
    	
    	/* Criando alpha(S2) */
    	RefExpr alpha = OPsDischargeUtils.refFuncao(OPsDischargeUtils.ALPHA);
    	ApplExpr alphaS2 = this.getFactory().createApplExpr(Arrays.asList(alpha, 
    			s2), false);
    	
    	/* Criano alpha(S2') */
    	DecorExpr dashedS2 = this.getFactory().createDecorExpr(s2, 
    			this.getFactory().createNextStroke());
    	ApplExpr alphaDashedS2 = 
    		this.getFactory().createApplExpr(Arrays.asList(alpha, dashedS2), 
    				false);
    	
    	/* alpha(S2) \cup alpha(S2') */
    	RefExpr cup = OPsDischargeUtils.refFuncao(OPsDischargeUtils.CUP);
    	ZExprList zList = 
    		this.getFactory().createZExprList(Arrays.asList(alphaS2, 
    				alphaDashedS2));
    	TupleExpr alphaS2AlphaDashedS2 = 
    		this.getFactory().createTupleExpr(zList);
    	ApplExpr alphaS2CupAlphaDashedS2 = 
    		this.getFactory().createApplExpr(Arrays.asList(cup, 
    				alphaS2AlphaDashedS2), true);
    	
    	/* (alpha(S2) \cup alpha(S2')) \cap alpha(S1) */
    	RefExpr cap = OPsDischargeUtils.refFuncao(OPsDischargeUtils.CAP);
    	ApplExpr alphaS1 = 
    		this.getFactory().createApplExpr(Arrays.asList(alpha, s1), false);
    	List<? extends Expr> listaArgsCapAux = 
    		Arrays.asList(alphaS2CupAlphaDashedS2, alphaS1);
    	ZExprList listaArgsCap = 
    		this.getFactory().createZExprList(listaArgsCapAux);
    	TupleExpr tuplaArgsCap = 
    		this.getFactory().createTupleExpr(listaArgsCap);
    	ApplExpr capTuplaArgsCap = 
    		this.getFactory().createApplExpr(Arrays.asList(cap, tuplaArgsCap), 
    				true);
    	
    	/* alpha(S2) \cup alpha(S2')) \cap alpha(S1) = emptyset */
    	RefExpr emptyset = OPsDischargeUtils.criarConjuntoVazio();
    	List<? extends Expr> listaUnitariaAux = 
    		Collections.singletonList(emptyset);
    	ZExprList listaUnitaria = 
    		this.getFactory().createZExprList(listaUnitariaAux);
    	SetExpr conjuntoUnitario = 
    		this.getFactory().createSetExpr(listaUnitaria);
    	MemPred op1 = 
    		this.getFactory().createMemPred(Arrays.asList(capTuplaArgsCap, 
    				conjuntoUnitario), true);

    	/* Adiciona a OP1 */
    	ops.add(op1);
    	  	
    	/* Criando a Lei */
    	String nome = this.createName("COD0616");
    	CircusLaw result = CircusPattUtils.createCircusLaw(nome, trans, ops);

    	LawNumberAnn id = new LawNumberAnn("ZeroAnd", nome);
    	result.getAnns().add(id);
    	LawTypeAnnUtils.insertAnnLawType(result, 
    			LawType.ACTION_REFINEMENT_SCHEMA);
    	
    	return result;
    }
    
    /**
     * Cria a lei de CRefine que permite rearranjar uma lista de
     * declarações, isto �, permite mudar a ordem das declarações
     * em um esquema
     * 
     * [ D | P ] = [ D' | P ]
     * dado que 
     * \alpha(D) = alpha(D')
     * \forall x @ \alpha(D) @
     * 		x \in typeof(x, D) \iff x \in typeof(x, D')
     * 
     * @return a lei do CRefine que implementa a regra acima
     */
    public CircusLaw createRuleRearrangeDeclList() {

    	/* LHS */
    	JokerDeclList d = this.getFactory().createJokerDeclList("dl1", null);
    	JokerPred p = this.getFactory().createJokerPred("p", null);
    	ZSchText text = this.getFactory().createZSchText(d, p);
    	SchExpr expr = this.getFactory().createSchExpr(text);
    	SchExprAction left = this.getFactory().createSchExprAction(expr);
    	
    	/* RHS */
    	JokerDeclList newD = this.getFactory().createJokerDeclList("dl2", null);
    	ZSchText text2 = this.getFactory().createZSchText(newD, p);
    	SchExpr expr2 = this.getFactory().createSchExpr(text2);
    	SchExprAction right = this.getFactory().createSchExprAction(expr2);
     	
     	/* Criando o transformer */
    	ActionTransformerPred trans = factory.createActionTransformerPred(null, 
    			Transformation.Equivalence, 
    			CircusUtils.DEFAULT_REFINEMENT_MODEL, 
    			Arrays.asList(left, right));

    	/* Criando as obrigaações de prova */
    	List<Pred> ops = new ArrayList<Pred>();
    	
    	/*
    	 * **************************************************** 
    	 * OP1: \alpha( expr ) = \alpha( expr2 )
    	 * ****************************************************
    	 */
    	
    	/* Criando alpha(expr) */
    	RefExpr alpha = OPsDischargeUtils.refFuncao(OPsDischargeUtils.ALPHA);
    	ApplExpr alphaExpr = 
    		this.getFactory().createApplExpr(Arrays.asList(alpha, expr), false);
    	
    	/* Criando alpha(expr2) */
    	ApplExpr alphaExpr2 = 
    		this.getFactory().createApplExpr(Arrays.asList(alpha, expr2), 
    				false);
    	
    	/* \alpha(expr) = \alpha(expr2) */
    	List<? extends Expr> listaUnitariaAux = 
    		Collections.singletonList(alphaExpr2);
    	ZExprList listaUnitaria = 
    		this.getFactory().createZExprList(listaUnitariaAux);
    	SetExpr conjuntoUnitario = 
    		this.getFactory().createSetExpr(listaUnitaria);
    	MemPred op1 = 
    		this.getFactory().createMemPred(Arrays.asList(alphaExpr, 
    				conjuntoUnitario), true);

    	/* Adiciona a OP1 */
    	ops.add(op1);
    	
    	String opstring2 = "The type of the variables in the second " +
		"declaration list has to be the same of the first one";

    	JokerPred op2 = this.getFactory().createJokerPred(opstring2, null);
    	op2.getAnns().add(new RearrangeDeclListPOMounter(d, newD, op2));
    	
    	ops.add(op2);
    	
    	/* Criando a Lei */
    	String nome = this.createName("COD0636");
    	CircusLaw result = CircusPattUtils.createCircusLaw(nome, trans, ops);

    	LawNumberAnn id = new LawNumberAnn("RearrageDeclList", nome);
    	result.getAnns().add(id);
    	LawTypeAnnUtils.insertAnnLawType(result, 
    			LawType.ACTION_REFINEMENT_SCHEMA);
    	
    	return result;
    }
    
    /**
     * Cria a lei de CRefine que permite rearranjar uma lista de
     * declarações, isto �, permite mudar a ordem das declarações
     * em um esquema
     * 
     * [ D | P ] = [ D' | P ]
     * dado que 
     * \alpha(D) = alpha(D')
     * \forall x @ \alpha(D) @
     * 		x \in typeof(x, D) \iff x \in typeof(x, D')
     * 
     * @return a lei do CRefine que implementa a regra acima
     */
    public CircusLaw createRuleRearrangeDeclListWithoutPred() {

    	/* LHS */
    	JokerDeclList d = this.getFactory().createJokerDeclList("dl1", null);
    	ZSchText text = this.getFactory().createZSchText(d, null);
    	SchExpr expr = this.getFactory().createSchExpr(text);
    	SchExprAction left = this.getFactory().createSchExprAction(expr);
    	
    	/* RHS */
    	JokerDeclList newD = this.getFactory().createJokerDeclList("dl2", null);
    	ZSchText text2 = this.getFactory().createZSchText(newD, null);
    	SchExpr expr2 = this.getFactory().createSchExpr(text2);
    	SchExprAction right = this.getFactory().createSchExprAction(expr2);
     	
     	/* Criando o transformer */
    	ActionTransformerPred trans = factory.createActionTransformerPred(null, 
    			Transformation.Equivalence, 
    			CircusUtils.DEFAULT_REFINEMENT_MODEL, 
    			Arrays.asList(left, right));

    	/* Criando as obrigações de prova */
    	List<Pred> ops = new ArrayList<Pred>();
    	
    	/*
    	 * **************************************************** 
    	 * OP1: \alpha( expr ) = \alpha( expr2 )
    	 * ****************************************************
    	 */
    	
    	/* Criando alpha(expr) */
    	RefExpr alpha = OPsDischargeUtils.refFuncao(OPsDischargeUtils.ALPHA);
    	ApplExpr alphaExpr = 
    		this.getFactory().createApplExpr(Arrays.asList(alpha, expr), false);
    	
    	/* Criando alpha(expr2) */
    	ApplExpr alphaExpr2 = 
    		this.getFactory().createApplExpr(Arrays.asList(alpha, expr2), 
    				false);
    	
    	/* \alpha(expr) = \alpha(expr2) */
    	List<? extends Expr> listaUnitariaAux = 
    		Collections.singletonList(alphaExpr2);
    	ZExprList listaUnitaria = 
    		this.getFactory().createZExprList(listaUnitariaAux);
    	SetExpr conjuntoUnitario = 
    		this.getFactory().createSetExpr(listaUnitaria);
    	MemPred op1 = 
    		this.getFactory().createMemPred(Arrays.asList(alphaExpr, 
    				conjuntoUnitario), true);

    	/* Adiciona a OP1 */
    	ops.add(op1);

    	String opstring2 = "The type of the variables in the second " +
    			"declaration list has to be the same of the first one";
    	
    	JokerPred op2 = this.getFactory().createJokerPred(opstring2, null);
    	op2.getAnns().add(new RearrangeDeclListPOMounter(d, newD, op2));
    	
    	/* Adiciona a OP2 */
    	ops.add(op2);
    	
    	/* Criando a Lei */
    	String nome = this.createName("COD0637");
    	CircusLaw result = CircusPattUtils.createCircusLaw(nome, trans, ops);

    	LawNumberAnn id = new LawNumberAnn("RearrageDeclListWithouPred", nome);
    	result.getAnns().add(id);
    	LawTypeAnnUtils.insertAnnLawType(result, 
    			LawType.ACTION_REFINEMENT_SCHEMA);
    	
    	return result;
    }
    
}
