/*
 * Projeto: Circus Refine
 */
package circusRefine.core.crules;

import java.io.StringWriter;

import circusRefine.core.annotations.UpdatingTermAnn;
import circusRefine.core.print.Printer;
import circusRefine.core.relations.RelationsAnnStack;

import net.sourceforge.czt.base.ast.Term;
import net.sourceforge.czt.base.visitor.TermVisitor;
import net.sourceforge.czt.base.visitor.VisitorUtils;
import net.sourceforge.czt.circus.ast.SeqAction;
import net.sourceforge.czt.circus.ast.VarDeclCommand;
import net.sourceforge.czt.circus.impl.SeqActionImpl;
import net.sourceforge.czt.circus.impl.VarDeclCommandImpl;

/**
 * Visitor respons�vel pela atualizacao da AST apos a aplicacao de 
 * uma lei
 * 
 * @author Cristiano Gurgel
 */
public class UpdateVisitor implements TermVisitor<Term> {

	/**
	 * Atualiza uma AST.
	 * 
	 * @param oldTerm o termo a ser substituido
	 * @param newTerm o termo substituto
	 * @param ast a AST onde sera efetuada a atualizacao
	 * @param changeTerm indica se sera uma substituição no proprio termo, 
	 * em geral devido uma aplicação de lei. No caso se for false, geralmente
	 * trata-se de uma substituição de nomes que coincidem.
	 * @return a nova AST atualizada
	 */
	public static Term update(Term oldTerm, Term newTerm, Term ast, boolean changeTerm) {
		if (changeTerm) {
			oldTerm.getAnns().add(new UpdatingTermAnn());
		}
		UpdateVisitor updater = new UpdateVisitor(oldTerm, newTerm, changeTerm);
		Term novaAST = ast.accept(updater);
		return novaAST;
	}

	/** O termo a ser substituido */
	private Term oldTerm;

	/** O termo substituto */
	private Term newTerm;

	/** Log da atualizacao */
	private StringWriter log;

	/** Flag para indicar se a substituição j� ocorreu */
	private boolean occured;

	/** Flag para indicar se a substituição ser� feita na equival�ncia
	 *  entre termos ou atrav�s de inserção de anotações, ou seja,
	 *  na referencia real do termo */
	private boolean toChangeTerm;

	/**
	 * Inicializa o Visitor com um termo a ser substituido por um
	 * termo substituto
	 * 
	 * @param oldT termo a ser substituido
	 * @param newT termo substituto
	 */
	private UpdateVisitor(Term oldT, Term newT, boolean toChange) {
		this.setOldTerm(oldT);
		this.setNewTerm(newT);
		this.setLog(new StringWriter());
		this.addMessage("Log da Atualizacao: " + this);
		this.occured = false;
		this.toChangeTerm = toChange; 
	}

	private Term getOldTerm() {
		return oldTerm;
	}

	private void setOldTerm(Term oldTerm) {
		this.oldTerm = oldTerm;
	}

	private Term getNewTerm() {
		return newTerm;
	}

	private void setNewTerm(Term newTerm) {
		this.newTerm = newTerm;
	}

	/**
	 * Acessa o log da classe
	 * 
	 * @return a informacao sobre unificacao
	 */
	public String getInfo() {
		return this.getLog().toString();
	}

	private StringWriter getLog() {
		return log;
	}

	private void setLog(StringWriter log) {
		this.log = log;
	}

	/**
	 * Adiciona uma mensagem ao log de unificacao
	 * 
	 * @param msg a mensagem a ser adicionada
	 */
	private void addMessage(String msg) {
		this.getLog().append(msg + "\n");
	}

	/**
	 * Visita a AST a ser atualizada trocando o Nodo no qual foi
	 * aplicada a lei. Todos os ancestrais do nodo trocado, sao
	 * substitu�dos por uma copia.
	 * 
	 * @param termo AST a ser visitada
	 * @return a nova AST
	 */
	public Term visitTerm(Term termo) {

		/* Termo nao tem a estrutura atualizada */
		Term result;
		Term test = null;

		boolean b1; 
		if (toChangeTerm) {
			b1 = (termo.getAnn(UpdatingTermAnn.class) == null);
		}
		else {
			b1 = !termo.equals(this.getOldTerm());
		}

		if ( !this.occured ) {
			if (b1) {

				/* 
				 * Visita as criancas, criando uma nova AST nas partes 
				 * necessarias 
				 */
				Term novoTermo = VisitorUtils.visitTerm(this, termo, true);

				if (novoTermo != termo) { /* Termo foi copiado */

					/* Retira RelationsAnn do Novo Termo */
					this.removeRelationsAnnStack(novoTermo);

					result = novoTermo;
	                if (result instanceof VarDeclCommandImpl){
	                	Term old = this.getOldTerm();
	                	Term newT = this.getNewTerm();
	                	VarDeclCommand var = (VarDeclCommand) result;
	                	var.getZDeclList();
	                	var.getCircusAction();
	                }
	                if (result instanceof SeqActionImpl){
	                	Term old = this.getOldTerm();
	                	Term newT = this.getNewTerm();
	                	SeqAction seq= (SeqAction) result;
	                	seq.getLeftAction();
	                	seq.getRightAction();
	                }

				} else { /* Termo n�o foi copiado */
					
					this.addMessage("Visita normal: " + novoTermo.getClass());

					/* Nada precisa ser feito */
					result = novoTermo;
				}
			} else {
				
                if (termo instanceof VarDeclCommandImpl){
                	Term old = this.getOldTerm();
                	Term newT = this.getNewTerm();
                	VarDeclCommandImpl var = (VarDeclCommandImpl) termo;
                	var.getZDeclList();
                	var.getCircusAction();
                }
                else if (termo instanceof SeqActionImpl){
                	Term old = this.getOldTerm();
                	Term newT = this.getNewTerm();
                	SeqAction seq= (SeqAction) termo;
                	seq.getLeftAction();
                	seq.getRightAction();
                }

				/* log */
				this.addMessage("Substituindo: " + termo.getClass());

				UpdatingTermAnn ann = termo.getAnn(UpdatingTermAnn.class);
				termo.getAnns().remove(ann);

				/* O termo eh atualizado */
				Term novoTermo = this.getNewTerm();
				novoTermo.getAnns().addAll(termo.getAnns());

				this.removeRelationsAnnStack(novoTermo);

				/* J� substituiu */
				this.occured = true;

				result = this.getNewTerm();

			}
		} else {
			result = termo;
		}

		return result;
	}

	private void removeRelationsAnnStack(Term novoTermo) {

		/* Retira RelationsAnn do Novo Termo */
		RelationsAnnStack rel = novoTermo.getAnn(RelationsAnnStack.class);
		if (rel != null) {
			boolean res = novoTermo.getAnns().remove(rel);
			this.addMessage(res + " Removendo relacionamento do novo Termo"); 
		}
	}

}
