/*
 * Projeto: Circus Refine
 * 
 * Autores: Alessandro Gurgel <alessandro87@consiste.dimap.ufrn.br>
 * 			Cristiano Castro  <crisgc@consiste.dimap.ufrn.br>
 */
package circusRefine.core;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.Reader;
import java.util.ArrayList;

import java.util.Arrays;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.ListIterator;
import java.util.Set;
import java.util.Vector;


import zb2smt.ResultStatusOP;
import zb2smt.ZB2SMTUtils;
import zb2smt.bol.Seq;
import net.sourceforge.czt.circus.util.Factory;
import net.sourceforge.czt.base.ast.ListTerm;
import net.sourceforge.czt.base.ast.Term;
import net.sourceforge.czt.base.util.UnmarshalException;
import net.sourceforge.czt.circus.ast.ActionPara;
import net.sourceforge.czt.circus.ast.BasicProcess;
import net.sourceforge.czt.circus.ast.CallAction;
import net.sourceforge.czt.circus.ast.ChannelPara;
import net.sourceforge.czt.circus.ast.ChannelSet;
import net.sourceforge.czt.circus.ast.ChannelSetPara;
import net.sourceforge.czt.circus.ast.CircusAction;
import net.sourceforge.czt.circus.ast.CircusActionList;
import net.sourceforge.czt.circus.ast.CircusProcess;
import net.sourceforge.czt.circus.ast.CircusStateAnn;
import net.sourceforge.czt.circus.ast.Communication;
import net.sourceforge.czt.circus.ast.ExtChoiceAction;
import net.sourceforge.czt.circus.ast.ExtChoiceActionIte;
import net.sourceforge.czt.circus.ast.GuardedAction;
import net.sourceforge.czt.circus.ast.IntChoiceAction;
import net.sourceforge.czt.circus.ast.IntChoiceActionIte;
import net.sourceforge.czt.circus.ast.MuAction;
import net.sourceforge.czt.circus.ast.OnTheFlyDefAnn;
import net.sourceforge.czt.circus.ast.ParallelProcessIte;
import net.sourceforge.czt.circus.ast.ProcessPara;
import net.sourceforge.czt.circus.ast.SchExprAction;
import net.sourceforge.czt.circus.ast.SkipAction;
import net.sourceforge.czt.circus.ast.Transformation;
import net.sourceforge.czt.circus.ast.TransformerPred;
import net.sourceforge.czt.circus.ast.VarDeclCommand;
import net.sourceforge.czt.circus.impl.Action1Impl;
import net.sourceforge.czt.circus.impl.Action2Impl;
import net.sourceforge.czt.circus.impl.ActionParaImpl;
import net.sourceforge.czt.circus.impl.BasicProcessImpl;
import net.sourceforge.czt.circus.impl.ChaosActionImpl;
import net.sourceforge.czt.circus.impl.ExtChoiceActionImpl;
import net.sourceforge.czt.circus.impl.ExtChoiceActionIteImpl;
import net.sourceforge.czt.circus.impl.ExtChoiceProcessImpl;
import net.sourceforge.czt.circus.impl.ExtChoiceProcessIteImpl;
import net.sourceforge.czt.circus.impl.GuardedActionImpl;
import net.sourceforge.czt.circus.impl.HideActionImpl;
import net.sourceforge.czt.circus.impl.HideProcessImpl;
import net.sourceforge.czt.circus.impl.IfGuardedCommandImpl;
import net.sourceforge.czt.circus.impl.IndexedProcessImpl;
import net.sourceforge.czt.circus.impl.IntChoiceActionImpl;
import net.sourceforge.czt.circus.impl.IntChoiceActionIteImpl;
import net.sourceforge.czt.circus.impl.IntChoiceProcessImpl;
import net.sourceforge.czt.circus.impl.IntChoiceProcessIteImpl;
import net.sourceforge.czt.circus.impl.InterleaveActionImpl;
import net.sourceforge.czt.circus.impl.InterleaveActionIteImpl;
import net.sourceforge.czt.circus.impl.InterleaveProcessIdxImpl;
import net.sourceforge.czt.circus.impl.InterleaveProcessImpl;
import net.sourceforge.czt.circus.impl.MuActionImpl;
import net.sourceforge.czt.circus.impl.ParallelActionImpl;
import net.sourceforge.czt.circus.impl.ParallelProcessImpl;
import net.sourceforge.czt.circus.impl.ParallelProcessIteImpl;
import net.sourceforge.czt.circus.impl.ParamActionImpl;
import net.sourceforge.czt.circus.impl.ParamProcessImpl;
import net.sourceforge.czt.circus.impl.PrefixingActionImpl;
import net.sourceforge.czt.circus.impl.ProcessParaImpl;
import net.sourceforge.czt.circus.impl.SeqActionImpl;
import net.sourceforge.czt.circus.impl.SeqActionIteImpl;
import net.sourceforge.czt.circus.impl.SeqProcessImpl;
import net.sourceforge.czt.circus.impl.SeqProcessIteImpl;
import net.sourceforge.czt.circus.impl.SkipActionImpl;
import net.sourceforge.czt.circus.impl.StopActionImpl;
import net.sourceforge.czt.circus.impl.VarDeclCommandImpl;
import net.sourceforge.czt.circus.util.CircusUtils;
import net.sourceforge.czt.circuspatt.ast.CircusPatternFactory;
import net.sourceforge.czt.circuspatt.ast.JokerParaList;
import net.sourceforge.czt.circuspatt.ast.JokerParaListBinding;
import net.sourceforge.czt.circuspatt.impl.CircusPatternFactoryImpl;
import net.sourceforge.czt.circuspatt.util.CircusLaw;
import net.sourceforge.czt.parser.circus.ParseUtils;
import net.sourceforge.czt.parser.util.ParseException;
import net.sourceforge.czt.rules.unification.UnificationException;
import net.sourceforge.czt.session.CommandException;
import net.sourceforge.czt.session.FileSource;
import net.sourceforge.czt.session.Markup;
import net.sourceforge.czt.session.SectionManager;
import net.sourceforge.czt.session.Source;
import net.sourceforge.czt.z.ast.AxPara;
import net.sourceforge.czt.z.ast.Box;
import net.sourceforge.czt.z.ast.ConstDecl;
import net.sourceforge.czt.z.ast.Decl;
import net.sourceforge.czt.z.ast.DeclList;
import net.sourceforge.czt.z.ast.Expr;
import net.sourceforge.czt.z.ast.Fact;
import net.sourceforge.czt.z.ast.Name;
import net.sourceforge.czt.z.ast.NameList;
import net.sourceforge.czt.z.ast.Para;
import net.sourceforge.czt.z.ast.Pred;
import net.sourceforge.czt.z.ast.SchExpr;
import net.sourceforge.czt.z.ast.Sect;
import net.sourceforge.czt.z.ast.Spec;
import net.sourceforge.czt.z.ast.TruePred;
import net.sourceforge.czt.z.ast.ZDeclList;
import net.sourceforge.czt.z.ast.ZName;
import net.sourceforge.czt.z.ast.ZNameList;
import net.sourceforge.czt.z.ast.ZParaList;
import net.sourceforge.czt.z.ast.ZSchText;
import net.sourceforge.czt.z.ast.ZSect;
import net.sourceforge.czt.z.impl.AndPredImpl;
import net.sourceforge.czt.z.impl.AxParaImpl;
import net.sourceforge.czt.z.impl.ConstDeclImpl;
import net.sourceforge.czt.z.impl.ImpliesPredImpl;
import net.sourceforge.czt.z.impl.MemPredImpl;
import net.sourceforge.czt.z.impl.OrPredImpl;
import net.sourceforge.czt.z.impl.RefExprImpl;
import net.sourceforge.czt.z.impl.SpecImpl;
import net.sourceforge.czt.z.impl.TupleExprImpl;
import net.sourceforge.czt.z.impl.VarDeclImpl;
import net.sourceforge.czt.z.impl.ZNameImpl;
import net.sourceforge.czt.z.impl.ZSectImpl;
import net.sourceforge.czt.zpatt.ast.Binding;
import net.sourceforge.czt.zpatt.ast.Sequent;
import circusRefine.core.crules.BindingGetter;
import circusRefine.core.crules.CRulesException;
import circusRefine.core.crules.CRulesUtils;
import circusRefine.core.crules.CancelledApplException;
import circusRefine.core.crules.CircusLawApplicationException;
import circusRefine.core.crules.UnifierVisitor;
import circusRefine.core.crules.UpdateVisitor;
import circusRefine.core.crules.anotations.LawApplAnn;
import circusRefine.core.crules.factories.LawsCreator;
import circusRefine.core.crules.parseArgument.ParseArgumentException;
import circusRefine.core.crules.parseArgument.ParseArgumentUtils;
import circusRefine.core.crules.utils.TermFromANameGetter;
import circusRefine.core.developments.DevelopmentsManager;
import circusRefine.core.developments.PODevelopmentTree;
import circusRefine.core.finder.ActionParaFinder;
import circusRefine.core.finder.BasicProcessFinder;
import circusRefine.core.finder.ProcessParaFinder;
import circusRefine.core.finder.SchemaFinder;
import circusRefine.core.opsdischarge.OPsDischargeUtils;
import circusRefine.core.opsdischarge.Provador;
import circusRefine.core.print.Printer;
import circusRefine.core.specificationeditor.SpecEditor;
import circusRefine.core.specificationeditor.SpecificationEditorException;
import circusRefine.core.storage.CarregamentoEspecificacao;
import circusRefine.core.storage.OpenSpecificationException;
import circusRefine.core.storage.StepOfExecution;
import circusRefine.core.storage.StorageUtils;
import circusRefine.core.storage.ToSave;
import circusRefine.core.util.ChildrenTermExtractor;
import circusRefine.core.util.ClonerVisitor;
import circusRefine.core.util.CodParamVisitor;
import circusRefine.core.util.POLog;
import circusRefine.util.CRefineException;
import circusRefine.util.CodNameParam;
import circusRefine.util.CodigoParametro;
import circusRefine.util.Internacional;
import circusRefine.util.OPTipos;
import circusRefine.util.Pair;
import circusRefine.util.circusstring.LatexToCircusTactics;
import circusRefine.util.zstring.ZtoLatex;
import circusRefine.Tactic.Excecao.LawNotFound;
import circusRefine.Tactic.Excecao.TacticNotFound;
import circusRefine.Tactic.Excecao.Unification;
import circusRefine.Tactic.Factories.TacticsCreator;
import circusRefine.Tactic.Parser.Parser2;
import circusRefine.Tactic.Principal.*;
import circusRefine.Tactic.Util.GerenciadorTaticas;
import circusRefine.Tactic.Util.Head;
import circusRefine.Tactic.Util.RCell;
import circusRefine.Tactic.Util.TacticAnswer;
import circusRefine.Tactic.Util.FNames;
import circusRefine.Tactic.Util.Tail;

/**
 * Gerenciador Interno do Circus Refine
 *
 * @author Alessandro Gurgel
 * @author Cristiano Gurgel
 */
public class InternalManager {

	/** Para executar o parser no arquivo */
	private final static SectionManager manager_ = 
		new SectionManager( "zpatt" );

	public static SectionManager getManager() {
		return InternalManager.manager_;
	}

	/** Referencia para o Gerenciador Externo, o gerenciador mais importante*/
	private ExternalManager gerInterface;


	/** A Fabrica de leis usadas no CircusRefine */
	private LawsCreator criador;

	/**
	 * Para a fábrica de táticas
	 */

	private TacticsCreator criadorTactics;
	private Vector<String> tiposTatica;
	private static Term var;

	CircusPatternFactory factory = new CircusPatternFactoryImpl();


	/** Objeto utilitário para a aplicação de leis */
	private CRulesUtils crUtils;

	/** 
	 * Objeto Utilitário utilizado para o salvamento/abertura do 
	 * programa 
	 */
	private StorageUtils strgUtils;

	/**
	 * Gerenciador de Desenvolvimentos;
	 */
	private DevelopmentsManager gerDevelopments;

	/** Lista de termos referentes as Leis */
	private java.util.List<CircusLaw> laws;


	/** Referencia para o Gerenciador de Táticas */
	private Parser2 gerParser;
	private GerenciadorTaticas gerTaticas;

	public GerenciadorTaticas getGerTaticas() {
		return gerTaticas;
	}

	public void setGerTaticas(GerenciadorTaticas gerTaticas) {
		this.gerTaticas = gerTaticas;
	}

	private Tatica taticaAtual;

	private List concat;

	/**
	 * Cria uma inst�ncia do Gerenciado Interno com o Gerenciador
	 * Externo definido
	 * 
	 * @param internacional informa��es de idioma da ferramenta
	 */
	public InternalManager(ExternalManager gerExt, DevelopmentsManager gerDev) {
		this.setGerInterface(gerExt);
		this.setDevelopmentManager(gerDev);

		this.setCrUtils(new CRulesUtils(this, 
				gerExt.getInternacional().getLocalizacao()));
		this.setCriador(new LawsCreator(this.getInternacional(), this));
		this.setStrgUtils(new StorageUtils());
		laws = this.createLaws();

		/** 
		 * Para as táticas
		 */
		gerParser = new Parser2();
		this.setGerParser(new Parser2());
		respostaLei = new LawAnswer();
		tiposTatica = new Vector<String>();
		setarTiposTatica();
		this.setGerTaticas(new GerenciadorTaticas(gerExt));
		//this.setCriadorTactics(new TacticsCreator(this));


	}

	public void setarTiposTatica() {
		this.tiposTatica.add("Tlaw");
		this.tiposTatica.add("TSemi");
		this.tiposTatica.add("TAlt");
		this.tiposTatica.add("TFails");
		this.tiposTatica.add("TTatica");
	}

	private void setDevelopmentManager(DevelopmentsManager gerDev) {
		this.gerDevelopments = gerDev;
	}

	public TacticsCreator getCriadorTactics() {
		return criadorTactics;
	}

	public void setCriadorTactics(TacticsCreator criadorTactics) {
		this.criadorTactics = criadorTactics;
	}


	private void setCriador(LawsCreator creator) {
		criador = creator;
	}

	/**
	 * Acessa a refer�ncia ao Gerenciador Externo
	 * 
	 * @return uma refer�ncia ao Gerenciador Externo do Circus Refine
	 */
	private ExternalManager getGerInterface() {
		return gerInterface;
	}

	/**
	 * Seta a refer�ncia ao Gerenciador Externo
	 * 
	 * @param gerInterface o novo gerenciador externo do Circus 
	 * 	Refine
	 */
	private void setGerInterface(ExternalManager gerInterface) {
		this.gerInterface = gerInterface;
	}

	/**
	 * Seta a refer�ncia ao Parser de Táticas
	 * 
	 * @param gerInterface
	 *            o novo gerenciador externo do Circus Refine
	 */
	private void setGerParser(Parser2 gerParser) {
		this.gerParser = gerParser;
	}

	/**
	 * @return the crUtils
	 */
	private CRulesUtils getCrUtils() {
		return crUtils;
	}

	/**
	 * @param crUtils the crUtils to set
	 */
	private void setCrUtils(CRulesUtils crUtils) {
		this.crUtils = crUtils;
	}

	/**
	 * @return the strgUtils
	 */
	private StorageUtils getStrgUtils() {
		return strgUtils;
	}

	/**
	 * @param strgUtils the strgUtils to set
	 */
	private void setStrgUtils(StorageUtils strgUtils) {
		this.strgUtils = strgUtils;
	}


	/**
	 * Retorna a classe de internacionalizacao do Circus Refine
	 * 
	 * @return a nova classe de internacionalizacao
	 */
	private Internacional getInternacional() {
		return this.getGerInterface().getInternacional();
	}

	/**
	 * Retorna a fabrica de leis do Circus Refine
	 * 
	 * @return a fabrica de leis
	 */
	private LawsCreator getCriador() {
		return criador;
	}



	/**
	 * Atualiza o programa com o novo, colocando o atual, no topo da
	 * pilha de undo de um desenvolvimento selecionado
	 * 
	 * @param novoPrograma o novo programa
	 */
	private void atualizarPrograma(Term novoPrograma) {
		int index = this.gerInterface.getSelectedDevelopment();
		this.gerDevelopments.getDevelopment(index).atualizarPrograma(novoPrograma);

	}

	/**
	 * Atualiza o programa com o novo, colocando o atual, no topo da
	 * pilha de undo de um desenvolvimento selecionado
	 * 
	 * @param novoPrograma o novo programa
	 */
	private void atualizarProgramaTatica(Term novoPrograma) {
		int index = this.gerInterface.getSelectedDevelopment();
		this.gerDevelopments.getDevelopment(index).atualizarProgramaTatica(novoPrograma);

	}


	/**
	 * Percorre a AST do programa par�metro e lista todas as Acoes 
	 * presentes neste programa. 
	 * 
	 * @return a Lista com as acoes presentes no programa corrente do
	 * 	Circus Refine
	 */
	public List<ActionPara> retornarActionsPara(Term prog) {
		List<ActionPara> result = new ArrayList<ActionPara>();
		result = ActionParaFinder.getActionsPara(prog);
		//this.setActionsPara(result);
		return result;
	}

	/**
	 * Percorre a AST do programa par�metro e lista todos os Processos 
	 * presentes neste programa. 
	 * 
	 * @return a Lista com os processos presentes no programa corrente do
	 * 	Circus Refine
	 * @see #listarNomesProcessPara()
	 */
	public List<ProcessPara> retornarProcessPara(Term prog) {
		List<ProcessPara> result = new ArrayList<ProcessPara>();
		result = ProcessParaFinder.getProcessPara(prog);
		//this.setProcessPara(result);
		return result;
	}

	/**
	 * Lista os {@link BasicProcess} presentes no programa atual
	 * 
	 * @return uma lista com os BasicProcess do Programa atual
	 */
	public ListTerm<BasicProcess> retornarBasicProcess(Term prog) {
		return BasicProcessFinder.listar(prog);
	}

	/**
	 * Executa uma pesquisa na lista de acoes de um programa.
	 * 
	 * @param nome o nome da acao a ser pesquisada
	 * @param ast A Arvere onde ocorrer� a busca
	 * @return a acao encontrada
	 */
	public CollectAnswer coletarCodigoAcao(String nome, Term ast) {
		ActionPara result = null;

		List<ActionPara> actions = this.retornarActionsPara(ast);

		for(ActionPara act : actions) {
			String nomeAct = act.getZName().getWord();
			if(nomeAct.equals(nome)) {
				result = act;
				break;
			}
		}

		return new CollectAnswer(TipoColeta.ACAO, nome, result);
	}


	/**
	 * Executa uma pesquisa na lista de processos de um programa.
	 * @param nome o nome do processo a ser pesquisada
	 * @param ast A Arvore onde ocorrer� a busca
	 * @return o processo encontrado
	 */
	public CollectAnswer coletarCodigoProcesso(String nome, Term ast) {
		ProcessPara result = null;

		List<ProcessPara> processes = this.retornarProcessPara(ast);

		for(ProcessPara pro : processes) {
			String nomeAct = pro.getZName().getWord();
			if(nomeAct.equals(nome)) {
				result = pro;
				break;
			}
		}

		CollectAnswer resposta = new CollectAnswer(TipoColeta.PROCESSO, nome, 
				result);
		return resposta;
	}

	/**
	 * Abre um desenvolvimento.
	 * 
	 * @param source
	 * @return
	 * @throws UnmarshalException 
	 * @throws IOException 
	 * @throws ParseException 
	 */
	public LoadAnswer openSpecification(Source source) throws ParseException, 
	IOException, UnmarshalException {
		LoadAnswer result = this.parserProg(source);
		return result; 

	}

	/**
	 * Salva o desenvolvimento atual do Circus Refine em um arquivo.
	 * 
	 * @param statusExterno o status do gerenciador externo
	 * @param arquivo o arquivo no qual o desenvolvimento ser� salvo
	 * @throws IOException 
	 * @see StorageUtils#salvarProg(StatusGerenciadorInternoSecondary, File) 
	 */
	public void salvar(File arquivo) throws IOException {
		this.getStrgUtils().salvarPrograma(this.getPassosToSave(arquivo), arquivo);
	}

	/**
	 * Salva os passos no arquivo passado
	 * @param arquivo
	 * @param passos
	 * @throws IOException
	 */
	public void salvar(File arquivo, ToSave passos) throws IOException {
		this.getStrgUtils().salvarPrograma(passos, arquivo);
	}


	/**
	 * 
	 * @return A lista de passos do desenvolvimento selecinado 
	 */
	private ToSave getPassos() {
		int i = this.gerInterface.getSelectedDevelopment();
		return this.gerDevelopments.getDevelopment(i).getPassos();
	}

	/**
	 *  Alem de retornar a lista de historico do desenvolvimento, carrega
	 *  os desenvolvimetnos filhos nessa lista
	 * @param arquivo 
	 * @return O passo do desenvolvimento selecinado para salvar o desenvolvimetno
	 * @throws IOException 
	 */
	private ToSave getPassosToSave(File arquivo) throws IOException {
		int i = this.gerInterface.getSelectedDevelopment();
		this.gerDevelopments.inserirSubDesenvolvimentoHistorico(i, arquivo);
		return this.gerDevelopments.getDevelopment(i).getPassos();
	}


	/**
	 * Abre um desenvolvimento salvo anteriormente
	 * 
	 * @param arquivo o arquivo a ser aberto
	 * @throws IOException 
	 * @throws OpenSpecificationException 
	 * @see {@link StorageUtils#abrirPrograma(File)}
	 */
	public ToSave abrir(File arquivo) throws IOException, OpenSpecificationException 
	{
		ToSave historico = this.getStrgUtils().abrirPrograma(arquivo, gerInterface);
		int count = 0;
		this.gerInterface.retornarTelaPrincipal().
		setarMaximumProgressBar(historico.getHistorico().size());
		try {
			((CarregamentoEspecificacao)historico.getHistorico().get(0)).setNome(arquivo.getName());

			/* Reexecuta os passos */
			for (StepOfExecution step : historico.getHistorico()) {
				step.doStep(this.getGerInterface());
				this.gerInterface.retornarTelaPrincipal().incrementProgress();
				count++;
			}
		}
		catch (	Exception e){
			e.printStackTrace();
			throw new OpenSpecificationException(e);
		}

		return historico;
	}

	/**
	 * Retorna a Lista de leis de Circus a serem carregadas no inicio
	 *  da execucao
	 * 
	 * @return a lista de Leis
	 */
	public List<CircusLaw> createLaws() {
		return this.getCriador().createLaws();
	}

	/**
	 * Parseia um programa a partir de uma fonte
	 * 
	 * @param source a fonte do qual se dese
	 * @return
	 * @throws ParseException
	 * @throws IOException
	 * @throws UnmarshalException
	 */
	public LoadAnswer parserProg(Source source) throws ParseException, IOException, 
	UnmarshalException {
		Term term = ParseUtils.parse(source, InternalManager.getManager());
		//term = NormalASTGenerator.gerarFormaNormal(term);

		/* Adicionar o passo de desenvolvimento ao hist�rico */
		String especificacao = this.retornarStringEspecificacao(source);

		return new LoadAnswer(term, especificacao);
	}

	/**
	 * Retorna a String com a especificação
	 * 
	 * @param source o source com a especificação
	 * @throws IOException caso haja algum erro na leitura do 
	 *  {@link Source}
	 */
	private String retornarStringEspecificacao(Source source) 
	throws IOException {
		Reader reader = source.getReader();

		int temp;
		StringBuilder str = new StringBuilder(); 
		while ( (temp = reader.read()) != -1 ) {
			str.append((char)(temp));
		}

		return str.toString();   	
	}

	/**
	 * Metodo que me retornar o tipo de Lei que sera aplicado. Podendo ser
	 * Equivalencia, Simulação ou Refinamento dependendo do tipo de Transformation
	 */
	public Transformation getType( CircusLaw c) {

		TransformerPred trPred;
		if (c.isActionLaw())
			trPred = c.getActionRel();
		else
			trPred = c.getProcessRel();

		Transformation result = trPred.getTransformation();

		return result;
	}


	/**
	 * Filtra, a partir de uma lista de leis, somente as leis que
	 *  podem ser aplicadas a um determinado programa.
	 *   
	 * @param prog o programa a ser testado.
	 * @param laws a lista de leis.
	 * @return uma nova lista de leis correspondendo as leis
	 *  filtradas. A ordem dessa lista respeita a ordem do iterador
	 *  da lista original.
	 */
	public List<CircusLaw> filterLaws(Term prog, List<CircusLaw> laws) {
		List<CircusLaw> result = new ArrayList<CircusLaw>();

		/* 
		 * Percorre toda a lista de leis filtrando somente as leis 
		 * aplicaveis ao termo 
		 */
		for (CircusLaw lei : laws) {

			if (this.getCrUtils().testApplicability(prog, lei)) {
				/* As leis tb podem ser filtradas pelo tipo de lei
				 * de acordo com o tipo de Desenvolvimento 
				 */
				int dev = this.gerInterface.getSelectedDevelopment();
				if (gerDevelopments.getDevelopment(dev) instanceof PODevelopmentTree){

					PODevelopmentTree subDev = (PODevelopmentTree)gerDevelopments.getDevelopment(dev);
					/* Equival�ncia aceita equival�ncia ou Refinamento em duplo sentido,
					 * mas que ainda n�o � tratado pela ferramenta*/
					if (subDev.getTipo().equals(Transformation.Equivalence)){
						Transformation typeLaw = this.getType(lei);
						if (typeLaw.equals(Transformation.Equivalence)){
							result.add(lei);
						}
					}
					/* Simulação aceita Equivalencia e Simula��es*/
					else if (subDev.getTipo().equals(Transformation.Simulation)) {
						Transformation typeLaw = this.getType(lei);
						if (typeLaw.equals(Transformation.Equivalence) || typeLaw.equals(Transformation.Simulation)){
							result.add(lei);
						}
					}
					/* Refinamento*/
					else {
						result.add(lei);
					}

				}
				else {
					result.add(lei);
				}

			}
		}

		return result;
	}

	/**
	 * Aplica uma lei na AST.
	 * 
	 * @param prog o nodo da AST onde ser� aplicada a lei
	 * @param leiSelecionada a lei que ser� aplicada
	 * @return uma resposta a lei que armazena o termo final da lei, as
	 * 		obriga��es de prova, o identificador do nodo programa.
	 * @paran ast A arvore sintatica do Desenvolvimento utilizado
	 * @throws CircusLawApplicationException caso algum erro tenha
	 *  ocorrido na aplicação
	 */
	public LawAnswer aplicarLei(Term prog, CircusLaw leiSelecionada, Term ast) 
	throws CRulesException, CancelledApplException {
		LawAnswer res =  this.getCrUtils().applyLaw(prog, leiSelecionada, ast);



		/* Tenta provar as OPs */
		this.provarOps(res.getOps());
		this.atualizarPrograma(res.getNovaAST());
		/* Verifica se o termo target foi alcan�ado*/

		return res;
	}


	/**
	 * Funcao responsavel por pegar um predicado e passa-lo para 
	 * algum provador de Teorema e obter a resposta.
	 * @param pred Predicado
	 */
	protected OPTipos automaticProof(Pred pred) {
		/*
		 * Convertendo Circus 2 Bol 
		 */
		ResultStatusOP result = ZB2SMTUtils.proveByVeriT(pred);

		if (result.equals(ResultStatusOP.CHECKED_TRUE)){
			return OPTipos.OP_CHECADA_TRUE;
		}
		else if (result.equals(ResultStatusOP.CHECKED_FALSE)){
			return OPTipos.OP_CHECADA_FALSE;
		}
		else {
			return OPTipos.OP_CHECADA_NRECONHECIDA;
		}
	}
	/**
	 * Metodo utilizado para criar arquivo do SMT
	 * @param body corpo do arquivo
	 * @param cont identificador do arquivo
	 * @return String do nome do arquivo
	 */
	private String criandoArquivoSMT(StringBuffer body, int cont) {
		File aux = new File("");
		String abspath = aux.getAbsolutePath();
		String fs = File.separator;
		String path = fs + "tp" + fs + "harvey" + fs +"exemplos_gerados" + fs; 
		path = abspath + path;

		String name = "circusPO-" + cont + ".smt";

		File file;
		file = new File (path + name);
		byte [] data = body.toString().getBytes();

		/* OutputStreams */
		FileOutputStream out;
		try {
			out = new FileOutputStream(file);
			out.write(data);
		} catch (FileNotFoundException e) {
			e.printStackTrace();
		}
		catch (IOException e){
			e.printStackTrace();
		}

		String result = path + name;

		return result;
	}


	/**
	 * M�todo utilit�rio para tentar provar uma lista de predicados
	 * proveniente de uma resposta de lei
	 * 
	 * @param lista a tupla de OPs e OPs extendidas e um espa�o para
	 *  setar o resultado da avaliação
	 */
	private void provarOps(List<Pair<Pair<Sequent, Sequent>, Pair<OPTipos,POLog>>> 
	lista) {
		for (Pair<Pair<Sequent, Sequent>,Pair<OPTipos, POLog>> tuplaPredicados : lista) {

			/* Predicado a provar */
			Pred extendido = tuplaPredicados.getFirst().getSecond().getPred();

			OPTipos result = OPTipos.OP_NCHECADA;
			/* tenta provar sintaticamente que o predicado e verdadeiro*/
			if (extendido != null) {
				result = this.resultProver(extendido);
				if (result.equals(OPTipos.OP_CHECADA_TRUE)) {
					String msg = this.gerInterface.getMessage("COD0717");
					tuplaPredicados.getSecond().getSecond().insertHistoryStep("- " + msg);
				}
				else if (result.equals(OPTipos.OP_CHECADA_FALSE)){
					String msg = this.gerInterface.getMessage("COD0718");
					tuplaPredicados.getSecond().getSecond().insertHistoryStep("- " + msg);
				}
				else {
					String msg = this.gerInterface.getMessage("COD0719");
					tuplaPredicados.getSecond().getSecond().insertHistoryStep("- " + msg);
				}
			} 
			if (result.equals(OPTipos.OP_NCHECADA) || 
					result.equals(OPTipos.OP_CHECADA_NRECONHECIDA)){
				/* Tentar provar atraves da provaddores automaticos*/
				Pred pred = tuplaPredicados.getFirst().getFirst().getPred();

				try {
					result = automaticProof(pred);
					if (result.equals(OPTipos.OP_CHECADA_TRUE)) {
						String msg = this.gerInterface.getMessage("COD0720");
						tuplaPredicados.getSecond().getSecond().insertHistoryStep("- " + msg);
					}
					else if (result.equals(OPTipos.OP_CHECADA_FALSE)){
						String msg = this.gerInterface.getMessage("COD0721");
						tuplaPredicados.getSecond().getSecond().insertHistoryStep("- " + msg);
					}
					else {
						/* Nao conseguiu provar*/
						String msg = this.gerInterface.getMessage("COD0722");
						tuplaPredicados.getSecond().getSecond().insertHistoryStep("- " + msg);


						/* Tentar marcar op para ir para um model checker */
						boolean toModelChecker = verify2ModelChecker(pred);

						if (toModelChecker){
							/* Nao conseguiu provar*/
							String msg2 = this.gerInterface.getMessage("COD0748");
							tuplaPredicados.getSecond().getSecond().insertHistoryStep("- " + msg2);	
						}

					}
				}
				catch (Exception e) {
					System.err.println("CREFINE : Nao conseguiu passar o predicado para SMT");//e.printStackTrace();
					String msg = this.gerInterface.getMessage("COD0723");
					tuplaPredicados.getSecond().getSecond().insertHistoryStep("- " + msg);

					/* Tentar marcar op para ir para um model checker */
					boolean toModelChecker = verify2ModelChecker(pred);

					if (toModelChecker){
						/* Nao conseguiu provar*/
						String msg2 = this.gerInterface.getMessage("COD0748");
						tuplaPredicados.getSecond().getSecond().insertHistoryStep("- " + msg2);	
					}
				}


			}
			tuplaPredicados.getSecond().setFirst(result);
		}
	}

	/**
	 * 
	 * @param pred
	 * @return True caso o predicado deva ser enviado para 
	 */
	private boolean verify2ModelChecker(Pred pred) {

		boolean result = false;
		String textoPred = Printer.printAll(pred, Markup.LATEX, false);
		String strDivergenceFree = OPsDischargeUtils.DIVERGENCE_FREE.substring(0
				, OPsDischargeUtils.DIVERGENCE_FREE.length() - 2);
		String strDeterministic = OPsDischargeUtils.DETERMINISTIC.substring(0
				, OPsDischargeUtils.DETERMINISTIC.length() - 2);
		if (textoPred.contains(strDivergenceFree) || 
				textoPred.contains(strDeterministic)){
			result = true;
		}


		return result;
	}


	/**
	 * Aplica uma lei quando da abertura de um refinamento salvo.
	 * 
	 * @param prog o n� da AST alvo da aplicação da lei
	 * @param leiSelecionada a lei selecionada para a aplicação
	 * @param params a lista de par�metros para a lei a ser aplicada
	 * @return a resposta da lei aplicada
	 * @throws CircusLawApplicationException caso algum erro tenha 
	 *  ocorrido na aplicação da lei
	 * @see #aplicarLei(Term, CircusLaw)
	 */
	public LawAnswer aplicarLei(Term prog, CircusLaw leiSelecionada, 
			List<String> params) throws CRulesException {
		/* Primeiro, muda o modo de aplicação da lei */
		this.getCrUtils().mudarParaModoOpenSave(params);
		return this.aplicarLei(prog, leiSelecionada, retornarProgAtual());
	}

	/**
	 * Insere um par�grafo na especificação a partir de uma string com
	 *  o par�grafo
	 * @param userInput a entrada do usu�rio para ser parseada
	 * @return a resposta do passo de inserção de par�grafo
	 * @throws ParseArgumentException 
	 */
	public InsertAnswer insertParagraph( String userInput ) 
	throws CRulesException, SpecificationEditorException {
		SpecEditor ins = SpecEditor.getInstance();

		InsertAnswer para = ins.insertNewPararagraph( userInput , 
				(Spec) this.gerInterface.retornarProgAtual() , 
				InternalManager.getManager() );
		return para;
	}

	/**
	 * Unifica o argumentos formais da lei com os argumentos reais
	 * passados pelo usu�rio
	 * 
	 * @param jokers o conjunto de jokers representando os argumentos
	 * 		 formais
	 * @return um conjunto de Binding com a unificação dos argumetos 
	 * 		 formais com os reais
	 * @throws CancelledApplException excess�o disparada quando o 
	 *  usu�rio cancela a aplicação de uma lei
	 * @throws ParseArgumentException caso algum argumento n�o seja 
	 *  corretamente <i>parseado</i>  
	 */
	public Set<Binding> unificarParametros(List<Term> jokers, 
			LawAnswer resposta) throws CRulesException, 
			CancelledApplException {
		List<String> args = 
			this.getGerInterface().mostrarTelaParametros(jokers);
		return this.unificarParametros(jokers, args.listIterator(), resposta);
	}

	public Set<Binding> unificarParametrosTactic(List<Term> jokers, List<String> args,
			LawAnswer resposta) throws CRulesException, 
			CancelledApplException {

		return this.unificarParametros(jokers, args.listIterator(), resposta);
	}

	/**
	 * Unifica os par�metros
	 * 
	 * @param jokers
	 * @param args
	 * @return
	 * @throws CommandException 
	 * @throws UnmarshalException 
	 * @throws IOException 
	 * @throws ParseException 
	 */
	public Set<Binding> unificarParametros(List<Term> jokers, 
			ListIterator<String> itr, LawAnswer resposta) 
			throws CRulesException {

		Set<Binding> result = new HashSet<Binding>();

		/*
		 * Executa o parse nos argumento
		 */
		for (Term joker : jokers) {
			try {
				String arg = itr.next();

				/* adicionando a string aos par�matros da lei */
				resposta.getParametrosUtilizados().add(arg);

				/* retorna o c�digo do joker */
				CodNameParam cod = CodParamVisitor.getCode(joker);

				/* Executa o parse no argumento */
				Term parsedArg = 
					ParseArgumentUtils.parseArguments(arg, cod.getCodigo());
				Binding bind = BindingGetter.getBinding(joker, parsedArg);
				result.add(bind);
			}
			catch (Exception e){
				throw new CRulesException(e);
			}

		}

		/* Retorna os bindings */
		return result;
	}


	public Set<Binding> unificarParametrosTatica(List<Term> jokers, 
			ListIterator<Object> itr, LawAnswer resposta,List<Pair<String,Term>> mapeamento) 
			throws CRulesException {

		Set<Binding> result = new HashSet<Binding>();

		/*
		 * Executa o parse nos argumento
		 */
		for (Term joker : jokers) {
			try {
				String arg = "";
				//itr.next();

				/* adicionando a string aos par�matros da lei */
				resposta.getParametrosUtilizados().add(arg);

				/* retorna o c�digo do joker */
				CodNameParam cod = CodParamVisitor.getCode(joker);

				/* Executa o parse no argumento */
				Term parsedArg = 
					ParseArgumentUtils.parseArguments(arg, cod.getCodigo());
				Binding bind = BindingGetter.getBinding(joker, parsedArg);
				result.add(bind);
			}
			catch (Exception e){
				throw new CRulesException(e);
			}

		}

		/* Retorna os bindings */
		return result;
	}



	public int getHistoricSizeOfFile(String fileName) throws IOException {
		ToSave historico = this.getStrgUtils().abrirPrograma(new File(fileName));
		return historico.getHistorico().size();
	}


	/**
	 * Metodo que realiza as adaptacoes para passar pro Provador
	 * e obtem o resultado do provador
	 * 
	 * @param pred o predicado a ser avaliado
	 * @return o status da OP de acordo com o resultado do provador
	 */
	private OPTipos resultProver(Pred pred) {
		OPTipos result = Provador.prove( pred );
		return result;
	}

	/**
	 * M�todo utilizado para remover um par�grafo inserido
	 */
	public void removeParagraph(String userInput) 
	throws CRulesException, SpecificationEditorException {
		SpecEditor ins = SpecEditor.getInstance();

		ins.removePararagraph( userInput , 
				(Spec) this.gerInterface.retornarProgAtual() , 
				InternalManager.getManager() );

	}
	/**
	 * 
	 * @return a corrente AST
	 */
	public Term retornarProgAtual() {
		return gerInterface.retornarProgAtual();
	}


	/**
	 * 
	 * @param passos passos do subdesenvolvimento a ser carregado
	 * @throws CRefineException
	 */
	public void abrirSubDesenvolvimento(LinkedList<StepOfExecution> historico) throws CRefineException{
		try {
			this.gerInterface.retornarTelaPrincipal().setarMaximumProgressBar(historico.size());
			/* Reexecuta os passos */
			for (StepOfExecution step : historico) {
				step.doStep(this.getGerInterface());
				this.gerInterface.retornarTelaPrincipal().incrementProgress();
			}
		}
		catch (	Exception e){
			e.printStackTrace();
			throw new OpenSpecificationException(e);
		}

	}


	/**
	 * Metodo utilizado para se guardar o contexto de uma 
	 *  Obrigação de Prova
	 * 
	 * @return A copia da AST corrente
	 */
	public Term retornarContexto() {
		/* Deve-se retornar uma copia ao inves da referencia*/
		Term contexto   = this.retornarProgAntesDeAplicacao();
		return ClonerVisitor.cloneTerm(contexto);
	}


	private Term retornarProgAntesDeAplicacao() {

		return this.gerInterface.retornarProgAntesDeAplicacao();
	}


	public boolean isSubDevelopment() {
		return this.gerInterface.isSubDevelopment(gerInterface.getSelectedDevelopment());
	}

	public boolean isSubDevelopment(int i) {
		return this.gerInterface.isSubDevelopment(i);
	}

	public int getSelectedDevelopment() {
		return this.gerInterface.getSelectedDevelopment();
	}

	/**
	 * @param selectedDevelopment Desenvolvimento Selecionado
	 * que deseja-se descobrir qual � o pai
	 * @return O indice do Pai do Desenvolvimento selecionado
	 */
	public int getPai(int selectedDevelopment) {
		return ((PODevelopmentTree)gerDevelopments.getDevelopment(selectedDevelopment)).getPai();
	}


	/**
	 * 
	 * @param selectedDevelopment
	 * @return O indice da OP que gerou esse desenvolvimento no 
	 * desenvolvimento pai
	 */
	public int getIndiceOP(int selectedDevelopment) {
		return ((PODevelopmentTree)gerDevelopments.getDevelopment(selectedDevelopment)).getIdxOp();
	}

	/**
	 * 
	 * @param pai
	 * @param indiceOp
	 * @return o Contexto em que foi gerado a OP de indiceOP
	 * do desenvolvimento indexado pelo valor de pai
	 */
	public Term getContextoAt(int pai, int indiceOp) {

		return gerInterface.getContextoAt(pai, indiceOp);
	}

	/**
	 * 
	 * @return A lista de leis carregadas
	 */
	public List<CircusLaw> getLaws() {
		return laws;
	}


	public Term getContextoAposAt(int pai, int indiceOp) {
		return gerInterface.getContextoAposAt(pai, indiceOp);
	} 	

	/**
	 * Método responsavel por realizar o parser na criação de uma tática
	 * @param file Arquivo que será realizado o parser
	 * @return res A tática 
	 */
	public Tatica parserTatica(File file) throws TacticNotFound,LawNotFound{

		Tatica res = gerParser.parser2(file, this);
		return res;
	}

	/**
	 * Método que retorna a lista de Táticas do sistema
	 * @return lista de Tática
	 */

	public List<Tatica> retornarListaTaticas() {
		List<Tatica> lista = gerTaticas.getListaTaticas();
		return lista;
	}

	/**
	 * Método que adiciona uma tática no sistema
	 */
	public void adicionarTatica(Tatica tatica) {
		gerTaticas.adicionarTatica(tatica);

	}


	/**
	 * Método responsavel por verificar se ocorreu algum erro 
	 * no parser
	 * @return true Caso não ocorreu erro
	 */
	public boolean verificarErroParser() {
		if (gerParser.verificarErroParser())
			return true;
		else
			return false;
	}


	/**
	 * Método responsavel por retornar o código ArcAngelC de um Arquivo
	 * @param source O arquivo que o parser será realizado
	 * @return str A string contendo o códiga da tática
	 * @throws IOException
	 */
	public String retornarStringEspecificacaoArcAngelC(Source source)
	throws IOException {
		Reader reader = source.getReader();

		int temp;
		StringBuilder str = new StringBuilder();
		while ((temp = reader.read()) != -1) {
			str.append((char) (temp));
		}

		return str.toString();
	}

	/**
	 * Método responsavel por escrever o código ArcAngelC no arquivo
	 * @param file O arquivo que rceberá a especificação ArcAngelC
	 * @param codigoArcAngel A especificação ArcAngelC
	 * @throws IOException
	 */
	public void escreverArquivo(File file, String codigoArcAngel) throws IOException {
		gerTaticas.escreverArquivo(file, codigoArcAngel);

	}

	/**
	 * Método que remove uma Tátia
	 * @param tatica a ser revomida
	 */
	public void removerTatica(Tatica tatica) {
		gerTaticas.removerTatica(tatica);

	}

	/**
	 * Método responsavel por verificar se o Term selecionado
	 * é um CircusProcess, CircusAction, IndexedProcessImpl entre outros
	 * que será utilizado pelo operadores 
	 * @param term O term selecionado pelo usuário
	 * @return true Case seja um CircusProcess ou um CircusAction
	 */
	public boolean verificarPrograma(Term term) {

		if (term instanceof CircusProcess ||
				term instanceof CircusAction)
			return true;

		else if (term instanceof IndexedProcessImpl)
			return true; 

		else if (term instanceof IfGuardedCommandImpl)
			return true;

		else if (term instanceof ParamActionImpl
				|| term instanceof ParamProcessImpl)
			return true;

		return false;
	}

	/**
	 * Método responsavel por verificar se o Term selecionado
	 * é um  ParamActionImpl or ParamProcessImpl
	 * que será utilizado pelo operadores 
	 * @param term O term selecionado pelo usuário
	 * @return true Case seja um CircusProcess ou um CircusAction
	 */
	public int verificarProgramaParam(Term term) {

		if (term instanceof ParamActionImpl)
			return 0;
		else if (term instanceof ParamProcessImpl)
			return 1;
		return -1;
	}


	/**
	 * Método responsável por verificar se o Term selecionado
	 * é um PrefixingActionImpl, GuardedActionImpl, CircusProcess, CircusAction
	 * entre outros
	 * @param noPrograma O programa preenchido com o term selecionado 
	 * @return term. List dos filhos do term
	 */
	public List<Term> verificarOperador(NoPrograma noPrograma){

		List<Term> filhos = new ArrayList<Term>();
		ChildrenTermExtractor extrator = new ChildrenTermExtractor();

		if (noPrograma.getPrograma() instanceof PrefixingActionImpl){ 

			filhos = noPrograma.getPrograma().accept(extrator);
		}

		else if (noPrograma.getPrograma() instanceof SeqActionImpl){

			//filhos = noPrograma.getPrograma().accept(extrator);
			SeqActionImpl seq = (SeqActionImpl) noPrograma.getPrograma();
			filhos.add(seq.getLeftAction());
			filhos.add(seq.getRightAction());
		}


		else if (noPrograma.getPrograma() instanceof ProcessParaImpl){

			filhos = noPrograma.getPrograma().accept(extrator);
		}

		else if (noPrograma.getPrograma() instanceof VarDeclCommandImpl){

			filhos = noPrograma.getPrograma().accept(extrator);
		}

		else if (noPrograma.getPrograma() instanceof BasicProcessImpl){

			filhos = noPrograma.getPrograma().accept(extrator);
		}

		else if (noPrograma.getPrograma() instanceof GuardedActionImpl){ 

			filhos = noPrograma.getPrograma().accept(extrator);

		}

		else if (noPrograma.getPrograma() instanceof MuActionImpl){

			filhos = noPrograma.getPrograma().accept(extrator);
			/*MuActionImpl mu = (MuActionImpl) noPrograma.getPrograma();
			filhos.add(mu.getCircusAction());
			filhos.add(mu.getName());*/
		}

		else if(noPrograma.getPrograma() instanceof ParamActionImpl
				|| noPrograma.getPrograma() instanceof ParamProcessImpl){

			filhos = noPrograma.getPrograma().accept(extrator);
		}

		else if (noPrograma.getPrograma() instanceof VarDeclCommandImpl){

			filhos = noPrograma.getPrograma().accept(extrator);
		}

		else if (noPrograma.getPrograma() instanceof IndexedProcessImpl){

			filhos = noPrograma.getPrograma().accept(extrator);
		}

		else if(noPrograma.getPrograma() instanceof HideActionImpl
				|| noPrograma.getPrograma() instanceof HideProcessImpl){

			filhos = noPrograma.getPrograma().accept(extrator);
		}

		else if(noPrograma.getPrograma() instanceof AndPredImpl){
			AndPredImpl and = (AndPredImpl) noPrograma.getPrograma();
			and.getRightPred();
			and.getLeftPred();
			and.getPred();
			filhos = noPrograma.getPrograma().accept(extrator);
			//ParseArgumentUtils.parseArguments(str, cod)
		}

		/**
		 * Para o TDefsbox
		 */
		else if (noPrograma.getPrograma() instanceof CircusProcess 
				|| noPrograma.getPrograma() instanceof CircusAction){
			filhos = noPrograma.getPrograma().accept(extrator);
		}

		return filhos;

	}
	
	public String verificarTipo(String prog) {

		String[] inicio = prog.split(" ");
		String res = "";

		if(inicio[0].equalsIgnoreCase("circprocess"));{
			res = processoParser(inicio);
		}

		return res;


	}
	
	private String processoParser(String[] processo){
		String corpoProcesso = " ";
		int count = 0, tam = processo.length;

		for (int i = 3; i < processo.length; i++) {
			if (processo[i].equals("circbegin")){
				count = i;
				break;
			}
		}
		while(count<tam){
			corpoProcesso += " " + processo[count];
			count++;
		}

		return corpoProcesso;


	}
	
	private String verificarCircus(String str) {
		String aux = "";
		String result = new String();
		LatexToCircusTactics CircusLatex = new LatexToCircusTactics();
		ZtoLatex Zlatex = new ZtoLatex();

		str = str.trim();
		String res2[] = str.split(" ");

		for (String s : res2) {
			if (Zlatex.get(s)!= null) {
				result += Zlatex.get(s).toLowerCase();
			}
			else if (CircusLatex.get(s) != null){
				result += "\\"+s;

			}
			else result += s;
			result += " ";
		}

		if (!aux.equals("")){
			if (CircusLatex.get(aux) != null)
				result = result.replace(aux, "\n\\"+aux);
		}

		return result;
	}



	public List<Pair<String,Term>> verificarOperadorApplies(NoPrograma noPrograma,String prog) throws CRulesException, Unification{

		List<Term> filhos = null;
		List<Pair<String,Term>> mapeamento =  new ArrayList<Pair<String,Term>>();

		ChildrenTermExtractor extrator = new ChildrenTermExtractor();
		Term newTerm;

		prog = this.verificarTipo(prog);
		prog = this.verificarCircus(prog);


		/**
		 * acoes
		 */

		if (noPrograma.getPrograma() instanceof SkipActionImpl
				|| noPrograma.getPrograma() instanceof StopActionImpl
				|| noPrograma.getPrograma() instanceof ChaosActionImpl){

			newTerm = parseAction(prog);
			mapeamento = this.criarJokersBasic(noPrograma.getPrograma(), newTerm);

		}

		else if (noPrograma.getPrograma() instanceof GuardedActionImpl){

			newTerm = parseAction(prog);
			try {
				mapeamento =  this.criarJokersGuard(noPrograma.getPrograma(), newTerm);
			} catch (Exception e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			}

		}

		else if (noPrograma.getPrograma() instanceof Action1Impl
				|| noPrograma.getPrograma() instanceof Action2Impl){

			newTerm = parseAction(prog);
			mapeamento = this.criarJokers(noPrograma.getPrograma(), newTerm);		
		}

		/**
		 * command
		 */

		else if (noPrograma.getPrograma() instanceof VarDeclCommandImpl){

			newTerm = parseAction(prog);
			this.criarJokersVar(noPrograma.getPrograma(), newTerm);

		}

		else if (noPrograma.getPrograma() instanceof SchExprAction){

			newTerm = parseAction(prog);
			mapeamento = this.criarJokersAcao(noPrograma.getPrograma(), newTerm);

		}

		else if(noPrograma.getPrograma() instanceof ParamActionImpl){

			newTerm = parseProcess(prog);
			ParamActionImpl p = (ParamActionImpl) noPrograma.getPrograma();
			p.getCircusAction();
			p.getDeclList();
			mapeamento = this.criarJokers(noPrograma.getPrograma(), newTerm);
		}

		else if (noPrograma.getPrograma() instanceof BasicProcessImpl){
			newTerm = parseProcess(prog);
			mapeamento = this.criarJokers(noPrograma.getPrograma(), newTerm);
		}

		else if (noPrograma.getPrograma() instanceof ProcessParaImpl){
			newTerm = parseProcessPara(prog);
			mapeamento = this.criarJokers(noPrograma.getPrograma(), newTerm);
		}

		else if (noPrograma.getPrograma() instanceof VarDeclCommandImpl){

			filhos = noPrograma.getPrograma().accept(extrator);
		}




		else if(noPrograma.getPrograma() instanceof ParallelProcessImpl){
			newTerm = this.parseAction(prog);
			mapeamento = this.criarJokers(noPrograma.getPrograma(), newTerm);
		}

		else if(noPrograma.getPrograma() instanceof ParallelProcessImpl){
			AndPredImpl and = (AndPredImpl) noPrograma.getPrograma();
			and.getRightPred();
			and.getLeftPred();
			and.getPred();
			filhos = noPrograma.getPrograma().accept(extrator);
		}


		/**
		 * Para o TDefsbox
		 */
		else if (noPrograma.getPrograma() instanceof CircusProcess 
				|| noPrograma.getPrograma() instanceof CircusAction){
			filhos = noPrograma.getPrograma().accept(extrator);
		}



		return mapeamento;

	}

	private List<Pair<String, Term>> criarJokers(Term programa, Term newTerm) throws Unification {

		List<Pair<String,Term>> list = new ArrayList<Pair<String,Term>>();

		list = visitTerm(programa, newTerm);

		return list;
	}

	private boolean criarJokersVar(Term programa, Term newTerm) throws Unification {
		Set<Binding> unificacao = null;
		boolean res = false;

		try {

			VarDeclCommandImpl var1 = (VarDeclCommandImpl) programa;
			VarDeclCommandImpl var2 = (VarDeclCommandImpl) newTerm;

			VarDeclCommand v1 = factory.createVarDeclCommand(var1.getZDeclList(),var1.getCircusAction());
			VarDeclCommand v2 = factory.createVarDeclCommand(var2.getZDeclList(),var1.getCircusAction());

			//unificacao = UnifierVisitor.unify(programa,newTerm);
			unificacao = UnifierVisitor.unify(v1,v2);

			res = true;

		} catch (UnificationException e) {
			throw new Unification("Termo selecionado: " + programa.toString()+ "Term Tática: " + newTerm.toString());
		}

		return res;
	}

	private List<Pair<String,Term>>  criarJokersGuard(Term programa, Term newTerm) throws Exception {
		Set<Binding> unificacao = null;
		List<Pair<String,Term>> list = new ArrayList<Pair<String,Term>>();

		try {

			GuardedActionImpl g1 = (GuardedActionImpl) programa;
			GuardedActionImpl g2 = (GuardedActionImpl) newTerm;

			GuardedAction ga1 = factory.createGuardedAction(g1.getCircusAction(), g1.getPred());
			GuardedAction ga2 = factory.createGuardedAction(g2.getCircusAction(), g2.getPred());

			unificacao = UnifierVisitor.unify(ga1,ga2);

			Pred pred  = g2.getPred();
			Pred pred2 = g1.getPred();

			if (pred instanceof MemPredImpl){
				MemPredImpl mem = (MemPredImpl) pred;
				MemPredImpl mem2 = (MemPredImpl) pred2;

				if (mem2.getLeftExpr() instanceof TupleExprImpl){
					TupleExprImpl t = (TupleExprImpl) mem.getLeftExpr();
					TupleExprImpl t2 = (TupleExprImpl) mem2.getLeftExpr();
					if (t.equals(t2))

						//o toString(0 retorna second,0 então é só separar por virgulas e tem a expressao
						t.getExprList();
					t.getZExprList();
					pred.getChildren();
					pred.toString();
					RefExprImpl ref = (RefExprImpl) mem.getRightExpr();
					ref.getExprList();
					ref.getZName().getWord();
				}
				else if (mem2.getLeftExpr() instanceof RefExprImpl){
					RefExprImpl ref = (RefExprImpl) mem.getLeftExpr();
					RefExprImpl ref2 = (RefExprImpl) mem2.getLeftExpr();
					ref.getExprList();
					ref.getZName();
					ref.getZExprList();
					ref2.getZName();

				}



			}
			else if (pred instanceof AndPredImpl){
				AndPredImpl ref = (AndPredImpl) pred;
				AndPredImpl ref2 = (AndPredImpl) pred2;
				ref.getLeftPred();
				ref2.getRightPred().toString();
				MemPredImpl mem = (MemPredImpl)  ref.getLeftPred();
				MemPredImpl mem2 = (MemPredImpl) ref2.getLeftPred();
				mem.getLeftExpr();
				mem.getRightExpr().toString();
				mem.getExpr().toArray();
				List<Expr> l = mem.getExpr();
				l.get(0);
				TupleExprImpl tu = (TupleExprImpl) mem.getExpr();
				tu.getZExprList().get(0);
				tu.getZExprList().get(1);
				ref.getLeftPred();
				ref.getRightPred();
				//List t = ref.getPred();
				ref.getAnd();
				ref2.getLeftPred();
				ref2.getRightPred();
				ref2.getAnd();

				TupleExprImpl t = (TupleExprImpl) ref.getLeftPred();
				TupleExprImpl t2 = (TupleExprImpl) ref2.getLeftPred();
				if (t.equals(t2))

					//o toString(0 retorna second,0 então é só separar por virgulas e tem a expressao

					t.getExprList();
				t.getZExprList();
				pred.getChildren();
				pred.toString();
				/*RefExprImpl ref2 = (RefExprImpl) t.getRightExpr();
				ref2.getExprList();
				ref2.getZName().getWord();*/
			}
			else if (pred instanceof OrPredImpl){
				OrPredImpl ref = (OrPredImpl) pred;
				OrPredImpl ref2 = (OrPredImpl) pred2;
				ref.getLeftPred();
				ref.getRightPred();
				List t = ref.getPred();

				ref2.getLeftPred();
				ref2.getRightPred();
				if (!ref.equals(ref2))
					throw new Exception("Predicados diferentes");

			}

			else if (pred instanceof ImpliesPredImpl){
				ImpliesPredImpl ref = (ImpliesPredImpl) pred;
				ImpliesPredImpl ref2 = (ImpliesPredImpl) pred2;
				ref.getLeftPred();
				ref.getRightPred();
				List t = ref.getPred();

				ref2.getLeftPred();
				ref2.getRightPred();
				if (!ref.equals(ref2))
					throw new Exception("Predicados diferentes");

			}

			list = visitTerm(programa, newTerm);

		} catch (UnificationException e) {
			throw new Unification("Termo selecionado: " + programa.toString()+ "Term Tática: " + newTerm.toString());
		}

		return list;

	}

	public List<Pair<String, Term>> visitTerm(Term lei, Term real) throws Unification {

		Pair<String, Term> mapeamento;
		String nomeProc = "";

		List<Pair<String, Term>> res = new ArrayList<Pair<String,Term>>();
		if (lei.getClass().isInstance(real)) {

			/* Testa o numero de filhos */
			if (lei.getChildren().length == real.getChildren().length) {


				/* Extrai os filhos */
				ChildrenTermExtractor extrator = new ChildrenTermExtractor();


				List<Term> childrenLaw = lei.accept(extrator);
				List<Term> childrenTerm = real.accept(extrator);

				Vector<List<Term>> listas = new Vector<List<Term>>();

				listas = verificarTipoEspecial(lei,real);

				if (listas.size() > 0){
					childrenTerm =  new ArrayList<Term>();
					childrenLaw  =  new ArrayList<Term>();
					childrenLaw   = listas.get(0);
					childrenTerm  = listas.get(1);
				}

				Iterator<Term> childrenTermItr = childrenTerm.iterator();

				/* Testa os tamanhos das listas */
				if (childrenLaw.size() != childrenTerm.size()) {
					throw new Unification("Impossível realizar a unificação: Quantidade de filhos diferente" 
							+ "Termo selecionado: " + real + "Term Tática: " + lei);
				}

				if (lei instanceof BasicProcessImpl){
					BasicProcess bpLei = (BasicProcess) lei;

					Term ast = this.gerInterface.getProgramaAtual();
					List<ProcessPara> proP = retornarProcessPara(ast);
					for (ProcessPara processPara : proP) {
						if (processPara.getCircusBasicProcess().getMainAction().
								equals(bpLei.getMainAction())){
							ProcessPara pp = (ProcessPara) processPara;
							nomeProc = processPara.getZName()+"";
							mapeamento = new Pair<String, Term>();
							mapeamento.setFirst("P");
							mapeamento.setSecond(processPara.getZName());
							res.add(mapeamento);
							break;
						}
					}

				}


				/* Constroi o resultado */			
				for (Term termLaw : childrenLaw) {
					Term termReal = childrenTermItr.next();
					if (termLaw != null) {
						mapeamento = new Pair<String, Term>();
						mapeamento.setFirst(termReal.toString());
						mapeamento.setSecond(termLaw);
						res.add(mapeamento);
					} 
				}

			} else 
				throw new Unification("Impossível realizar a unificação: Quantidade de filhos diferente" 
						+ "do Termo selecionado: " + real + " com o Term Tática: " + lei);


		}else
			throw new Unification("Impossível realizar a unificação" 
					+ "do Termo selecionado: " + real + " com o Term da Tática: " + lei);

		return res;

	}

	private Vector<List<Term>> verificarTipoEspecial(Term lei,Term real) {

		List<Term> t1 = new ArrayList<Term>();
		List<Term> t2 = new ArrayList<Term>();
		Vector<List<Term>> listas = new Vector<List<Term>>();

		if (lei instanceof SeqActionImpl){

			SeqActionImpl seq  = (SeqActionImpl) lei;
			SeqActionImpl seq2 = (SeqActionImpl) real;
			t1.add(seq.getLeftAction());
			t1.add(seq.getRightAction());
			t2.add(seq2.getLeftAction());
			t2.add(seq2.getRightAction());


		}

		else if (lei instanceof SeqActionIteImpl){

			SeqActionIteImpl seq  = (SeqActionIteImpl) lei;
			SeqActionIteImpl seq2 = (SeqActionIteImpl) real;
			t1.add(seq.getCircusAction());
			t1.add(seq.getDeclList());
			t2.add(seq2.getCircusAction());
			t2.add(seq2.getDeclList());
		}
		else if(lei instanceof ExtChoiceActionImpl){
			ExtChoiceAction ext  = (ExtChoiceAction) lei;
			ExtChoiceAction ext2 = (ExtChoiceAction) real;
			t1.add(ext.getLeftAction());
			t1.add(ext.getRightAction());
			t2.add(ext2.getLeftAction());
			t2.add(ext2.getRightAction());
		}
		else if (lei instanceof ExtChoiceActionIte){

			ExtChoiceActionIte seq  = (ExtChoiceActionIte) lei;
			ExtChoiceActionIte seq2 = (ExtChoiceActionIte) real;
			t1.add(seq.getCircusAction());
			t1.add(seq.getDeclList());
			t2.add(seq2.getCircusAction());
			t2.add(seq2.getDeclList());
		}


		else if(lei instanceof IntChoiceActionImpl){
			IntChoiceAction intC  = (IntChoiceAction) lei;
			IntChoiceAction intC2 = (IntChoiceAction) real;
			t1.add(intC.getLeftAction());
			t1.add(intC.getRightAction());
			t2.add(intC2.getLeftAction());
			t2.add(intC2.getRightAction());
		}

		else if (lei instanceof IntChoiceActionIte){

			IntChoiceActionIte seq  = (IntChoiceActionIte) lei;
			IntChoiceActionIte seq2 = (IntChoiceActionIte) real;
			t1.add(seq.getCircusAction());
			t1.add(seq.getDeclList());
			t2.add(seq2.getCircusAction());
			t2.add(seq2.getDeclList());
		}

		else if(lei instanceof ParallelActionImpl){
			ParallelActionImpl par1  = (ParallelActionImpl) lei;
			ParallelActionImpl par2 = (ParallelActionImpl) real;
			t1.add(par1.getLeftAction());
			t1.add(par1.getRightAction());
			t1.add(par1.getLeftNameSet());
			t1.add(par1.getRightNameSet());
			t1.add(par1.getChannelSet());
			t2.add(par2.getLeftAction());
			t2.add(par2.getRightAction());
			t2.add(par2.getLeftNameSet());
			t2.add(par2.getRightNameSet());
			t2.add(par2.getChannelSet());	
		}

		else if(lei instanceof HideActionImpl){
			HideActionImpl par1  = (HideActionImpl) lei;
			HideActionImpl par2 = (HideActionImpl) real;
			t1.add(par1.getCircusAction());
			t1.add(par1.getChannelSet());
			t2.add(par2.getCircusAction());
			t2.add(par2.getChannelSet());	
		}

		else if(lei instanceof InterleaveActionImpl){
			InterleaveActionImpl inter1  = (InterleaveActionImpl) lei;
			InterleaveActionImpl inter2  = (InterleaveActionImpl) real;
			t1.add(inter1.getLeftAction());
			t1.add(inter1.getRightAction());
			t1.add(inter1.getLeftNameSet());
			t1.add(inter1.getRightNameSet());
			t2.add(inter2.getLeftAction());
			t2.add(inter2.getRightAction());
			t2.add(inter2.getLeftNameSet());
			t2.add(inter2.getRightNameSet());
		}

		else if(lei instanceof InterleaveActionIteImpl){
			InterleaveActionIteImpl par1  = (InterleaveActionIteImpl) lei;
			InterleaveActionIteImpl par2 =  (InterleaveActionIteImpl) real;
			t1.add(par1.getCircusAction());
			t1.add(par1.getNameSet());
			t1.add(par1.getDeclList());
			t2.add(par2.getCircusAction());
			t2.add(par2.getNameSet());	
			t2.add(par2.getDeclList());
		}

		else if(lei instanceof ParallelProcessImpl){
			ParallelProcessImpl par1  = (ParallelProcessImpl) lei;
			ParallelProcessImpl par2 = (ParallelProcessImpl) real;
			t1.add(par1.getLeftProcess());
			t1.add(par1.getRightProcess());
			t1.add(par1.getChannelSet());
			t2.add(par2.getLeftProcess());
			t2.add(par2.getRightProcess());
			t2.add(par2.getChannelSet());

		}

		if (t1.size() >0 ) {
			listas.add(t1);
			listas.add(t2);			
		}

		return listas;
	}


	private List<Pair<String,Term>> criarJokersBasic(Term programa, Term newTerm) throws Unification {

		Set<Binding> unificacao = null;
		List<Pair<String,Term>> list = new ArrayList<Pair<String,Term>>();
		Pair<String, Term> mapeamento = new Pair<String, Term>();

		/**
		 * Criando o JokerAction
		 */
		try {

			if (programa instanceof SkipActionImpl){
				SkipActionImpl skip1 = (SkipActionImpl) newTerm;
				SkipActionImpl skip2 = (SkipActionImpl) programa;
				unificacao = UnifierVisitor.unify(skip1,skip2);
				mapeamento.setFirst("Skip");
			}
			else if(programa instanceof StopActionImpl){
				StopActionImpl stop1 = (StopActionImpl) newTerm;
				StopActionImpl stop2 = (StopActionImpl) programa;
				unificacao = UnifierVisitor.unify(stop1,stop2);
				mapeamento.setFirst("Stop");
			}
			else if(programa instanceof ChaosActionImpl){
				ChaosActionImpl chaos1 = (ChaosActionImpl) newTerm;
				ChaosActionImpl chaos2 = (ChaosActionImpl) programa;
				unificacao = UnifierVisitor.unify(chaos1,chaos2);
				mapeamento.setFirst("Chaos");
			}


			/**
			 * Adicionando o par na lista
			 */

			mapeamento.setSecond(programa);
			list.add(mapeamento);

		} catch (UnificationException e) {
			throw new Unification("Termo selecionado: " + programa.toString()+ "Term Tática: " + newTerm.toString());
		}

		return list;


	}

	/**
	 * Método que retornar a lista de Ações e cria um RCell para cada ação
	 * @param term O term 
	 * @return list RCell com todas as ações
	 */
	public Vector retornarListaAcoes(Term term){

		Vector list = new Vector();
		CircusAction leftAction = null, rightAction =  null;
		CircusProcess leftProc = null, rightProc = null;
		NoPrograma noProg1 = null, noProg2 = null;

		if (term instanceof SeqActionImpl){
			SeqActionImpl seq = (SeqActionImpl) term;
			leftAction = seq.getLeftAction();
			rightAction = seq.getRightAction();
		}

		else if (term instanceof SeqProcessImpl){
			SeqProcessImpl seq = (SeqProcessImpl) term;
			leftProc = seq.getLeftProcess();
			rightProc = seq.getRightProcess();
		}

		else if (term instanceof ExtChoiceActionImpl){
			ExtChoiceActionImpl ext = (ExtChoiceActionImpl) term;
			leftAction = ext.getLeftAction();
			rightAction = ext.getRightAction();
		}

		else if (term instanceof ExtChoiceProcessImpl){
			ExtChoiceProcessImpl ext = (ExtChoiceProcessImpl) term;
			leftProc = ext.getLeftProcess();
			rightProc = ext.getRightProcess();

		}

		else if (term instanceof IntChoiceActionImpl){
			IntChoiceActionImpl intC = (IntChoiceActionImpl) term;
			leftAction = intC.getLeftAction();
			rightAction = intC.getRightAction();
		}

		else if (term instanceof IntChoiceProcessImpl){
			IntChoiceProcessImpl intC = (IntChoiceProcessImpl) term;
			leftProc = intC.getLeftProcess();
			rightProc = intC.getRightProcess();
		}


		else if (term instanceof ParallelActionImpl){
			ParallelActionImpl par = (ParallelActionImpl) term;
			leftAction = par.getLeftAction();
			rightAction = par.getRightAction();
		}

		else if (term instanceof ParallelProcessImpl){
			ParallelProcessImpl par = (ParallelProcessImpl) term;
			leftProc = par.getLeftProcess();
			rightProc = par.getRightProcess();
		}

		else if (term instanceof InterleaveActionImpl){
			InterleaveActionImpl interL = (InterleaveActionImpl) term;
			leftAction = interL.getLeftAction();
			rightAction = interL.getRightAction();
		}

		else if (term instanceof InterleaveProcessImpl){
			InterleaveProcessImpl interL = (InterleaveProcessImpl) term;
			leftProc = interL.getLeftProcess();
			rightProc = interL.getRightProcess();
		}

		/*		else if (term instanceof VarDeclCommand){
			VarDeclCommand interL = (VarDeclCommand) term;
			leftAction = interL.getCircusAction();


		}*/

		if (leftAction != null){
			noProg1 = new NoPrograma(leftAction);
			noProg2 = new NoPrograma(rightAction);
		}

		else if (leftProc != null){
			noProg1 = new NoPrograma(leftProc);
			noProg2 = new NoPrograma(rightProc);
		}

		/**
		 * Criando os RCell´s
		 */
		RCell novoRCell1 = new RCell(noProg1);
		list.addElement(novoRCell1);


		RCell novoRCell2 = new RCell(noProg2);
		list.addElement(novoRCell2);

		return list;
	}


	public Vector getHideAction(Term term){

		CircusAction cAction =  null;
		CircusProcess cProcess = null;
		ChannelSet cs = null;
		Vector list = new Vector();
		NoPrograma novoP = null;
		RCell novoRCell;

		if (term instanceof HideActionImpl){
			HideActionImpl hideA = (HideActionImpl) term;
			cAction = hideA.getCircusAction();
			List l = hideA.getAnns();
			cs = hideA.getChannelSet();
			novoP = new NoPrograma(cAction);

		}
		else if (term instanceof HideProcessImpl){
			HideProcessImpl hideP = (HideProcessImpl) term;
			cProcess = hideP.getCircusProcess();
			List l = hideP.getAnns();
			cs = hideP.getChannelSet();
			novoP = new NoPrograma(cProcess);
		}
		/**
		 * Criando um novo NoPrograma e RCell
		 */


		novoRCell = new RCell(novoP);
		list.addElement(novoRCell);


		return list;
	}
	/**
	 * Método que retornar a lista de Ações dos tipos indexados
	 * e cria um RCell para cada ação
	 * @param term O term 
	 * @return list RCell com todas as ações
	 */
	public Vector retornarListaAcoesIde(Term term){


		Vector list = new Vector();
		CircusAction cAction =  null;
		CircusProcess cProcess = null;
		NoPrograma noProg = null;

		if (term instanceof SeqActionIteImpl){
			SeqActionIteImpl seq = (SeqActionIteImpl) term;
			cAction = seq.getCircusAction();
		}

		if (term instanceof SeqProcessIteImpl){
			SeqProcessIteImpl seq = (SeqProcessIteImpl) term;
			cProcess = seq.getCircusProcess();
		}

		else if (term instanceof ExtChoiceActionIte){
			ExtChoiceActionIteImpl ext = (ExtChoiceActionIteImpl) term;
			cAction = ext.getCircusAction();
		}

		else if (term instanceof ExtChoiceProcessIteImpl){
			ExtChoiceProcessIteImpl ext = (ExtChoiceProcessIteImpl) term;
			cProcess = ext.getCircusProcess();
		}

		else if (term instanceof IntChoiceActionIteImpl){
			IntChoiceActionIteImpl intC = (IntChoiceActionIteImpl) term;
			cAction = intC.getCircusAction();
		}

		else if (term instanceof IntChoiceProcessIteImpl){
			IntChoiceProcessIteImpl intC = (IntChoiceProcessIteImpl) term;
			cProcess = intC.getCircusProcess();
		}

		else if (term instanceof ParallelProcessIte){
			ParallelProcessIteImpl par = (ParallelProcessIteImpl) term;
			cProcess = par.getCircusProcess();
		}
		else if (term instanceof InterleaveProcessIdxImpl){
			InterleaveProcessIdxImpl intL = (InterleaveProcessIdxImpl) term;
			cProcess = intL.getCircusProcess();
		}

		if(cAction != null)
			noProg = new NoPrograma(cAction);
		else if(cProcess != null )
			noProg = new NoPrograma(cProcess);

		RCell novoRCell = new RCell(noProg);
		list.addElement(novoRCell);

		return list;
	}

	/**
	 * Método utilizado para montar a resposta de uma aplicação da Tática
	 * @param aux Utilizada para complementar a reconstrução, 
	 * pode ser uma comunicação, um predicado um nome
	 * @param resposta Vetor com as respostas das táticas
	 * @param termInicial Para definir qual o tipo do programa Inicial
	 * @return
	 */
	public Vector montarRespostaUnaria(Term aux, Vector respostas, Term termInicial) {

		Vector resposta = new Vector();
		NoPrograma noPrograma;


		for (int i = 0; i < respostas.size(); i++) {

			RCell rCellResultado = (RCell)respostas.get(i);
			RCell rCellRespostaFinal = this.criarReconstrucaoUnaria(aux, rCellResultado,termInicial);

			resposta.add(rCellRespostaFinal);
		}

		return resposta;

	}

	/**
	 * Método responsavel por montar a resposta se uma tatica binaria
	 * @param rCell1 
	 * @param rCell2
	 * @param programa Identiicador do tipo de programa
	 * @return
	 */
	public RCell montarRespostaBinaria(RCell rCell1, RCell rCell2, Term programa) {

		Vector resposta = new Vector();
		NoPrograma noPrograma;
		Term novoTermo = programa;


		if (programa instanceof SeqActionImpl){
			novoTermo = this.montarSeqAction(rCell1, rCell2, programa);
		}

		else if (programa instanceof ExtChoiceActionImpl){
			novoTermo = this.montarExtChoice(rCell1, rCell2, programa);
		}

		else if (programa instanceof IntChoiceActionImpl){
			novoTermo = this.montarIntChoice(rCell1, rCell2, programa);
		}

		else if (programa instanceof ParallelActionImpl){
			novoTermo = this.montarParallelAction(rCell1, rCell2, programa);
		}

		else if (programa instanceof InterleaveActionImpl){
			novoTermo = this.montarParallelAction(rCell1, rCell2, programa);
		}

		/**
		 * Criando um novo RCell
		 */
		RCell rCellFinal = this.criarRCell(novoTermo);
		return rCellFinal;

	}


	/**
	 * Método que realiza a reconstrução dos combinadores estruturais
	 * @param Term aux
	 * @param RCell 
	 * @return RCell prefixing reconstruido
	 */

	private RCell criarReconstrucaoUnaria(Term aux, RCell rCellResultado,
			Term termInicial) {

		Term termo = rCellResultado.getNoPrograma().getPrograma();
		Term novoTermo = null;


		if (termInicial instanceof PrefixingActionImpl){

			novoTermo = this.criarPrefixing(aux, rCellResultado, termo);
		}

		else if(termInicial instanceof GuardedActionImpl){

			novoTermo = this.criarGuarded(aux, rCellResultado, termo);

		}

		else if(termInicial instanceof MuActionImpl){

			novoTermo = this.criarMuAction(aux, rCellResultado, termo);

		}

		/*else if(termInicial instanceof SeqActionImpl)
		//	novoTermo = this.criarSeqAction(aux, rCellResultado, termo);

		else if(termInicial instanceof ExtChoiceActionImpl)
		//	novoTermo = this.criarExtAction(aux, rCellResultado, termo);
		 */

		/**
		 * Criando um novo RCell;
		 */
		RCell rCellFinal = this.criarRCell(novoTermo,rCellResultado);
		return rCellFinal;

	}

	/**
	 * Método que cria um novo rCell
	 * @param term Term para a crição de um novo rCell
	 * @param rCellResultado Utilizada para pegar as OP´s
	 * @return rCell Novo rCell
	 */
	private RCell criarRCell(Term novoTermo, RCell rCellResultado) {


		NoPrograma novoP = new NoPrograma(novoTermo);
		RCell rCell = new RCell(novoP,rCellResultado.getOps());

		return rCell;
	}

	/**
	 * Método que cria um novo rCell
	 * @param term Term para a crição de um novo rCell
	 * @return rCell Novo rCell
	 */
	private RCell criarRCell(Term novoTermo) {


		NoPrograma novoP = new NoPrograma(novoTermo);
		RCell rCell = new RCell(novoP);

		return rCell;
	}

	/**
	 * Método que realiza a reconstrução do Prefixing
	 * @param Term comunication
	 * @param RCell 
	 * @return Term prefixing reconstruido
	 */
	private Term criarPrefixing(Term comunication, RCell rCellResultado,
			Term prefixing){
		PrefixingActionImpl prefix = (PrefixingActionImpl) prefixing;
		/**
		 * Extraindo os componentes do Prefixing
		 */
		CircusAction acao = (CircusAction) rCellResultado.noPrograma.getPrograma();
		Communication comm = (Communication) comunication;

		/**
		 * Setando o novo Prefixing
		 */
		prefix.setCircusAction(acao);
		prefix.setCommunication(comm);

		return prefix;
	}

	/**
	 * Método que realiza a reconstrução do Prefixing
	 * @param Term predicate
	 * @param RCell 
	 * @return Term guarded reconstruido
	 */
	private Term criarGuarded(Term predicate, RCell rCellResultado,
			Term guarded){

		GuardedActionImpl prefix = (GuardedActionImpl) guarded;
		/**
		 * Extraindo os componentes do Guarded
		 */
		CircusAction acao = (CircusAction) rCellResultado.noPrograma.getPrograma();
		Pred pred = (Pred) predicate;

		/**
		 * Setando o novo Guarded
		 */
		prefix.setCircusAction(acao);
		prefix.setPred(pred);

		return prefix;
	}


	/**
	 * Método utilizado para realizar a reconstrução de um MuAction
	 * @param name
	 * @param rCellResultado
	 * @param muAction
	 * @return
	 */
	public Term criarMuAction(Term name, RCell rCellResultado,
			Term muAction){

		MuActionImpl mu = (MuActionImpl) muAction;

		/**
		 * Extraindo os componentes do MuAction
		 */
		CircusAction acao = (CircusAction) rCellResultado.getNoPrograma().getPrograma();
		Name nameMu = (Name) name;

		/**
		 * Setando o novo Guarded
		 */
		mu.setCircusAction(acao);
		mu.setName(nameMu);

		return mu;
	}


	/**
	 * Método que pega as ações do IfGuardedCommand
	 * @param Term ifGuardedCommand
	 * @return Vector com todos os rCell
	 */

	public Vector retornarGuardedAction(Term ifGuardedCommand) {

		IfGuardedCommandImpl g = (IfGuardedCommandImpl) ifGuardedCommand;
		CircusActionList ga = g.getGuardedActionList();
		Vector list = new Vector();

		/**
		 * Aqui pega os GuardedAction do IfGuardedCommandImpl
		 */
		for(CircusAction act1 : ga) {
			GuardedAction act = (GuardedAction)act1;


			/**
			 * Cria um RCell para cada GuardedAction 
			 */
			NoPrograma noProg = new NoPrograma(act);
			RCell novoRCell = new RCell(noProg);
			list.addElement(novoRCell);

		}

		return list;
	}


	/**
	 * Método que pega as ações do IndexedProcessImpl
	 * @param Term indexedProcess
	 * @return Vector com todos os rCell
	 */

	public Vector retornarAcoesIndexedProcess(NoPrograma noPrograma){

		IndexedProcessImpl ind = (IndexedProcessImpl) noPrograma.getPrograma();
		CircusProcess circusP = ind.getCircusProcess();

		Vector list = new Vector();

		/**
		 * Criando um novo NoPrograma
		 */
		NoPrograma noProg = new NoPrograma(circusP);

		/**
		 * Criando um novo RCell
		 */
		RCell novoRCell = new RCell(noProg);
		list.addElement(novoRCell);


		return list;
	}


	/**
	 * Método que retorna todas as ações ou processos parametrizadas
	 * @param Term ParamProcessImpl or ParamActionImpl
	 * @return Vector com todos os rCell
	 */

	public Vector retornarParamImpl(NoPrograma noPrograma, int res){

		CircusProcess circusP;
		CircusAction circusA;
		NoPrograma noProg = null;

		if (res == 0){
			ParamActionImpl paramA = (ParamActionImpl) noPrograma.getPrograma();
			circusA = paramA.getCircusAction();

			/**
			 * Criando um novo NoPrograma
			 */
			noProg = new NoPrograma(circusA);

		}
		else if (res == 1){
			ParamProcessImpl paramA = (ParamProcessImpl) noPrograma.getPrograma();
			circusP = paramA.getCircusProcess();
			/**
			 * Criando um novo NoPrograma
			 */
			noProg = new NoPrograma(circusP);

		}


		Vector list = new Vector();


		/**
		 * Criando um novo RCell
		 */
		RCell novoRCell = new RCell(noProg);
		list.addElement(novoRCell);


		return list;
	}

	/**
	 * Método que pega as ações do IndexedProcessImpl
	 * @param Term indexedProcess
	 * @return Vector com todos os rCell
	 */

	public Vector retornarAcoesParametrisaProcess(NoPrograma noPrograma){

		IndexedProcessImpl ind = (IndexedProcessImpl) noPrograma.getPrograma();
		CircusProcess circusP = ind.getCircusProcess();

		Vector list = new Vector();

		/**
		 * Criando um novo NoPrograma
		 */
		NoPrograma noProg = new NoPrograma(circusP);

		/**
		 * Criando um novo RCell
		 */
		RCell novoRCell = new RCell(noProg);
		list.addElement(novoRCell);


		return list;
	}


	/**
	 * Método responsavel por montar as respostas do operadores binários
	 * @param rCell1 A primeira parte do resposta
	 * @param rCell2 A primeira parte do resposta
	 * @param term O term que será montado
	 * @return rCell O RCell contndo o term
	 */
	/*public RCell montarRespostaBinaria(RCell rCell1, RCell rCell2, Term term) {


		Vector list = new Vector();
		CircusAction caLeft = null, caRight =  null;

		if (term instanceof SeqActionImpl){

			SeqActionImpl seq = (SeqActionImpl) term;

	 *//**
	 * Criando os CircusAction
	 *//*
			 caLeft = (CircusAction) rCell1.getNoPrograma().getPrograma();
			 caRight = (CircusAction) rCell2.getNoPrograma().getPrograma();

	  *//**
	  * Setando os CircusAction
	  *//*

			left = seq.getLeftAction();
			right = seq.getRightAction();
		}

		else if (term instanceof ExtChoiceAction){
			ExtChoiceActionImpl ext = (ExtChoiceActionImpl) term;
			left = ext.getLeftAction();
			right = ext.getRightAction();
		}

		else if (term instanceof IntChoiceAction){
			IntChoiceActionImpl intC = (IntChoiceActionImpl) term;
			left = intC.getLeftAction();
			right = intC.getRightAction();
		}

		else if (term instanceof ParallelAction){
			ParallelActionImpl par = (ParallelActionImpl) term;
			left = par.getLeftAction();
			right = par.getRightAction();
		}

		else if (term instanceof InterleaveAction){
			InterleaveActionImpl interL = (InterleaveActionImpl) term;
			left = interL.getLeftAction();
			right = interL.getRightAction();
		}


	   *//**
	   * Criando um novo RCell
	   *//*

		 NoPrograma noProgramaSeq = new NoPrograma(se);
		 RCell rCellSeq = new RCell(noProgramaSeq);

		 return novoRCell;
	}
	    */

	/**
	 * Método utilizado para reconstruir a resposta de um SeqActionImpl
	 * @param rCell1
	 * @param rCell2
	 * @param extChoice
	 * @return rCell 
	 */
	public Term montarSeqAction(RCell rCell1, RCell rCell2, Term seqAction) {

		SeqActionImpl se = (SeqActionImpl) seqAction;

		/**
		 * Criando os CircusAction
		 */

		if (rCell1.getNoPrograma().getPrograma() instanceof CallAction){
			CallAction caLeft = (CallAction) rCell1.getNoPrograma().getPrograma();
			se.setLeftAction(caLeft);
		}
		else if (rCell1.getNoPrograma().getPrograma() instanceof ParallelActionImpl){
			ParallelActionImpl caLeft = (ParallelActionImpl) rCell1.getNoPrograma().getPrograma();
			se.setLeftAction(caLeft);
		}

		else if (rCell1.getNoPrograma().getPrograma() instanceof InterleaveActionImpl){
			InterleaveActionImpl caLeft = (InterleaveActionImpl) rCell1.getNoPrograma().getPrograma();
			se.setLeftAction(caLeft);
		}

		else if (rCell1.getNoPrograma().getPrograma() instanceof PrefixingActionImpl){
			PrefixingActionImpl caLeft = (PrefixingActionImpl) rCell1.getNoPrograma().getPrograma();
			se.setLeftAction(caLeft);
		}

		else if (rCell1.getNoPrograma().getPrograma() instanceof MuActionImpl){
			MuActionImpl caLeft = (MuActionImpl) rCell1.getNoPrograma().getPrograma();
			se.setLeftAction(caLeft);
		}

		else if (rCell1.getNoPrograma().getPrograma() instanceof VarDeclCommandImpl){
			VarDeclCommandImpl caLeft = (VarDeclCommandImpl) rCell1.getNoPrograma().getPrograma();
			se.setLeftAction(caLeft);
		}

		else if (rCell1.getNoPrograma().getPrograma() instanceof SeqActionImpl){
			SeqActionImpl caLeft = (SeqActionImpl) rCell1.getNoPrograma().getPrograma();
			se.setLeftAction(caLeft);
		}

		if (rCell2.getNoPrograma().getPrograma() instanceof CallAction){
			CallAction caRight = (CallAction) rCell2.getNoPrograma().getPrograma();
			se.setRightAction(caRight);
		}
		else if (rCell2.getNoPrograma().getPrograma() instanceof InterleaveActionImpl){
			InterleaveActionImpl caRight = (InterleaveActionImpl) rCell2.getNoPrograma().getPrograma();
			se.setLeftAction(caRight);
		}

		else if (rCell2.getNoPrograma().getPrograma() instanceof ParallelActionImpl){
			ParallelActionImpl caRight = (ParallelActionImpl) rCell2.getNoPrograma().getPrograma();
			se.setRightAction(caRight);
		}

		else if (rCell2.getNoPrograma().getPrograma() instanceof PrefixingActionImpl){
			PrefixingActionImpl caRight = (PrefixingActionImpl) rCell2.getNoPrograma().getPrograma();
			se.setRightAction(caRight);
		}

		else if (rCell2.getNoPrograma().getPrograma() instanceof MuActionImpl){
			MuActionImpl caRight = (MuActionImpl) rCell2.getNoPrograma().getPrograma();
			se.setRightAction(caRight);
		}

		else if (rCell2.getNoPrograma().getPrograma() instanceof VarDeclCommandImpl){
			VarDeclCommandImpl caRight = (VarDeclCommandImpl) rCell2.getNoPrograma().getPrograma();
			se.setRightAction(caRight);
		}

		else if (rCell2.getNoPrograma().getPrograma() instanceof SeqActionImpl){
			SeqActionImpl caRight = (SeqActionImpl) rCell2.getNoPrograma().getPrograma();
			se.setRightAction(caRight);
		}
		/*CircusAction caLeft = (CircusAction) rCell1.getNoPrograma().getPrograma();
		CircusAction caRight = (CircusAction) rCell2.getNoPrograma().getPrograma();*/

		/**
		 * Setando os CircusAction
		 */

		/*se.setLeftAction(caLeft);
		se.setRightAction(caRight);*/

		/**
		 * Criando um novo RCell
		 */

		/*NoPrograma noProgramaSeq = new NoPrograma(se);
		RCell rCellSeq = new RCell(noProgramaSeq);
		 */
		return se;
	}


	/**
	 * Método utilizado para reconstruir a resposta de um ExtChoiceActionImpl
	 * @param rCell1
	 * @param rCell2
	 * @param extChoice
	 * @return rCell 
	 */
	public Term montarExtChoice(RCell rCell1, RCell rCell2, Term extChoice) {

		ExtChoiceActionImpl ext = (ExtChoiceActionImpl) extChoice;

		/**
		 * Criando os CircusAction
		 */
		CircusAction caLeft = (CircusAction) rCell1.getNoPrograma().getPrograma();
		CircusAction caRight = (CircusAction) rCell2.getNoPrograma().getPrograma();

		/**
		 * Setando os CircusAction
		 */

		ext.setLeftAction(caLeft);
		ext.setRightAction(caRight);

		return ext;
	}

	/**
	 * Método utilizado para reconstruir a resposta de um IntChoiceActionImpl
	 * @param rCell1
	 * @param rCell2
	 * @param intChoice
	 * @return rCell 
	 */
	public Term montarIntChoice(RCell rCell1, RCell rCell2, Term intChoice) {

		IntChoiceActionImpl intC = (IntChoiceActionImpl) intChoice;

		/**
		 * Criando os CircusAction
		 */
		CircusAction caLeft = (CircusAction) rCell1.getNoPrograma().getPrograma();
		CircusAction caRight = (CircusAction) rCell2.getNoPrograma().getPrograma();

		/**
		 * Setando os CircusAction
		 */

		intC.setLeftAction(caLeft);
		intC.setRightAction(caRight);

		return intC;
	}


	/**
	 * Método utilizado para reconstruir a resposta de um ParallelActionImpl
	 * @param rCell1
	 * @param rCell2
	 * @param intChoice
	 * @return rCell 
	 */
	public Term montarParallelAction(RCell rCell1, RCell rCell2, Term parralel) {

		ParallelActionImpl par = (ParallelActionImpl) parralel;


		/**
		 * Criando os CircusAction
		 */

		/*if (rCell1.getNoPrograma().getPrograma() instanceof CallAction){
			CallAction caLeft = (CallAction) rCell1.getNoPrograma().getPrograma();
			par.setLeftAction(caLeft);
		}
		else if (rCell1.getNoPrograma().getPrograma() instanceof ParallelActionImpl){
			ParallelActionImpl caLeft = (ParallelActionImpl) rCell1.getNoPrograma().getPrograma();
			par.setLeftAction(caLeft);
		}

		else if (rCell1.getNoPrograma().getPrograma() instanceof PrefixingActionImpl){
			PrefixingActionImpl caLeft = (PrefixingActionImpl) rCell1.getNoPrograma().getPrograma();
			par.setLeftAction(caLeft);
		}

		else if (rCell1.getNoPrograma().getPrograma() instanceof MuActionImpl){
			MuActionImpl caLeft = (MuActionImpl) rCell1.getNoPrograma().getPrograma();
			par.setLeftAction(caLeft);
		}

		else if (rCell1.getNoPrograma().getPrograma() instanceof VarDeclCommandImpl){
			VarDeclCommandImpl caLeft = (VarDeclCommandImpl) rCell1.getNoPrograma().getPrograma();
			par.setLeftAction(caLeft);
		}

		else if (rCell1.getNoPrograma().getPrograma() instanceof SeqActionImpl){
			SeqActionImpl caLeft = (SeqActionImpl) rCell1.getNoPrograma().getPrograma();
			par.setLeftAction(caLeft);
		}

		if (rCell2.getNoPrograma().getPrograma() instanceof CallAction){
			CallAction caRight = (CallAction) rCell2.getNoPrograma().getPrograma();
			par.setRightAction(caRight);
		}
		else if (rCell2.getNoPrograma().getPrograma() instanceof ParallelActionImpl){
			ParallelActionImpl caRight = (ParallelActionImpl) rCell2.getNoPrograma().getPrograma();
			par.setRightAction(caRight);
		}

		else if (rCell2.getNoPrograma().getPrograma() instanceof PrefixingActionImpl){
			PrefixingActionImpl caRight = (PrefixingActionImpl) rCell2.getNoPrograma().getPrograma();
			par.setRightAction(caRight);
		}

		else if (rCell2.getNoPrograma().getPrograma() instanceof MuActionImpl){
			MuActionImpl caRight = (MuActionImpl) rCell2.getNoPrograma().getPrograma();
			par.setRightAction(caRight);
		}

		else if (rCell2.getNoPrograma().getPrograma() instanceof VarDeclCommandImpl){
			VarDeclCommandImpl caRight = (VarDeclCommandImpl) rCell2.getNoPrograma().getPrograma();
			par.setRightAction(caRight);
		}

		else if (rCell2.getNoPrograma().getPrograma() instanceof SeqActionImpl){
			SeqActionImpl caRight = (SeqActionImpl) rCell2.getNoPrograma().getPrograma();
			par.setRightAction(caRight);
		}
*/
		
		par.setLeftAction((CircusAction) rCell1.getNoPrograma().getPrograma());
		par.setRightAction((CircusAction) rCell2.getNoPrograma().getPrograma());

		return par;
	}
	/**
	 * Método responsavel por retornar uma lei 
	 * @param lei Nome da lei que está na tática
	 * @return leiSelecionada A lei sendo uma CircusLaw
	 */
	public CircusLaw retornarLei(String lei)  {

		String semHifen = null;
		CircusLaw leiSelecionada = null;
		semHifen = leiSemHifem(lei);
		if (semHifen != null)
			lei = semHifen;
		List<CircusLaw> laws = getLaws();
		lei = lei.trim();

		for (CircusLaw law : laws) {
			String newLei = law.getName();
			if (newLei.equalsIgnoreCase(lei)) {
				leiSelecionada = law;
				break;
			}
			if (lei.startsWith("Par Interleave")
					&& newLei.startsWith("Par Interleave")) {
				leiSelecionada = law;
				break;
			}

		}
		/*if (leiSelecionada == null) {
			JOptionPane
			.showMessageDialog(null, gerInterface.getInternacional()
					.retornarMensagem("LawNotValidCRulesMsg")
					+ " : " + lei, gerInterface.getInternacional()
					.retornarMensagem("LawNotFound"),
					JOptionPane.ERROR_MESSAGE);

		}*/
		return leiSelecionada;

	}

	/**
	 * Método que retira o hifen do nome da lei
	 * @param lei Nome da lei 
	 * @return semHifen String sem o hifen
	 */
	private String leiSemHifem(String lei) {
		String semHifen = null;
		if (lei.contains("_") || lei.contains("-"))
			semHifen = lei.replace("_", " ");

		return semHifen;
	}


	/**
	 * Método chamado pela classe TLaw para aplicar uma Lei
	 * 
	 * @param termo O termo selecionado 
	 * @param leiSelecionada A lei que será aplicada
	 * @param mapeamento 
	 * @throws CRulesException
	 * @throws CancelledApplException
	 */

	public LawAnswer aplicarLeiTatica(NoPrograma noPrograma,
			CircusLaw leiSelecionada, List<Object> args, List<Pair<String, Term>> mapeamento) 
	throws CancelledApplException,
	CRulesException {


		LawAnswer result = this.getCrUtils().applyLawTactic(noPrograma.getPrograma(), 
				leiSelecionada, this.gerDevelopments.getDevelopment(
						gerInterface.retornarTelaDesenvolvimento().getSelectedDevelopment()).getProgAtual(),
						args,mapeamento);

		this.provarOps(result.getOps());

		/**
		 * Atualizar a AST
		 */

		if (!GerenciadorTaticas.controlTFails){
			if (result.getNovaAST() != null){
				Term ast2 = result.getNovaAST();
				gerTaticas.setLastAst(this.gerDevelopments.getDevelopment(
						gerInterface.retornarTelaDesenvolvimento().getSelectedDevelopment()).getProgAtual());
				this.atualizarProgramaTatica(ast2);
			}
		}


		//gerInterface.atualizarTelaCodigo();


		/**
		 * Setar a resposta da lei
		 */
		this.setRespostaLei(result);

		return result;

	}

	public LawAnswer respostaLei;

	public LawAnswer getRespostaLei() {
		return respostaLei;
	}

	public void setRespostaLei(LawAnswer res) {
		this.respostaLei = res;
	}

	/**
	 * Método responsavel por montar a resposta de tipo Prefixing
	 * @param comunication
	 * @param resposta
	 * @return
	 */
	public Vector montarRespostaPrefixing(Term comunication, Vector resposta) {

		Vector respostaPrefixing = new Vector();


		for (int i = 0; i < resposta.size(); i++) {

			RCell rCellResultado = (RCell)resposta.get(i);

			RCell rCellRespostaFinal = new RCell(rCellResultado.getNoPrograma(),
					rCellResultado.getOps());


			respostaPrefixing.add(rCellRespostaFinal);
		}

		return respostaPrefixing;

	}

	public TacticAnswer aplicarTatica(Tatica tatica, RCell rcell) 

	throws Exception, Unification{

		TacticAnswer resposta = null;
		
		/**
		 * Aplicando a tática
		 */

		resposta = tatica.aplicar(rcell, this);


		/**
		 * Colocar na pilha de Undo
		 */
		Term ast = gerInterface.getProgramaAtual();
		//gerInterface.atulizarPilhaTaticas(ast);

		if (ast!=getRespostaLei().getNovaAST()){
			gerInterface.atulizarPilhaTaticas(ast);
			ast = gerInterface.getProgramaAtual();
			this.atualizarProgramaTatica(ast);

		}

		/*if (getRespostaLei().getNovaAST() == null){

			ast = gerInterface.getProgramaAtual();
			this.atualizarProgramaTatica(ast);			
		}
		else{
			ast = getRespostaLei().getNovaAST();
			this.atualizarProgramaTatica(ast);
		}

		gerInterface.atualizarTelaCodigo();*/

		return resposta;
	}

	/**
	 * Método utilizado para atualizar apos a apicação de uma tática
	 * @throws CRefineException 
	 */
	public void atualizarAposAplicacaoTatica() throws CRefineException {
		gerInterface.tirarSelecao();
		gerInterface.esvaziarRedoPilhas();
		gerInterface.retornarTelaPrincipal().habilitarUndo();
		

		gerInterface.atualizarTelaCodigo();

		gerInterface.checkFinishedSubDevelopment(getRespostaLei().getNovaAST());

		GerenciadorTaticas ger = new GerenciadorTaticas();

	}

	public Term parseProg(String especificacao)  throws ParseException, IOException,
	UnmarshalException, CRulesException{

		/* Executa o parse no argumento */
		Source source = this.criarArquivo(especificacao);
		Term term = ParseUtils.parse(source, InternalManager.getManager());
		return term;

	}


	public Term parseTeste(String especificacao)  throws ParseException, IOException,
	UnmarshalException, CRulesException{

		/* Executa o parse no argumento */
		Source source = this.criarArquivo(especificacao);
		Term term = ParseArgumentUtils.parseGlobalParagraph(source, InternalManager.getManager());
		return term;

	}

	/**
	 * Metodo responsavel por realizar o parser no nome do beginEndBox 
	 * @param especificacao O nome do parametro
	 * @return term O term parseado
	 * @throws ParseException
	 * @throws IOException
	 * @throws UnmarshalException
	 * @throws CRulesException
	 */
	public Term parserBeginEndBox(String especificacao) throws ParseException, IOException, 
	UnmarshalException, CRulesException {

		/* Executa o parse no argumento */
		Term parsedArg = 
			ParseArgumentUtils.parseArguments(especificacao, CodigoParametro.NAME);

		return parsedArg;
	}

	/**
	 * Método utilizado para criar um arquivo para realizar o parser
	 * @param especificacao String com a especificação a ser inserida no arquivo
	 * @return source Que é parametro apra realizar o parser
	 * @throws IOException 
	 */
	private Source criarArquivo(String especificacao) throws IOException {

		String pathPrograma = "./tacticsArcAngelC/arquivoExtra.tex";
		File file = new File(pathPrograma);
		gerTaticas.escreverArquivoParser(file, especificacao);
		FileSource source = new FileSource(file);

		return source;
	}

	public List<Term> getBeginEndBox(NoPrograma noProgInicial) {

		BasicProcess bp = (BasicProcess) noProgInicial.getPrograma();

		List<Term> children = new ArrayList<Term>();

		children.add(bp.getMainAction());

		for (Para para :  bp.getZParaList()) {
			if (para instanceof AxPara) {

				AxPara axPara = (AxPara) para;
				ConstDecl temp = (ConstDecl) 
				axPara.getZSchText().getZDeclList().get(0);
				children.add(temp.getName());

			}
			else if (para instanceof ActionPara){
				ActionPara acPara = (ActionPara) para;

				if (!children.get(0).equals(acPara));
				children.add(acPara.getName());
			}
		}

		return children;
	}

	/**
	 * Método responsavel por verificar se o nome passado na tatica
	 * existe no processo
	 * @param children A lista com todos os filhos do processo
	 * @param paragraph O nome passado como parametro na tatica
	 * @return true Caso o nome tenha na lista de children
	 */
	public RCell verificarNomeProcesso(List<Term> children, String paragraph) {

		Term result = null;
		RCell rCell = null;

		for (Term term :  children) {

			if (term instanceof ZNameImpl) {

				ZName zName = (ZName) term;
				if(zName.getWord().equals(paragraph)){
					Term ast = gerInterface.getProgramaAtual();
					result = TermFromANameGetter.getTerm(zName, ast);
					rCell = this.criarRCell(result);
					break;
				}

			}



			else if (term instanceof ProcessPara){
				ProcessPara pp = (ProcessPara) term;
				String nomeP = pp.getName() + "";
				if (nomeP.equals(paragraph)){
					rCell = this.criarRCell(result);
				}

			}

		}

		return rCell;
	}


	/**
	 * Método responsavel por verificar se o nome passado na tatica
	 * existe no processo
	 * @param nome O nome passado como parametro na tatica
	 * @return true Caso o nome tenha na lista de children
	 */
	public RCell verificarRecurssaoAST(String nome) {

		Term result = null;
		RCell rCell = null;
		Term ast = gerInterface.getProgramaAtual();

		/**
		 * Criando uma instacia pra o Spec
		 */

		Spec spec = (Spec) ast;
		ListTerm<Sect> lista =  spec.getSect();	


		//ZName names = (ZName)lista.get(0);

		ZSect zSect = null;
		for ( Sect section : lista ) {
			if ( section instanceof ZSect ) {
				zSect = (ZSect) section;
				if (zSect.getName().equals(nome)){
					ZName zName = (ZName) zSect;
					result = TermFromANameGetter.getTerm(zName, ast);
					rCell = this.criarRCell(result);
					break;
				}
			}
		}

		for (Term term :  lista) {

			if (term instanceof ZNameImpl) {

				ZName zName = (ZName) term;
				if(zName.getWord().equals(nome)){

					result = TermFromANameGetter.getTerm(zName, ast);
					rCell = this.criarRCell(result);
					break;
				}

			}




		}

		return rCell;
	}


	/**
	 * Metodo responsavel por verificar os tipos dos paragrafos 
	 * da especificacao
	 * @param noProgInicial O programa inicial
	 * @return list de termos
	 */
	public List<Term> verificarSpecImp(NoPrograma noProgInicial) {

		List<Term> list = new ArrayList<Term>();

		Term ast = this.getProgramaAtual();
		Spec specification = (Spec) ast;
		ListTerm<Sect> lista = specification.getSect();		
		int z =0;

		for (int i = 0; i < lista.size(); i++) {

			if (lista.get(i) instanceof ZSectImpl){
				z = i;	
				break;
			}
		}
		ZSect secao = (ZSect)lista.get(z);
		ZParaList listaPara = secao.getZParaList();

		for (Para p : listaPara) {
			p.toString();
			if (p instanceof ProcessPara) {

				ProcessPara pp = (ProcessPara)p;
				list.add(pp);

			}
			else if (p instanceof ActionPara){
				ActionPara ap = (ActionPara)p;
				ap.getCircusAction();
				list.add(ap);
			}
			else if (p instanceof AxPara) {

				AxPara axPara = (AxPara) p;

				ConstDecl temp = (ConstDecl) 
				axPara.getZSchText().getZDeclList().get(0);
				temp.getName();
				temp.getExpr();
				list.add(axPara);


			}
		}
		return list;
	}



	/**
	 * Metodo responsavel por verificar os tipos dos paragrafos 
	 * da especificacao
	 * @param noProgInicial O programa inicial
	 * @return list de termos
	 */
	public List<Term> verificarSpecNewAst(Term ast, String processo) {

		List<Term> list = new ArrayList<Term>();

		Spec specification = (Spec) ast;
		ListTerm<Sect> lista = specification.getSect();		
		int z =0;

		for (int i = 0; i < lista.size(); i++) {

			if (lista.get(i) instanceof ZSectImpl){
				z = i;	
				break;
			}
		}
		ZSect secao = (ZSect)lista.get(z);
		ZParaList listaPara = secao.getZParaList();

		for (Para p : listaPara) {
			if (p instanceof ProcessPara) {
				ProcessPara pp = (ProcessPara)p;
				if (pp.getName().toString().equals(processo)){
					list.add(pp);
					break;
				}
			}
		}
		return list;
	}
	public List<Term> verificarSpecImpAcoes(Term noProgInicial,String processo) {

		List<Term> list = new ArrayList<Term>();
		List<Term> acoes = new ArrayList<Term>();

		List<Pair<String,Pair<String,Object>>> listArgs = gerTaticas.getListArgs();


		for (Pair<String,Pair<String,Object>> pair : listArgs) {
			Pair<String,Object> second = (Pair<String,Object>) pair.getSecond();
			if (second.getFirst().equals(processo)){
				processo = (String) second.getSecond();
				break;
			}

		}



		if (noProgInicial instanceof SpecImpl){

			Spec specification = (Spec) noProgInicial;
			ListTerm<Sect> lista = specification.getSect();		
			int z =0;

			for (int i = 0; i < lista.size(); i++) {

				if (lista.get(i) instanceof ZSectImpl){
					z = i;	
					break;
				}
			}
			ZSect secao = (ZSect)lista.get(z);
			ZParaList listaPara = secao.getZParaList();

			for (Para p : listaPara) {
				if (p instanceof ProcessPara){
					ProcessPara ap = (ProcessPara) p;
					if (processo.trim().equalsIgnoreCase(ap.getName()+"")){  	
						list.add(ap.getCircusBasicProcess());
						break;
					}

				}

			}
		}
		BasicProcess bp = (BasicProcess) list.get(0);
		for (Para para :  bp.getZParaList()) {
			if (para instanceof ActionPara){
				ActionPara acPara = (ActionPara) para;
				String nome = acPara.getZName()+"";
				if (nome.equals("Flows")){
					acoes.add(acPara.getCircusAction());
					break;
				}
			}

		}
		return acoes;
	}
	/**
	 * Metodo responsavel por retornar o paragrafo como um term do seu tipo
	 * @param paragraphs A lista de todos os paragrafos da especificacao
	 * @return term O termo
	 */
	public Term getParagraph(List<Term> paragraphs, String paragraph) {


		paragraph = this.verificarParagrafo(paragraph);

		for (Term t : paragraphs){

			if (t instanceof ProcessPara){
				ProcessPara pp = (ProcessPara) t;
				pp.getCircusProcess();
				if (paragraph.trim().equalsIgnoreCase(pp.getName() + ""))
					return pp;
			}
			else if(t instanceof ActionPara){
				ActionPara ac = (ActionPara) t;
				if (paragraph.equalsIgnoreCase(ac.getName() + ""))
					return ac;
			}

		}
		return null;
	}

	private String verificarParagrafo(String paragraph) {

		String res = "";
		List<Pair<String,Pair<String,Object>>>  listArgs = gerTaticas.getListArgs();

		for (Pair<String,Pair<String,Object>> pair : listArgs) {
			Pair<String,Object> second = (Pair<String,Object>) pair.getSecond();
			if (second.getFirst().equals(paragraph)){
				res = (String) second.getSecond();
				break;
			}
		}
		return res;
	}

	private String verificarHead(String list,List<Pair<String, Pair<String, Object>>> listArgs) {


		ArrayList<String> lista = new ArrayList<String>();

		for (Pair<String,Pair<String,Object>> pair : listArgs) {
			Pair<String,Object> second = (Pair<String,Object>) pair.getSecond();
			if (second.getFirst().trim().equals(list)){
				lista = (ArrayList<String>) second.getSecond();
				break;

			}
		}
		return lista.get(0);
	}

	/**
	 * Metodo que cria uma composicao paralela
	 * @param lista
	 * @return
	 */
	public Term getParallel(List<String> lista) {
		// TODO Auto-generated method stub
		return null;
	}

	/**
	 * Método responsavel por verificar se o nome passado na tatica
	 * existe no processo
	 * @param children A lista com todos os filhos do processo
	 * @param paragraph O nome passado como parametro na tatica
	 * @return true Caso o nome tenha na lista de children
	 */
	public boolean verificarNomeParagrafo(List<Term> paragraphs, String nome) {


		/*
		 * Verificar na lista de mapeamento
		 */

		List<Pair<String,Pair<String,Object>>> listArgs = gerTaticas.getListArgs();
		boolean res = true;


		for (Pair<String,Pair<String,Object>> pair : listArgs) {
			Pair<String,Object> second = (Pair<String,Object>) pair.getSecond();
			if (second.getFirst().equals(nome)){
				res = true;
				break;
			}

		}


		/*		for (Term term :  paragraphs) {

			if (term instanceof ProcessPara){
				ProcessPara pp = (ProcessPara) term;
				String nomeP = pp.getName() + "";
				if (nomeP.equalsIgnoreCase(nome)){
					res = true;
					break;
				}
			}

		}*/

		return res;
	}


	public boolean verificarProg(String prog, Term termInicial) throws IOException, ParseException, UnmarshalException, CRulesException {

		String[] resTemp = null;

		Printer printer = new Printer();

		Source source = this.criarArquivo(prog);

		Term t = this.parseProg(prog);

		resTemp = printer.visitTermTactic(termInicial);

		String stringCircus = gerInterface.TraduzirToLatex(prog);

		String newResTemp = this.formatarString(resTemp);
		/**
		 * tem 	que tirar o / antes ;
		 * 
		 */

		for (String string : resTemp) {
			if (prog.equals(stringCircus))
				return true;
		}

		return false;

	}

	private String formatarString(String[] resTemp) {

		for (String string : resTemp) {
			if (string.contains("\\"))
				string.replace("\\", "");
		}
		return null;
	}

	/**
	 * Método que chama a tela de erro
	 * @param erro O erro que ocorreu no parser
	 * @param detalhe O detalhe do erro
	 */
	public void telaErro(String erro, String detalhe, String info) {
		gerInterface.telaErro(erro, detalhe, info);

	}


	public int getLinha(){
		return gerParser.setarLinha();
	}


	/**
	 * Metodo para parsear um processo 
	 * @param prog
	 * @return newTerm
	 * @throws CRulesException
	 * @throws Unification 
	 */
	private Term parseProcess(String prog) throws CRulesException, Unification {

		Term newTerm = ParseArgumentUtils.parseArguments(prog,CodigoParametro.PROCESS);
		return newTerm;

	}

	/**
	 * Metodo para parsear um processo para ProcessPara
	 * @param prog
	 * @return newTerm
	 * @throws CRulesException
	 * @throws Unification 
	 */
	private Term parseProcessPara(String prog) throws CRulesException, Unification {

		Term newTerm = ParseArgumentUtils.parseArguments(prog,CodigoParametro.PROCESSPARA);
		return newTerm;

	}

	/**
	 * Metodo para parsear uma acao
	 * @param prog
	 * @return
	 * @throws CRulesException
	 * @throws Unification 
	 */
	private Term parseAction(String prog) throws CRulesException, Unification {

		Term newTerm = ParseArgumentUtils.parseArguments(prog,CodigoParametro.ACTION);
		return newTerm;
	}


	private String getCommunication(String prog) {

		String res = prog.trim();
		String[] inicio = res.split(" ");
		return inicio[0];

	}

	private List<Pair<String,Term>> criarJokersAcao(Term procInicial,Term newTerm) throws Unification {

		Set<Binding> unificacao = null;
		List<Pair<String,Term>> list = new ArrayList<Pair<String,Term>>();


		try {

			unificacao = UnifierVisitor.unify(procInicial,newTerm);
			list = visitTerm(procInicial, newTerm);

		} catch (UnificationException e) {
			throw new Unification("Termo selecionado: " + procInicial.toString()+ "Term Tática: " + newTerm.toString());
		}

		return list;
	}

	public void setMapeamento(List<Pair<String, Term>> mapeamento) {
		gerTaticas.setMapeamento(mapeamento);

	}

	public List<Pair<String, Term>> getMapeamento() {

		return gerTaticas.getMapeamento();
	}


	/**
	 * Método que verifica se já existe a lei e/ou a tatica no sistema
	 * @param nome O nome da lei
	 */
	public boolean ifExistsLaw(String nome) {
		if (retornarLei(nome) == null)
			return false;
		else
			return true;

	}

	/**
	 * Método que verifica se já existe a tatica no sistema
	 * @param nome O nome da tatica
	 */
	public boolean ifExistsTactic(String nome) {

		List<Tatica> taticas = gerInterface.retornarListaTaticas();
		if (taticas.size() == 0)
			return false;
		for (Tatica tatica : taticas) {
			if (tatica.getId().equalsIgnoreCase(nome))
				return true;
		}
		return false;

	}

	/**
	 * Método que verifica se a lei é uma lei de parametro
	 * @param argumentos Lista com os argumentos
	 * @param nomeLei O nome da lei
	 */
	public boolean ifExistsArgs(ArrayList<Object> argumentos, String nome) {
		for (Object args : argumentos) {
			if (args.equals(nome))
				return true;
		}
		return false;
	}

	public List<Term> FNames(String image) {

		Term inicial = gerInterface.getProgramaAtual();
		List<Term> par = null;

		if (inicial instanceof SpecImpl){
			par = this.verificarSpecImpAcoes(inicial,image);
		}

		return par;
	}

	public ArrayList<Pair<String,Pair<String,Object>>>  verificarArgumentos(List<Object> argumentosPassados,
			Tatica tatica) {

		List<Term> acoes = new ArrayList<Term>();
		int i = 0;
		boolean exist = false;
		List<Object> passados = argumentosPassados;
		ArrayList<Object> argumentosTatica = tatica.getArgumentos();
		ArrayList<Pair<String,Pair<String,Object>>> retorno = new ArrayList<Pair<String,Pair<String,Object>>>();
		List<Pair<String,Pair<String,Object>>> listArgs = gerTaticas.getListArgs();
		for (Object object : argumentosPassados) {
			Pair<String,Pair<String,Object>> pair = 
				new Pair<String,Pair<String,Object>>();
			pair.setFirst(tatica.getId());

			Pair<String,Object> second = new Pair<String, Object>();
			if (object instanceof ArrayList){
				ArrayList lista = (ArrayList) object;
				if(this.getTiposTatica().contains(lista.get(0)+"")){

					String teste = tatica.getArgumentos().get(0).toString() + tatica.getId();
					if (argumentosTatica.get(i).equals(teste)){
						second.setFirst(argumentosTatica.get(i)+"");
					}
					else if (argumentosTatica.get(i).toString().contains(tatica.getId()))
						second.setFirst(argumentosTatica.get(i)+"");
					else if(argumentosTatica.get(i).equals(tatica.getArgumentos().get(0).toString())){	
						second.setFirst(argumentosTatica.get(i)+""+tatica.getId());
						ArrayList<Object> newArgumentos = new ArrayList<Object>();
						String newArg = tatica.getArgumentos().get(0).toString(); 
						newArgumentos.add(newArg+tatica.getId());
					}
					else 
						second.setFirst(argumentosTatica.get(i)+"");
					if(listArgs.size() == 0){

						second.setSecond(lista.get(1));
					}
					if (second.getSecond() == null){
						second.setSecond(lista.get(1));
					}
					boolean exists = false;

					for (Pair<String, Pair<String, Object>> pair2 : listArgs) {

						if(pair.getFirst().equals(pair2.getFirst())){

							Pair<String, Object> second2 = (Pair<String, Object>) pair2.getSecond();
							if (second.getFirst().equals(second2.getFirst())){
								if (pair2.getFirst().equals(tatica.getId())){
									second2.setSecond(second.getSecond());
									exists = true;
									break;
								}
							}
						}
					}
					if (!exists || listArgs.size() == 0){
						pair.setSecond(second);
						retorno.add(pair);				
					}

				}
				else {
					second.setFirst(argumentosTatica.get(i)+"");
					second.setSecond(argumentosPassados.get(i));

					pair.setSecond(second);
					retorno.add(pair);
				}
			}else if (object instanceof FNames){
				FNames f = (FNames) object;
				acoes = this.FNames(f.getFname());
				second.setFirst(argumentosTatica.get(i)+"");
				second.setSecond(acoes);
				for (Pair<String,Pair<String, Object>> pair2 : listArgs) {
					Pair<String, Object> al = (Pair<String, Object>) pair2.getSecond();
					if (argumentosTatica.get(i).equals(al.getFirst())){
						al.setSecond(acoes);
						exist=true;
						break;
					}
				}
				if(!exist){
					pair.setSecond(second);
					retorno.add(pair);	
				}
			}
			else if (object instanceof Head){
				Head head = (Head)object;
				second.setFirst(argumentosTatica.get(i)+"");
				ArrayList list = new ArrayList();
				for (Pair<String,Pair<String, Object>> pair2 : listArgs) {
					Pair<String, Object> al = (Pair<String, Object>) pair2.getSecond();
					if (al.getFirst().trim().equals(head.getList())){
						list = (ArrayList) al.getSecond();
						break;
					}
				}
				for (Pair<String,Pair<String, Object>> pair2 : listArgs) {
					Pair<String, Object> al = (Pair<String, Object>) pair2.getSecond();
					if(tatica.getId().equals(pair2.getFirst())){
						if (list == null){
							al.setSecond(null);
							return null;
						}
						else
							al.setSecond(list.get(0));
						exist = true;
						break;
					}
				}if (!exist){
					second.setSecond(list.get(0));
					pair.setSecond(second);
					retorno.add(pair);	
				}

			}
			else if (object instanceof Tail){
				Tail tail = (Tail) object;
				second.setFirst(argumentosTatica.get(i)+"");

				for (Pair<String,Pair<String, Object>> pair2 : listArgs) {
					Pair<String, Object> al = (Pair<String, Object>) pair2.getSecond();
					if (al.getFirst().trim().equals(tail.getList())
							&& pair2.getFirst().equals(tatica.getId())){
						ArrayList list = (ArrayList) al.getSecond();
						if(tatica.getId().equals(pair2.getFirst())){
							if (list.size() == 1){
								al.setSecond(null);
								break;
							}
							else{
								list.remove(0);
								al.setSecond(list);
								break;
							}					   
						}
						else
							second.setSecond(list.get(list.size()-1));
						pair.setSecond(second);
						retorno.add(pair);	
						break;
					}
				}
			}
			else {
				second.setFirst(argumentosTatica.get(i)+"");
				second.setSecond(argumentosPassados.get(i));
				Object aux = "";
				if (!second.getFirst().equals(second.getSecond()+"")){
					for (Pair<String, Pair<String, Object>> pair2 : listArgs) {
						Pair<String, Object> second2 = (Pair<String, Object>) pair2.getSecond();
						String sec = second.getSecond()+"";
						if (sec.trim().equals(second2.getFirst().trim())){
							aux = second2.getSecond();
							break;
						}
					}
					for (Pair<String, Pair<String, Object>> pair2 : listArgs) {
						Pair<String, Object> second2 = (Pair<String, Object>) pair2.getSecond();
						if (tatica.getId().equals(pair2.getFirst())){
							if(second.getFirst().trim().equals(second2.getFirst())){
								if(!aux.equals(""))
									second2.setSecond(aux);
								exist=true;
								break;
							}
						}

					}


					if(!exist){
						if (!aux.equals(""))
							second.setSecond(aux);
						pair.setSecond(second);
						retorno.add(pair);
					}
				}
			}
			i++;
		}




		return retorno;
	}

	public void setListArgs(ArrayList<Pair<String,Pair<String,Object>>> lista) {
		gerTaticas.setListArgs(lista);

	}

	/**
	 * Método que cria um ZName 
	 * @param object List
	 * @return term ZName
	 */
	public ZName getZName(Object nome) {
		ZName name = factory.createZName(nome.toString(), 
				factory.createZStrokeList(), null);
		return name;
	}

	public Vector<String> getTiposTatica() {

		return tiposTatica;
	}

	public BasicProcess getBasicProcess(Term programa) {
		BasicProcess bp = this.factory.createBasicProcess();

		return null;
	}

	public Term getNewProcess(RCell rCell) {

		Term ast = gerInterface.getProgramaAtual();
		NoPrograma noProgInicial = new NoPrograma(ast);
		ProcessPara pp = (ProcessPara) GerenciadorTaticas.lastProcess;
		List<Term> paragraphs = this.verificarSpecNewAst(ast, pp.getName().toString());
		return paragraphs.get(0);
	}

	public Term leiVarExp(Term programa) {
		/**
		 * Montar o novo Paralelismo
		 */
		Term leftAction = null;
		Term rightAction = null;
		Vector<Term> vec = new Vector<Term>();
		ParallelActionImpl newPar = null;

		if (programa instanceof ParallelActionImpl){
			ParallelActionImpl par = (ParallelActionImpl) programa;
			/**
			 * Verificando se as duas ações são um VarDecl
			 */
			if(par.getLeftAction() instanceof VarDeclCommandImpl)
				leftAction = par.getLeftAction();
			else leftAction = this.getActionVarExp(par.getLeftAction());


			if(par.getRightAction() instanceof VarDeclCommandImpl)
				rightAction = par.getRightAction();
			else rightAction = this.getActionVarExp(par.getRightAction());

			/**
			 * Montar paralelismo 
			 */
			newPar = (ParallelActionImpl) par;
			CircusAction left = (CircusAction) leftAction;
			CircusAction right = (CircusAction) rightAction;
			newPar.setLeftAction(left);
			newPar.setRightAction(right);
		}
		return newPar;
	}

	private Term getActionVarExp(Term action) {

		Term newAction = null;
		Vector<RCell> vec = this.retornarListaAcoes(action);
		for (int i = 0; i < vec.size(); i++) {
			RCell term = (RCell) vec.get(i);
			if (term.getNoPrograma().getPrograma()
					instanceof VarDeclCommandImpl){
				newAction = term.getNoPrograma().getPrograma();
				break;      
			}
			else newAction = term.getNoPrograma().getPrograma();
		}
		return newAction;
	}

	public RCell montarRespostaVar(DeclList decl, RCell rCellResposta, Term var) {


		CircusAction ca = null;
		VarDeclCommand varFinal = (VarDeclCommand) var;
		varFinal.setDeclList(decl);
		ca = (CircusAction) rCellResposta.getNoPrograma().getPrograma();
		varFinal.setCircusAction(ca);

		NoPrograma programa = new NoPrograma(varFinal);
		RCell rcellFinal = new RCell(programa);

		return rcellFinal;
	}

	public LawAnswer leiJoinBlocks(Term programa) {
		this.var = programa;
		VarDeclCommand var = (VarDeclCommand) programa;
		CircusAction action = var.getCircusAction();
		VarDeclCommand var1 = (VarDeclCommand) action;
		CircusAction action2 = var1.getCircusAction();

		ZDeclList lista = factory.createZDeclList();

		ZDeclList listZD = var.getZDeclList();
		for (Decl decl : listZD) {
			lista.add(decl);
		}

		ZDeclList listZD2 = var1.getZDeclList();
		for (Decl decl : listZD2) {
			lista.add(decl);
		}


		VarDeclCommand v = factory.createVarDeclCommand();

		VarDeclCommand varFinal = (VarDeclCommand) v;
		varFinal.setCircusAction(action2);
		varFinal.setDeclList(lista);
	
		Term ast = gerInterface.getProgramaAtual();
		Term inicial = this.var;
		Term novaAST = UpdateVisitor.update(inicial, varFinal, ast,true);


		Spec spec = (Spec) ast;
		ListTerm<Sect> lista2 =  spec.getSect();	


		ZSect secao = (ZSect)lista2.get(0);
		ZParaList listaPara = secao.getZParaList();
		for (Para p : listaPara) {

			// TODO melhorar essa parte do programa
			if ( p instanceof ProcessParaImpl){
				ProcessPara pp = (ProcessPara) p;
				BasicProcess bp = (BasicProcess) pp.getCircusBasicProcess();
				listaPara = bp.getZParaList();
				for (Para para : listaPara) {
					if(para instanceof ActionPara){
						ActionPara ap = (ActionPara) para;
						if(ap.getCircusAction() instanceof SeqActionImpl){
							SeqActionImpl seq = (SeqActionImpl) ap.getCircusAction();
							CircusAction right = (CircusAction) seq.getLeftAction();
							if (seq.getRightAction() instanceof MuActionImpl){
								MuActionImpl mu = (MuActionImpl) seq.getRightAction();
								SeqActionImpl seq2 = (SeqActionImpl) mu.getCircusAction();
								if (seq2.getLeftAction() instanceof VarDeclCommandImpl){
									VarDeclCommand var2 = (VarDeclCommand) seq2.getLeftAction();
									var2.setCircusAction(action2);
									var2.setDeclList(lista);
								}
							}
							break;
						}

					}

				}
			}
		}

		LawAnswer result = new LawAnswer();
		result.setNovaAST(spec);
		result.setProgResultante(varFinal);

		if (result.getNovaAST() != null){
			Term ast2 = result.getNovaAST();
			gerTaticas.setLastAst(this.gerDevelopments.getDevelopment(
					gerInterface.retornarTelaDesenvolvimento().getSelectedDevelopment()).getProgAtual());
			this.atualizarProgramaTatica(ast2);
		}




		return result;
	}

	public Term getProgramaAtual(){
		return gerInterface.getProgramaAtual();
	}

	public LawAnswer leiMainVarState(Term programa) {

		LawAnswer result = new LawAnswer();
		Term ast = gerInterface.getProgramaAtual();
		CircusAction mainAction = null;
		ZDeclList listaZDecl = factory.createZDeclList();
		AxPara newAxPara = factory.createAxPara();


		BasicProcess process = factory.createBasicProcess();

		ZDeclList listaFinalSchema = factory.createZDeclList();
		String nameMain = "";
		ZName nomeEstado = null;
		ZDeclList newDeclaracoes = factory.createZDeclList();
		ZSchText newSch = factory.createZSchText();
		ZParaList newParaList = factory.createZParaList();
		String nomeState;
		SchExpr estado = factory.createSchExpr();
		ZSect secao = factory.createZSect();
		String lastProcess = GerenciadorTaticas.process;
		

		Spec spec = (Spec) ast;
		ListTerm<Sect> listaSpec =  spec.getSect();	

		for (Sect sect : listaSpec) {
			if(sect instanceof ZSect){
				secao = (ZSect)sect;
				break;
			}
		}

		ZParaList listaPara = secao.getZParaList();
		for (Para p : listaPara) {
			if ( p instanceof ProcessParaImpl){
				ProcessPara pp = (ProcessPara) p;
				BasicProcess bp = (BasicProcess) pp.getCircusBasicProcess();
				if(lastProcess.equals(pp.getZName()+"")){
				AxPara state = (AxPara) bp.getStatePara();
				nomeState = bp.getStateParaName()+"";
				if (nomeState.contains("default")){
					return result;
				}
				ZParaList listaParaProc = bp.getZParaList();
				for (Para para : listaParaProc) {
					if(para instanceof ActionPara){
						ActionPara ap = (ActionPara) para;
						String main = ap.getName()+"";
						if(main.contains("mainAction")){
							if (ap.getCircusAction() instanceof VarDeclCommandImpl){
								VarDeclCommand var = (VarDeclCommand) ap.getCircusAction();
								ZDeclList listZD = var.getZDeclList();
								mainAction = var.getCircusAction();
								SeqActionImpl seqNovo = (SeqActionImpl) var.getCircusAction();
								OnTheFlyDefAnn otfda = factory.createOnTheFlyDefAnn();
								seqNovo.getAnns().add(otfda);
								ap.setCircusAction(seqNovo);
								for (Decl decl : listZD) {
									listaZDecl.add(decl);
								}
								break;
							}
						}
					}
				}
				break;
			}
			}
		}


		for (Para p : listaPara) {
			if ( p instanceof ProcessParaImpl){
				ProcessPara pp = (ProcessPara) p;
				BasicProcess bp = (BasicProcess) pp.getCircusBasicProcess();
				process = bp;
				if(lastProcess.equals(pp.getZName()+"")){
				AxPara state = (AxPara) bp.getStatePara();

				ZNameList name = state.getName();
				ZSchText text = state.getZSchText();
				ZDeclList decl = text.getZDeclList();
				for (Decl decl2 : decl) {
					if (decl2 instanceof ConstDeclImpl){
						ConstDeclImpl con = (ConstDeclImpl) decl2;

						//* Nome do Schema

						nomeState = con.getZName()+"";
						if (nomeState.contains("default")){
							result.setProgResultante(null);
							return result;
						}
						nomeEstado = factory.createZName(nomeState, factory.createZStrokeList(), null);
						estado = SchemaFinder.getSchema(nomeState, crUtils.getInterno().retornarProgAtual());
						ZDeclList declaracoes = estado.getZSchText().getZDeclList();
						declaracoes.getDecl();
						for (Decl decl3 : declaracoes.getDecl()) {
							if (decl3 instanceof VarDeclImpl){
								VarDeclImpl var3 = (VarDeclImpl) decl3;
								listaFinalSchema.add(decl3);

							}
						}

						listaZDecl.addAll(listaFinalSchema);
						estado.getZSchText().setDeclList(listaZDecl);
						newDeclaracoes = estado.getZSchText().getZDeclList();
						newSch.setDeclList(newDeclaracoes);
						newAxPara = factory.createAxPara(name, newSch, state.getBox());
						newParaList.add(newAxPara);
					}

				}
				newAxPara = state;
				break;

			}
		}
		}


		Term novaAST = UpdateVisitor.update(spec, ast, ast,true);

		result.setNovaAST(spec);
		result.setProgResultante(process);
		return result;

	}


	public LawAnswer leiMainVarState2(Term programa) {

		LawAnswer result = new LawAnswer();
		ZDeclList listaZDecl = factory.createZDeclList();
		ZDeclList listaFinalSchema = factory.createZDeclList();
		CircusAction mainAction = null;
		String nameMain = "";
		ZName nomeEstado = null;
		ZDeclList newDeclaracoes = factory.createZDeclList();
		ZSchText newSch = factory.createZSchText();
		ZParaList newParaList = factory.createZParaList();
		AxPara newAxPara = factory.createAxPara();
		int count = 0;
		ZSect secao = factory.createZSect();
		BasicProcess bp = factory.createBasicProcess();
		String lastProcess = GerenciadorTaticas.process;
		BasicProcess process = factory.createBasicProcess();
		

		Term ast = gerInterface.getProgramaAtual();

		/**
		 * Recebendo a especificacao
		 */
		Spec spec = (Spec) ast;
		ListTerm<Sect> listaSpec =  spec.getSect();	

		for (Sect sect : listaSpec) {
			if(sect instanceof ZSect){
				secao = (ZSect)sect;
				break;
			}
		}


		ZParaList listaPara = secao.getZParaList();
		for (Para para : listaPara) {
			if ( para instanceof ProcessPara){
				ProcessPara pp = (ProcessPara) para;
				bp = (BasicProcess) pp.getCircusBasicProcess();
				if(lastProcess.equals(pp.getZName()+"")){
				AxPara state = (AxPara) bp.getStatePara();
				String nomeState = bp.getStateParaName()+"";
				if (nomeState.contains("default")){

					ZParaList listaParaProc = bp.getZParaList();
					for (Para para2 : listaParaProc) {
						if(para2 instanceof ActionPara){
							ActionPara ap = (ActionPara) para2;
							String main = ap.getName()+"";
							if(main.contains("mainAction")){
								
								if (ap.getCircusAction() instanceof VarDeclCommandImpl){
									VarDeclCommand var = (VarDeclCommand) ap.getCircusAction();
									ZDeclList listZD = var.getZDeclList();
									mainAction = var.getCircusAction();
									CircusAction seqNovo = (CircusAction) var.getCircusAction();
									OnTheFlyDefAnn otfda = factory.createOnTheFlyDefAnn();
									seqNovo.getAnns().add(otfda);
									ap.setCircusAction(seqNovo);
									for (Decl decl : listZD) {
										listaZDecl.add(decl);
									}
									break;
								}
								
							}
							}
						}
					}
				break;
				}
			}
		}

		ProcessPara pp = factory.createProcessPara();
		for (Para para : listaPara) {
			if ( para instanceof ProcessParaImpl){
				pp = (ProcessPara) para;
				bp = (BasicProcess) pp.getCircusBasicProcess();
				process = (BasicProcess) bp;
				if(lastProcess.equals(pp.getZName()+"")){
				ZParaList lista2 = bp.getZParaList();
				AxPara state = (AxPara) bp.getStatePara();
				String nomeState = bp.getStateParaName()+"";
				if (nomeState.contains("default")){
					newSch.setDeclList(listaZDecl);
					state.setSchText(newSch);
					break;
				}
				break;
			}
			}
		}

		Term novaAST = UpdateVisitor.update(spec, ast, ast,true);

		result.setNovaAST(spec);
		result.setProgResultante(process);

		return result;
	}

	/**
	 * Método responsavel por atualizar a lista de Táticas
	 * @param tatica A Tática a ser atualizada
	 */
	public void updateTacticList(Tatica tatica) {
		gerInterface.updateTacticList(tatica);
	}

	public void setDecl(DeclList decl) {
		this.decl = decl;

	}
	DeclList decl;
	PrefixingActionImpl pref;

	public PrefixingActionImpl getPref() {
		return pref;
	}

	public void setPref(PrefixingActionImpl pref) {
		this.pref = pref;
	}

	public DeclList getDecl() {
		return decl;

	}

	public Term varrerSpec(Term applicable, Term novo, Term ast) {

		Spec spec = (Spec) ast;

		ListTerm<Sect> lista =  spec.getSect();	



		ZSect secao = (ZSect)lista.get(0);
		ZParaList listaPara = secao.getZParaList();
		for (Para p : listaPara) {
			if ( p instanceof ProcessParaImpl){
				ProcessPara pp = (ProcessPara) p;
				BasicProcess bp = (BasicProcess) pp.getCircusBasicProcess();
				listaPara = bp.getZParaList();
				for (Para para : listaPara) {
					if(para instanceof ActionPara){
						ActionPara ap = (ActionPara) para;
						if(ap.getCircusAction() instanceof SeqActionImpl){
							SeqActionImpl seq = (SeqActionImpl) ap.getCircusAction();
							if (ap.getCircusAction().equals(applicable)){
								VarDeclCommand varNovo = (VarDeclCommand) novo;
								OnTheFlyDefAnn otfda = factory.createOnTheFlyDefAnn();
								varNovo.getAnns().add(otfda);
								ap.setCircusAction(varNovo);

								break;
							}
							if (seq.getRightAction() instanceof MuActionImpl){
								MuActionImpl mu = (MuActionImpl) seq.getRightAction();
								SeqActionImpl seq2 = (SeqActionImpl) mu.getCircusAction();
								if (seq2.equals(applicable)){
									VarDeclCommand varNovo = (VarDeclCommand) novo;
									mu.setCircusAction(varNovo);
									break;
								}
								if (seq2.getLeftAction() instanceof VarDeclCommandImpl){
									VarDeclCommand var = (VarDeclCommand) seq2.getLeftAction();
									VarDeclCommandImpl res = (VarDeclCommandImpl) novo;
									var.setCircusAction(res);
									break;
								}

							}
							break;
						}

					}

				}
			}
		}

		return spec;
	}

	public Term ParSeqStep(Term applicable, Term novo, Term ast) {
		Spec spec = (Spec) ast;
		ListTerm<Sect> lista =  spec.getSect();	


		ZSect secao = (ZSect)lista.get(0);
		ZParaList listaPara = secao.getZParaList();
		for (Para p : listaPara) {

			// TODO melhorar essa parte do programa
			if ( p instanceof ProcessParaImpl){
				ProcessPara pp = (ProcessPara) p;
				BasicProcess bp = (BasicProcess) pp.getCircusBasicProcess();
				listaPara = bp.getZParaList();
				for (Para para : listaPara) {
					if(para instanceof ActionPara){
						ActionPara ap = (ActionPara) para;
						if(ap.getCircusAction() instanceof SeqActionImpl){
							SeqActionImpl seq = (SeqActionImpl) ap.getCircusAction();
							if (seq.getRightAction() instanceof MuActionImpl){
								MuActionImpl mu = (MuActionImpl) seq.getRightAction();
								SeqActionImpl seq2 = (SeqActionImpl) mu.getCircusAction();
								if (seq2.getLeftAction() instanceof VarDeclCommandImpl){
									VarDeclCommand var = (VarDeclCommand) seq2.getLeftAction();
									SeqActionImpl res = (SeqActionImpl) novo;
									if (var.getCircusAction() instanceof SeqActionImpl){
										SeqActionImpl seq3 = (SeqActionImpl) var.getCircusAction();
										seq3.setRightAction(res);

									}
								}
							}
							break;
						}

					}

				}
			}
		}

		return spec;
	}

	public Term ParComm(Term applicable, Term novo, Term ast) {

		ParallelActionImpl parNovo = (ParallelActionImpl) novo;
		Spec spec = (Spec) ast;
		ListTerm<Sect> lista =  spec.getSect();	


		ZSect secao = (ZSect)lista.get(0);
		ZParaList listaPara = secao.getZParaList();
		for (Para p : listaPara) {

			// TODO melhorar essa parte do programa
			if ( p instanceof ProcessParaImpl){
				ProcessPara pp = (ProcessPara) p;
				BasicProcess bp = (BasicProcess) pp.getCircusBasicProcess();
				listaPara = bp.getZParaList();
				for (Para para : listaPara) {
					if(para instanceof ActionPara){
						ActionPara ap = (ActionPara) para;
						if(ap.getCircusAction() instanceof SeqActionImpl){
							SeqActionImpl seq = (SeqActionImpl) ap.getCircusAction();
							if (seq.getRightAction() instanceof MuActionImpl){
								MuActionImpl mu = (MuActionImpl) seq.getRightAction();
								SeqActionImpl seq2 = (SeqActionImpl) mu.getCircusAction();
								if (seq2.getLeftAction() instanceof VarDeclCommandImpl){
									VarDeclCommand var = (VarDeclCommand) seq2.getLeftAction();
									if (var.getCircusAction() instanceof SeqActionImpl){
										SeqActionImpl seq3 = (SeqActionImpl) var.getCircusAction();
										if (seq3.getRightAction() instanceof ParallelActionImpl){
											ParallelActionImpl par = (ParallelActionImpl) seq3.getRightAction();
											par.setLeftAction(parNovo.getRightAction());
											par.setRightAction(parNovo.getLeftAction());
											par.setLeftNameSet(parNovo.getRightNameSet());
											par.setRightNameSet(parNovo.getLeftNameSet());
										}

									}
								}
							}
							break;
						}

					}

				}
			}
		}

		return spec;
	}

	public Term SeqAssoc(Term applicable, Term novo, Term ast) {

		SeqActionImpl seqNovo = (SeqActionImpl) novo;
		Spec spec = (Spec) ast;
		ListTerm<Sect> lista =  spec.getSect();	


		ZSect secao = (ZSect)lista.get(0);
		ZParaList listaPara = secao.getZParaList();
		for (Para p : listaPara) {

			// TODO melhorar essa parte do programa
			if ( p instanceof ProcessParaImpl){
				ProcessPara pp = (ProcessPara) p;
				BasicProcess bp = (BasicProcess) pp.getCircusBasicProcess();
				listaPara = bp.getZParaList();
				for (Para para : listaPara) {
					if(para instanceof ActionPara){
						ActionPara ap = (ActionPara) para;
						if(ap.getCircusAction() instanceof SeqActionImpl){
							SeqActionImpl seq = (SeqActionImpl) ap.getCircusAction();
							if (seq.getRightAction() instanceof MuActionImpl){
								MuActionImpl mu = (MuActionImpl) seq.getRightAction();
								SeqActionImpl seq2 = (SeqActionImpl) mu.getCircusAction();
								if (seq2.getLeftAction() instanceof VarDeclCommandImpl){
									VarDeclCommand var = (VarDeclCommand) seq2.getLeftAction();
									if (var.getCircusAction() instanceof SeqActionImpl){
										SeqActionImpl seq3 = (SeqActionImpl) var.getCircusAction();
										if (seq3.equals(applicable)){
											seq3.setLeftAction(seqNovo.getLeftAction());
											seq3.setRightAction(seqNovo.getRightAction());
											seq3.getRightAction();
											seq3.getLeftAction();
											break;
										}						                            
										if (seq3.getRightAction() instanceof SeqActionImpl){
											SeqActionImpl seq4 = (SeqActionImpl) seq3.getRightAction();
											if (seq4.equals(applicable)){
												seq4.setLeftAction(seqNovo.getLeftAction());
												seq4.setRightAction(seqNovo.getRightAction());
												seq4.getRightAction();
												seq4.getLeftAction();
												break;
											}
										}

									}
								}
								break;
							}

						}

					}

				}
			}
		}

		return spec;
	}

	public Term SeqUnit2(Term applicable, Term novo, Term ast) {
		SeqActionImpl seqNovo = (SeqActionImpl) novo;
		Spec spec = (Spec) ast;
		ListTerm<Sect> lista =  spec.getSect();	


		ZSect secao = (ZSect)lista.get(0);
		ZParaList listaPara = secao.getZParaList();
		for (Para p : listaPara) {

			// TODO melhorar essa parte do programa
			if ( p instanceof ProcessParaImpl){
				ProcessPara pp = (ProcessPara) p;
				BasicProcess bp = (BasicProcess) pp.getCircusBasicProcess();
				listaPara = bp.getZParaList();
				for (Para para : listaPara) {
					if(para instanceof ActionPara){
						ActionPara ap = (ActionPara) para;
						if(ap.getCircusAction() instanceof SeqActionImpl){
							SeqActionImpl seq = (SeqActionImpl) ap.getCircusAction();
							if (seq.getRightAction() instanceof MuActionImpl){
								MuActionImpl mu = (MuActionImpl) seq.getRightAction();
								SeqActionImpl seq2 = (SeqActionImpl) mu.getCircusAction();
								if (seq2.equals(applicable)){
									seq2.setLeftAction(seqNovo.getLeftAction());
									seq2.setRightAction(seqNovo.getRightAction());
									break;
								}
							}
						}
					}
				}
			}
		}


		return spec;
	}

	public Term VarExpRec(Term applicable, Term novo, Term ast) {

		VarDeclCommand varNovo = (VarDeclCommand) novo;
		Spec spec = (Spec) ast;
		ListTerm<Sect> lista =  spec.getSect();	


		ZSect secao = (ZSect)lista.get(0);
		ZParaList listaPara = secao.getZParaList();
		for (Para p : listaPara) {

			// TODO melhorar essa parte do programa
			if ( p instanceof ProcessParaImpl){
				ProcessPara pp = (ProcessPara) p;
				BasicProcess bp = (BasicProcess) pp.getCircusBasicProcess();
				listaPara = bp.getZParaList();
				for (Para para : listaPara) {
					if(para instanceof ActionPara){
						ActionPara ap = (ActionPara) para;
						if(ap.getCircusAction() instanceof SeqActionImpl){
							SeqActionImpl seq = (SeqActionImpl) ap.getCircusAction();
							if (seq.getRightAction().equals(applicable)){
								seq.setRightAction(varNovo);
								seq.setLeftAction(seq.getLeftAction());
								break;
							}
						}
					}
				}
			}
		}


		return spec;
	}

	public void escreveTaticaSerializada(Tatica tatica) {
		try {
			gerTaticas.writeTactic(tatica);
		} catch (FileNotFoundException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		} catch (IOException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}

	}

	public Tatica getTTatica(TTatica tTatica, List<Object> argMap) throws TacticNotFound {

		List listaArgs = gerTaticas.getListArgs();
		Tatica tatica = new Tatica();
		boolean ifExists = false;
		for (int i =  0; i < listaArgs.size() ; i++) {
			Pair<String,Pair<String,Object>> pair = (Pair<String,Pair<String,Object>>) listaArgs.get(i);
			if (tTatica.getNome().contains(pair.getFirst())){
				Pair<String,Object> second = (Pair<String,Object>) pair.getSecond();
				TacticComponent t = (TacticComponent) second.getSecond();
				Tatica t2 = new Tatica();
				t2.settComp(t);
				t2.setId(second.getFirst());
				t2.setArgumentos(t.getArgumentos());
				String path = GerenciadorTaticas.DIRETORIO_TATICAS_SERIALIZE+
				pair.getFirst()+".tac";
				t2.setPath(path);
				tatica = t2;
				ifExists = true;
				break;
			}
			else if(pair.getFirst().equals(tTatica.getNome()+ "rec")){
				ifExists = true;
				GerenciadorTaticas.rec = true;
				return null;				
			}
		}
		if (!ifExists)
			throw new TacticNotFound(tTatica.getNome());

		return tatica;
	}

	public Term contatenar(String left, String right,
			List<Pair<String, Term>> mapeamento) {

		Term term = null;

		for (Pair<String, Term> pair : mapeamento) {
			if (left.equals(pair.getFirst().toString())){
				left = pair.getSecond().toString();
				String nome = left + right;
				ZName name = this.getZName(nome);
				Term ast = this.retornarProgAtual();
				term = TermFromANameGetter.getTerm(name, ast);
				break;
			}
		}	

		return term;

	}

	public void updateTacticList() {
		gerInterface.updateTacticList();
		
	}

}

