#define _XOPEN_SOURCE 600

#include <pthread.h>
#include <stdlib.h>
#include <stdio.h>
#include "tmwtypes.h"

#define THREADS 2

#define Air_IN_NO_ACTIVE_CHILD         (0U)
#define Air_IN_Off                     (1U)
#define Air_IN_On                      (2U)
#define Air_IN_PowerOff                (1U)
#define Air_IN_PowerOn                 (2U)
#define Air_event_CLOCK                (1)
#define Air_event_SWITCH               (0)
#define CALL_EVENT		       (-1)

typedef struct {
  uint8_T airflow;                     /* '<Root>/Air Controller' */
} BlockIO_Air;

/* Block states (auto storage) for system '<Root>' */
typedef struct {
  uint8_T is_active_c1_Air;            /* '<Root>/Air Controller' */
  uint8_T is_c1_Air;                   /* '<Root>/Air Controller' */
  uint8_T is_active_FAN2;              /* '<Root>/Air Controller' */
  uint8_T is_FAN2;                     /* '<Root>/Air Controller' */
  uint8_T is_active_SpeedValue;        /* '<Root>/Air Controller' */
  uint8_T is_active_FAN1;              /* '<Root>/Air Controller' */
  uint8_T is_FAN1;                     /* '<Root>/Air Controller' */
} D_Work_Air;

/* External inputs (root inport signals with auto storage) */
typedef struct {
  real_T temp;                         /* '<Root>/temp' */
  real_T inputevents[2];               /* '<Root>/ input events ' */
} ExternalInputs_Air;

/* External outputs (root outports fed by signals with auto storage) */
typedef struct {
  uint8_T airflow;                     /* '<Root>/airflow' */
} ExternalOutputs_Air;

int32_T _sfEvent_Air_;

/* Block signals (auto storage) */
BlockIO_Air Air_B;

/* Block states (auto storage) */
D_Work_Air Air_DWork;


/* External inputs (root inport signals with auto storage) */
ExternalInputs_Air Air_U;

/* External outputs (root outports fed by signals with auto storage) */
ExternalOutputs_Air Air_Y;


// Barrier variable
pthread_barrier_t barr;

static void synchronise(void) {
	int rc = pthread_barrier_wait(&barr);
	if (rc != 0 && rc!=PTHREAD_BARRIER_SERIAL_THREAD) {
		printf("Could not wait on barrier\n");
		exit(-1);
	}
}


/* Forward declaration for local functions */
static void Air_chartstep_c1_Air(void);

/* Function for Stateflow: '<Root>/Air Controller' */
static void Air_chartstep_c1_Air(void)
{
  /* During: Air Controller */
  if (Air_DWork.is_active_c1_Air == 0) {
    /* Entry: Air Controller */
    Air_DWork.is_active_c1_Air = 1U;

    /* Transition: '<S1>:16' */
    /* Entry 'PowerOff': '<S1>:9' */
    Air_DWork.is_c1_Air = Air_IN_PowerOff;
    Air_B.airflow = 0U;
  } else {
    switch (Air_DWork.is_c1_Air) {
     case Air_IN_PowerOff:
      /* During 'PowerOff': '<S1>:9' */
      if (_sfEvent_Air_ == Air_event_SWITCH) {
        /* Transition: '<S1>:17' */
        /* Exit 'PowerOff': '<S1>:9' */
        /* Entry 'PowerOn': '<S1>:8' */
        Air_DWork.is_c1_Air = Air_IN_PowerOn;

        /* Entry 'FAN1': '<S1>:7' */
        Air_DWork.is_active_FAN1 = 1U;

        /* Transition: '<S1>:13' */
        /* Entry 'Off': '<S1>:5' */
        Air_DWork.is_FAN1 = Air_IN_Off;

        /* Entry 'FAN2': '<S1>:3' */
        Air_DWork.is_active_FAN2 = 1U;

        /* Transition: '<S1>:12' */
        /* Entry 'Off': '<S1>:1' */
        Air_DWork.is_FAN2 = Air_IN_Off;

        /* Entry 'SpeedValue': '<S1>:4' */
        Air_DWork.is_active_SpeedValue = 1U;
      }
      break;

     case Air_IN_PowerOn:
      /* During 'PowerOn': '<S1>:8' */
      if (_sfEvent_Air_ == Air_event_SWITCH) {
        /* Transition: '<S1>:18' */
        /* Exit 'SpeedValue': '<S1>:4' */
        Air_DWork.is_active_SpeedValue = 0U;

        /* Exit 'Off': '<S1>:1' */
        Air_DWork.is_FAN2 = (uint8_T)Air_IN_NO_ACTIVE_CHILD;

        /* Exit 'On': '<S1>:2' */
        /* Exit 'FAN2': '<S1>:3' */
        Air_DWork.is_active_FAN2 = 0U;

        /* Exit 'Off': '<S1>:5' */
        Air_DWork.is_FAN1 = (uint8_T)Air_IN_NO_ACTIVE_CHILD;

        /* Exit 'On': '<S1>:6' */
        /* Exit 'FAN1': '<S1>:7' */
        Air_DWork.is_active_FAN1 = 0U;

        /* Exit 'PowerOn': '<S1>:8' */
        /* Entry 'PowerOff': '<S1>:9' */
        Air_DWork.is_c1_Air = Air_IN_PowerOff;
        Air_B.airflow = 0U;
      } else {
        /* During 'FAN1': '<S1>:7' */
	// this is executed by the server
	// we must wait for FAN1 to be ready for execution
        synchronise();
        /* During 'FAN2': '<S1>:3' */
        switch (Air_DWork.is_FAN2) {
         case Air_IN_Off:
          /* During 'Off': '<S1>:1' */
          if (Air_U.temp >= 150.0) {
            /* Transition: '<S1>:11' */
            /* Exit 'Off': '<S1>:1' */
            /* Entry 'On': '<S1>:2' */
            Air_DWork.is_FAN2 = Air_IN_On;
          }
          break;

         case Air_IN_On:
          /* During 'On': '<S1>:2' */
          if (Air_U.temp < 150.0) {
            /* Transition: '<S1>:10' */
            /* Exit 'On': '<S1>:2' */
            /* Entry 'Off': '<S1>:1' */
            Air_DWork.is_FAN2 = Air_IN_Off;
          }
          break;

         default:
          /* Transition: '<S1>:12' */
          /* Entry 'Off': '<S1>:1' */
          Air_DWork.is_FAN2 = Air_IN_Off;
          break;
        }
	synchronise();
        /* During 'SpeedValue': '<S1>:4' */
        Air_B.airflow = (uint8_T)((Air_DWork.is_FAN1 == Air_IN_On) +
          (Air_DWork.is_FAN2 == Air_IN_On));
      }
      break;

     default:
      /* Transition: '<S1>:16' */
      /* Entry 'PowerOff': '<S1>:9' */
      Air_DWork.is_c1_Air = Air_IN_PowerOff;
      Air_B.airflow = 0U;
      break;
    }
  }
}

static void Air_FAN1(void)
{
	// we must wait for FAN1 to be ready for execution
	synchronise();
        /* During 'FAN1': '<S1>:7' */
        switch (Air_DWork.is_FAN1) {
         case Air_IN_Off:
          /* During 'Off': '<S1>:5' */
          if (Air_U.temp >= 120.0) {
            /* Transition: '<S1>:14' */
            /* Exit 'Off': '<S1>:5' */
            /* Entry 'On': '<S1>:6' */
            Air_DWork.is_FAN1 = Air_IN_On;
          }
          break;

         case Air_IN_On:
          /* During 'On': '<S1>:6' */
          if (Air_U.temp < 120.0) {
            /* Transition: '<S1>:15' */
            /* Exit 'On': '<S1>:6' */
            /* Entry 'Off': '<S1>:5' */
            Air_DWork.is_FAN1 = Air_IN_Off;
          }
          break;

         default:
          /* Transition: '<S1>:13' */
          /* Entry 'Off': '<S1>:5' */
          Air_DWork.is_FAN1 = Air_IN_Off;
          break;
        }
	synchronise();
}

static void Air_output(int_T tid)
{
  int32_T c_previousEvent;
  if (Air_U.inputevents[0]) {
    /* Stateflow: '<Root>/Air Controller' */
    /* Gateway: Air Controller */
   
      /* Event: '<S1>:21' */
      c_previousEvent = _sfEvent_Air_;
      _sfEvent_Air_ = Air_event_SWITCH;
      Air_chartstep_c1_Air();
      _sfEvent_Air_ = c_previousEvent;
  }

  if (Air_U.inputevents[1]) {
      /* Event: '<S1>:22' */
      c_previousEvent = _sfEvent_Air_;
      _sfEvent_Air_ = Air_event_CLOCK;
      Air_chartstep_c1_Air();
      _sfEvent_Air_ = c_previousEvent;
    }

  /* Outport: '<Root>/airflow' */
  Air_Y.airflow = Air_B.airflow;
}

void MdlInitialize(void)
{
  /* Machine initializer */
  _sfEvent_Air_ = CALL_EVENT;

  /* InitializeConditions for Stateflow: '<Root>/Air Controller' */
  Air_DWork.is_active_FAN1 = 0U;
  Air_DWork.is_FAN1 = 0U;
  Air_DWork.is_active_FAN2 = 0U;
  Air_DWork.is_FAN2 = 0U;
  Air_DWork.is_active_SpeedValue = 0U;
  Air_DWork.is_active_c1_Air = 0U;
  Air_DWork.is_c1_Air = 0U;
  Air_B.airflow = 0U;
}


/* the next functions are used to simulate the execution of the chart, they set up threads, read inputs and write outputs. */

/* thread that execute the state FAN1 iteratively */
void *FAN1(void *arg) {
	int i = 0;
	while(1) {
		i++;
		Air_FAN1();
	}
}

/* this variables is used to count the cycles and simulate the value of the input data and events */
int cycle = 0;

/* this function is responsible for reading the inputs. In this case, it generates the input according
to the cycle number. In a real implementation, this function would not exist, as before the cycle of
the chart is executed, the structure Air_U would be populated appropriately. */
void read_inputs() {
	if (Air_Y.airflow == 0) {
		Air_U.temp=Air_U.temp+2;
	} else if (Air_Y.airflow == 1) {
		Air_U.temp=Air_U.temp+1;
	} else if (Air_Y.airflow == 2) {
		Air_U.temp=Air_U.temp-1;
	}
	if (cycle == 1 ) Air_U.inputevents[0] = 1;
	else Air_U.inputevents[0] = 0;
	Air_U.inputevents[1]=1;
	printf("Temperature: %f\n", Air_U.temp);
	cycle++;
}

/* this function is similar to the previous one, but for outputs. In this case, it simply prints the output and waits
for 0.1 second. */
void write_outputs() {
	printf("Airflow: %d\n", Air_Y.airflow);
	usleep(100000);
}

/* thread that executes the chart iteratively. In a real application, this function would probably involve
a scheduler that controls the execution of Air_output with respect to the rest of the system. */
void *ExecuteChart(void *arg) {
	MdlInitialize();
	while(1) {
		read_inputs();
		Air_output(1);
		write_outputs();

	}
}
		
int main(int argc, char **argv) {
    pthread_t thr[THREADS];

    // Barrier initialization
	if(pthread_barrier_init(&barr, NULL, THREADS))
	{
		printf("Could not create a barrier\n");
		return -1;
	}
	printf("Barrier created successfully\n");
	if(pthread_create(&thr[0],NULL,&FAN1,NULL)) {
		printf("Could not create thread 0\n");
		return -1;
	}
	printf("Thread fan1 created\n");
	if(pthread_create(&thr[1],NULL,&ExecuteChart,NULL)) {
		printf("Could not create thread 1\n");
		return -1;
	}
	printf("Thread executechart created\n");
	if(pthread_join(thr[1], NULL))
 	{
  		printf("Could not join thread %d\n", 1);
    		return -1;
	}
	if(pthread_join(thr[0], NULL))
 	{
  		printf("Could not join thread %d\n", 0);
    		return -1;
	}
    return 0;
}
