package uk.ac.york.tokeneer;

import javax.realtime.AperiodicParameters;
import javax.realtime.HighResolutionTime;
import javax.realtime.PriorityParameters;
import javax.safetycritical.OneShotEventHandler;
import javax.safetycritical.PriorityScheduler;
import javax.safetycritical.StorageConfigurationParameters;

import uk.ac.york.tokeneer.realworld.Alarm;
import uk.ac.york.tokeneer.realworld.Door;
import uk.ac.york.tokeneer.realworld.DoorState;
import uk.ac.york.tokeneer.realworld.Latch;
import uk.ac.york.tokeneer.realworld.LatchState;

/**
 * {@link OneShotEventHandler} for activating an Alarm after the timeout has
 * been exceeded.
 * 
 * @author Jon Co
 * 
 */
public class AlarmTimeout extends AbstractTimeout {

	private static final String TAG = "AlarmTimeout";

	// Peripheral controls - used to determine security status
	private final Alarm alarm;
	private final Door door;
	private final Latch latch;

	/**
	 * Default Constructor
	 * 
	 * @param timeout
	 *            Duration before the alarm should be sounded if the door is in
	 *            an insecure state.
	 * @param controller
	 *            Interface to real world controls. Used to activate the alarm.
	 * @param state
	 *            The internal state of the system.
	 */
	// @formatter:off
	public AlarmTimeout(
			final HighResolutionTime timeout,
			final Alarm alarm,
			final Door door,
			final Latch latch) {
		
		super(timeout,
				new PriorityParameters(PriorityScheduler.instance().getMaxPriority()), 
				new AperiodicParameters(),
				new StorageConfigurationParameters(4096, 4096, 4096), 
				4096);
		this.alarm = alarm;
		this.door = door;
		this.latch = latch;
	}
	// @formatter:on

	@Override
	public void handleEvent() {
		Log.d(TAG, "Alarm Timeout Expired");
		if (!( this.door.getState() == DoorState.CLOSED
				&& this.latch.getState() == LatchState.LOCKED )) {
			this.alarm.activate();
		}
	}

	/**
	 * Start the timeout for the alarm.
	 */
	public synchronized void startTimeout() {
		Log.d(TAG, "Alarm Timeout Started");
		super.startTimeout();
	}

	@Override
	public synchronized boolean deschedule() {
		Log.d(TAG, "Alarm Timeout Stopped");
		return super.deschedule();
	}
	
}
