package uk.ac.york.tokeneer;

import java.util.concurrent.atomic.AtomicReference;

import javax.realtime.RelativeTime;
import javax.safetycritical.AperiodicEvent;
import javax.safetycritical.Mission;

import uk.ac.york.tokeneer.realworld.Alarm;
import uk.ac.york.tokeneer.realworld.Door;
import uk.ac.york.tokeneer.realworld.DummyAlarm;
import uk.ac.york.tokeneer.realworld.DummyDoor;
import uk.ac.york.tokeneer.realworld.DummyFingerReader;
import uk.ac.york.tokeneer.realworld.DummyLatch;
import uk.ac.york.tokeneer.realworld.DummyTokenReader;
import uk.ac.york.tokeneer.realworld.FingerReader;
import uk.ac.york.tokeneer.realworld.Latch;
import uk.ac.york.tokeneer.realworld.TokenReader;

/**
 * Main {@link Mission} for Tokeneer System.
 * 
 * This is the only {@code Mission} in this application.
 * 
 * @author Jon Co
 * 
 */
public class TokeneerMission extends Mission {
	
	private final ConfigData configData;

	// State of the current User Entry Process
	private final AtomicReference<UserEntryState> userEntryState;

	// Door Peripherals
	private final Alarm alarm;
	private final Door door;
	private final Latch latch;

	// User Input Peripherals
	private final TokenReader tokenReader;
	private final FingerReader fingerReader;

	/**
	 * Default Constructors
	 */
	public TokeneerMission(ConfigData configData) {
		super();
		
		this.configData = configData;
		
		this.userEntryState = new AtomicReference<UserEntryState>(
				UserEntryState.IDLE);
		
		this.alarm = new DummyAlarm();
		this.door = new DummyDoor();
		this.latch = new DummyLatch();
		
		this.tokenReader = new DummyTokenReader();
		this.fingerReader = new DummyFingerReader();
	}

	@Override
	public long missionMemorySize() {
		// TODO: This is a best guess, what should this be set to?
		return 1000000;
	}

	// @formatter:off
	@Override
	protected void initialize() {
		
		/* ====================================================================
		 * HANDLERS FOR DOOR SECURITY 
		 * ====================================================================
		 */
		// Alarm Timeout
		final AlarmTimeout alarmTimeout = 
				new AlarmTimeout(
						new RelativeTime(this.configData.getAlarmTimeout(), 0), 
						this.alarm,
						this.door,
						this.latch);
		
		// Latch Timeout
		final LatchTimeout latchTimeout = 
				new LatchTimeout(
						new RelativeTime(this.configData.getLatchTimeout(), 0), 
						this.latch);

		// Handler for updating door system status
		new DoorLatchAlarmUpdateHandler(
				new RelativeTime(this.configData.getUpdateInterval(), 0),
				this.alarm, 
				this.door, 
				this.latch);
		
		/* ====================================================================
		 * HANDLERS FOR USER ENTRY
		 * ====================================================================
		 */
		// Timeouts
		final TokenRemovalTimeout tokenRemovalTimeout = 
				new TokenRemovalTimeout(
						new RelativeTime(this.configData.getTokenRemovalTimout(), 0),
						this.userEntryState,
						this.tokenReader);
		
		final FingerTimeout fingerTimeout =
				new FingerTimeout(
						new RelativeTime(this.configData.getFingerTimeout(), 0),
						this.userEntryState);
		
		// Token Removal
		final TokenRemovalHandler tokenRemovalHandler =
				new TokenRemovalHandler(
						this.userEntryState,
						this.latch,
						this.tokenReader,
						latchTimeout,
						alarmTimeout);
		final AperiodicEvent tokenRemovalEvent = 
				new AperiodicEvent(tokenRemovalHandler);
		
		// Token Input
		final TokenInuptHandler tokenInputHandler = 
				new TokenInuptHandler(
						this.userEntryState,
						this.tokenReader,
						fingerTimeout);
		final AperiodicEvent tokenInputEvent = 
				new AperiodicEvent(tokenInputHandler);
		
		// Finger Input
		final FingerInputHandler fingerInputHandler = new FingerInputHandler(
				this.userEntryState,
				this.fingerReader, 
				tokenRemovalTimeout);
		final AperiodicEvent fingerInputEvent = 
				new AperiodicEvent(fingerInputHandler);

		// Dummy input simulator
		new DummyInputHandler(
				tokenRemovalEvent, 
				tokenInputEvent, 
				fingerInputEvent);
	}
	// @formatter:on
}
