package javax.safetycritical;

import javax.realtime.AbsoluteTime;
import javax.realtime.AperiodicParameters;
import javax.realtime.HighResolutionTime;
import javax.realtime.OneShotTimer;
import javax.realtime.PriorityParameters;
import javax.realtime.RelativeTime;

/**
 * This class permits the automatic execution of time-triggered code. The
 * handle- AsyncEvent method behaves as if the handler were attached to a
 * one-shot timer asynchronous event. This class is abstract, non-abstract
 * sub-classes must implement the method handleAsyncEvent and may override the
 * default cleanup method.
 * 
 * XXX: Implementation has not been tested or verified - was written for
 * undergrad project. Where available, JavaDoc has been taken from the draft
 * spec.
 * 
 * @author Jon Co
 * 
 */
public abstract class OneShotEventHandler extends ManagedEventHandler {

	// Timer for one shot event
	private OneShotTimer timer;

	/**
	 * Constructor
	 * 
	 * @param priority
	 *            Specifies the priority parameters for this event handler. Must
	 *            not be null.
	 * @param time
	 *            Specifies the time at which the handler should be release. A
	 *            relative time is relative to the start of the associated
	 *            mission. An absolute time that is before the mission is
	 *            started is equivalent to a relative time of 0. A null
	 *            parameter is equivalent to a relative time of 0.
	 * @param release
	 *            Specifies the aperiodic release parameters, in particular the
	 *            deadline miss handler. A null parameters indicates that there
	 *            is no deadline associated with this handler.
	 * 
	 * @param storage
	 *            Specifies the storage parameters; it must not be null.
	 * @param size
	 *            Size of this handlers private memory.
	 */
	public OneShotEventHandler(PriorityParameters priority,
			HighResolutionTime time, AperiodicParameters release,
			StorageConfigurationParameters storage, long size) {
		this(priority, time, release, storage, size, null);
	}

	/**
	 * Constructor
	 * 
	 * @param priority
	 *            Specifies the priority parameters for this event handler. Must
	 *            not be null.
	 * @param time
	 *            Specifies the time at which the handler should be release. A
	 *            relative time is relative to the start of the associated
	 *            mission. An absolute time that is before the mission is
	 *            started is equivalent to a relative time of 0. A null
	 *            parameter is equivalent to a relative time of 0.
	 * @param release
	 *            Specifies the aperiodic release parameters, in particular the
	 *            deadline miss handler. A null parameters indicates that there
	 *            is no deadline associated with this handler.
	 * 
	 * @param storage
	 *            Specifies the storage parameters; it must not be null.
	 * @param size
	 *            Size of this handlers private memory.
	 * @param name
	 *            Name of this handler.
	 */
	public OneShotEventHandler(PriorityParameters priority,
			HighResolutionTime time, AperiodicParameters release,
			StorageConfigurationParameters storage, long size, String name) {
		super(priority, release, storage, size, name);

		if (time == null) {
			time = new RelativeTime(0, 0);
		}

		this.timer = new OneShotTimer(time, this);
	}

	// XXX: What does the join method actually do? Copied from PEH
	void join() {
		this.timer.stop();
		super.join();
	}

	public boolean deschedule() {
		final boolean isRunning = this.timer.isRunning();
		this.timer.stop();
		return isRunning;
	}

	public AbsoluteTime getNextReleaseTime(AbsoluteTime dest)
			throws IllegalStateException {
		return this.timer.getFireTime(dest);
	}

	public void scheduleNextReleaseTime(HighResolutionTime time) {
		this.timer.reschedule(time);
		this.timer.start();
	}

}
