/**
 * @author Frank Zeyda, Kun Wei
 */
package cdx;

import javax.safetycritical.AperiodicEvent;

/**
 * This class is used to control the execution of OutputCollisionsHandler.
 * In particular, it ensures the release of this handler once all instances
 * of DetectorHandler have finished their computational work.
 *
 */


public class DetectorControl {
  
  /** 
   * An array of bool values to express each DetectorHandler to be 
   * idle or done
   */
  private final boolean[] idle;

  /* Note that the following field is not in the OhCircus class model; this
   * is because the class model only accounts for the data operations in the
   * class but not the active behaviour of firing the SCJ event below. */

   /** The event to release OutputCollisionHandler */
   private final AperiodicEvent output;

  /**
   * Constructor
   * @param event the event to release OutputCollisionHandler
   * @param n the number of DetectorHandler in parallel
   */
  public DetectorControl(AperiodicEvent event, int n) {
    idle = new boolean[n];
    output = event;
  }

  /** 
   * Method to record the state of each DetectorHandler.
   * True if a DetectorHandler has finished, otherwise false
   */
  public synchronized void start() {
    for (int index = 0; index < idle.length; index++) {
      idle[index] = false;
    }
  }

  /** 
   * Method to release OutputCollisionHandler if all DetectorHandlers have
   * finihsed.
   */ 
  public synchronized void notify(int id) {
    idle[id - 1] = true;
    if (done()) {
        /* Release handler to output the detected collisions. */
        output.fire();
    }
  }

  /**
   * Method to check whether all DetectorHandlers have finished.
   * @return true if all have done, otherwise false.
   */
  protected synchronized boolean done() {
    boolean result = true;
    for (int index = 0; index < idle.length; index++) {
      if (!idle[index]) { result = false; }
    }
    return result;
  }
}
