/**
 *  This file is part of miniCDx benchmark of oSCJ.
 *
 *   miniCDx is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU Lesser General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   miniCDx is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public License
 *   along with miniCDx.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 *   Copyright 2009, 2010
 *   @authors  Daniel Tang, Ales Plsek, Kun Wei, Frank Zeyda
 *
 *   See: http://sss.cs.purdue.edu/projects/oscj/
 */
package cdx;

import java.util.Set;

/**
 * The class StateTable records the state of the previous frame.
 *
 * Allocated CallSign objects and Vector3d objects for positions are held in
 * mission memory.
 */
public class StateTable {
  /** Mapping CallSign -> Vector3d. We use a customised HashMap here. */
  final private CHashMap motionVectors;

  /* Fields to manage pool of pre-allocation objects in mission memory. */
  
  /** The pre-allocated callsigns */
  public CallSign[] allocatedCallSigns;
  /** The pre-allocated vectors */
  public Vector3d[] allocatedVectors;
  /** To record the used pairs of callsign/vector */
  public int usedSlots;

  /**Constructor that initialise CHahsMap, callsign and vectors. */
  public StateTable() {
    motionVectors = new CHashMap(Constants.MAX_OF_PLANES);
    initObjectPool();
  }

  /** Method to allocate memory for callsigns and vectors*/
  private void initObjectPool() {
    allocatedCallSigns = new CallSign[Constants.NUMBER_OF_PLANES];
    allocatedVectors = new Vector3d[Constants.NUMBER_OF_PLANES];
    /* Pre-allocate all objects during construction. */
    for (int i = 0; i < allocatedVectors.length; i++){
      allocatedCallSigns[i] = new CallSign();
      allocatedVectors[i] = new Vector3d();
    }
    usedSlots = 0;
  }

  /** Return a set of all callsigns */
  public Set getCallSigns() {
    return motionVectors.keySet();
  }

  /** Put a callsign and the related vector(x,y,z) into the StateTable. If
   * the callsign has existed, it updates the value; if not, add in as a new
   * entry.
   * @param callsign   a callsign of aircraft
   * @param x          x axis of the postion
   * @param y          y axis of the position
   * @param z          z axis of the position
   */
  public void put(CallSign callsign,
    final float x, final float y, final float z) {

    Vector3d v = (Vector3d) motionVectors.get(callsign);

    if (v == null) {
      /* No position vector yet in the map for callsign. */
      if (usedSlots == allocatedCallSigns.length) {
        Error.abort("Exceeding storage capacity in StateTable.");
      }
      /* Obtain pre-allocated CallSign object from the store. */
      CallSign c = allocatedCallSigns[usedSlots];

      /* Copy content of callsign to the respective persistent object. */
      for(int i = 0; i < callsign.get().length; i++) {
        c.get()[i] = callsign.get()[i];
      }

      /* Obtain pre-allocated Vector3d object from the store. */
      v = allocatedVectors[usedSlots++];

      /* Put new entry into the (customised) CHashMap. */
      motionVectors.put(c, v);
    }

    /* Finally update the components of the position vector. */
    v.x = x;
    v.y = y;
    v.z = z;
  }

  /**
   * Method to return a position by its callsign
   * @param callsign  the callsign for aircraft
   * @return a postion 
   */
  public Vector3d get(final CallSign callsign) {
    return (Vector3d) motionVectors.get(callsign);
  }

  /** 
   * Method to remove a postion from the StateTable by its callsign
   * @param callsign   the aircraft to be removed from the table
   */
  public Vector3d remove(final CallSign callsign) {
    return (Vector3d) motionVectors.remove(callsign);
  }
}
