/**
 *
 * @author Frank Zeyda, Kun Wei
 */
package cdx;

import javax.realtime.PriorityParameters;

import javax.safetycritical.AperiodicEvent;
import javax.safetycritical.AperiodicEventHandler;
import javax.safetycritical.AperiodicParameters;
import javax.safetycritical.PriorityScheduler;
import javax.safetycritical.StorageParameters;

import javacp.util.Iterator;
import javacp.util.List;

/**
 * DetectorHandler instances carry out the actual collisions detection.
 *
 * Each DetectorHandler takes in a list of lists of motions, each set contains
 * the motions that span cross a same voxel. 
 * 
 * The collisions detection carried out here considers the Z axis, and also 
 * updates the collisions number recorded in the CDxMission.
 * 
 * If this DetectorHanndler has finished, it updates its state in 
 * DetectorControl.
 */

public class DetectorHandler extends AperiodicEventHandler {

  /** To get the control of the current mission. */
  public final CDxMission mission;
  /** To get the control of DetectorControl.  */
  public final DetectorControl control;
  /** The Identity of this DetectorHandler. */
  public final int id;

  /**
   * Constructor
   *
   * @param mission         the current mission
   * @param control         the control of DectectoControl
   * @param id              the assigned ID
   * @param event_bound     the event to release this DetectorHandler
   */
   public DetectorHandler(
      CDxMission mission,
      DetectorControl control,
      int id,
      AperiodicEvent event_bound) {
    super(
      new PriorityParameters(Constants.DETECTOR_PRIORITY),
      new AperiodicParameters(null, null),
      new StorageParameters(32768, 4096, 4096),
      event_bound, "DetectorHandler");
    this.mission = mission;
    this.control = control;
    this.id = id;
  }

 /** 
  * Major actions happen within handlerEvent() including calculation, 
  * updation of the number of collisions, and notification to DetectorControl. 
  */
  public void handleEvent() {
    /* Local variable to hold the collisions result for the partition. */
    int colls;
   
    /* Calculate the number of collisions for this detector's work. */
    colls = CalcPartCollisions();

    /* Record collisions result with the mission. */
    mission.recColls(colls);

    /* Notify DetectorControl that this handler has finished. */
    control.notify(id);
  }

  /* Implementation of the CalcPartCollisions action in the S anchor. */

 
  /**
   * Method to get the allocated list of lists of motions by its ID, and detect 
   * collisions
   * @return the number of collisions
   */
  
  public int CalcPartCollisions() {
    int colls = 0;

    /* Get work for this detector via the shared Partition object. */
    List work = mission.work.getDetectorWork(id);

    /* Check each set of motions  */
    for (Iterator iter = work.iterator(); iter.hasNext(); ) {
        colls += determineCollisions((List) iter.next());
    }

    return colls;
  }

  /**
   * Compute the number of collisions for a List of Motion objects.
   * @param motions  the list of motions to be checked
   * @return         the number of collisions within this list
   */
  public int determineCollisions(final List motions) {
    int colls = 0;
    /* Comprare each pair of motions in order  */
    for (int i = 0; i < motions.size() - 1; i++) {
      Motion one = (Motion) motions.get(i);
      for (int j = i + 1; j < motions.size(); j++) {
        Motion two = (Motion) motions.get(j);
        Vector3d v = one.findIntersection(two);
        if (v != null) { colls++; }
      }
    }
    return colls;
  }
}
