/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License. 
 *
 */
package org.apache.bcel.generic;

import org.apache.bcel.classfile.ConstantCP;
import org.apache.bcel.classfile.ConstantPool;

/**
 * Super class for InvokeInstruction and FieldInstruction, since they have
 * some methods in common!
 *
 * @version $Id: FieldOrMethod.java 1551450 2013-12-17 02:51:54Z dbrosius $
 * @author  <A HREF="mailto:m.dahm@gmx.de">M. Dahm</A>
 */
public abstract class FieldOrMethod extends NameSignatureInstruction implements LoadClass {

    private static final long serialVersionUID = 2036985877748835708L;


    /**
     * Empty constructor needed for the Class.newInstance() statement in
     * Instruction.readInstruction(). Not to be used otherwise.
     */
    FieldOrMethod() {
    }


    /**
     * @param index to constant pool
     */
    protected FieldOrMethod(short opcode, int index) {
        super(opcode, index);
    }


    /** @return name of the referenced class/interface
     *  @deprecated If the instruction references an array class,
     *    this method will return "java.lang.Object".
     *    For code generated by Java 1.5, this answer is
     *    sometimes wrong (e.g., if the "clone()" method is
     *    called on an array).  A better idea is to use
     *    the getReferenceType() method, which correctly distinguishes
     *    between class types and array types.
     */
    @Deprecated
    public String getClassName( ConstantPoolGen cpg ) {
        ConstantPool cp = cpg.getConstantPool();
        ConstantCP cmr = (ConstantCP) cp.getConstant(index);
        String className = cp.getConstantString(cmr.getClassIndex(),
                org.apache.bcel.Constants.CONSTANT_Class);
        if (className.startsWith("[")) {
            // Turn array classes into java.lang.Object.
            return "java.lang.Object";
        }
        return className.replace('/', '.');
    }


    /** @return type of the referenced class/interface
     * @deprecated If the instruction references an array class,
     *    the ObjectType returned will be invalid.  Use
     *    getReferenceType() instead.
     */
    @Deprecated
    public ObjectType getClassType( ConstantPoolGen cpg ) {
        return ObjectType.getInstance(getClassName(cpg));
    }


    /**
     * Return the reference type representing the class, interface,
     * or array class referenced by the instruction.
     * @param cpg the ConstantPoolGen used to create the instruction
     * @return an ObjectType (if the referenced class type is a class
     *   or interface), or an ArrayType (if the referenced class
     *   type is an array class)
     */
    public ReferenceType getReferenceType( ConstantPoolGen cpg ) {
        ConstantPool cp = cpg.getConstantPool();
        ConstantCP cmr = (ConstantCP) cp.getConstant(index);
        String className = cp.getConstantString(cmr.getClassIndex(),
                org.apache.bcel.Constants.CONSTANT_Class);
        if (className.startsWith("[")) {
            return (ArrayType) Type.getType(className);
        } else {
            className = className.replace('/', '.');
            return ObjectType.getInstance(className);
        }
    }


    /** @return type of the referenced class/interface
     */
    public ObjectType getLoadClassType( ConstantPoolGen cpg ) {
        return getClassType(cpg);
    }
}
