/*
 * Decompiled with CFR 0.152.
 */
package gov.nasa.jpf.tools;

import gov.nasa.jpf.Config;
import gov.nasa.jpf.JPF;
import gov.nasa.jpf.ListenerAdapter;
import gov.nasa.jpf.jvm.AnnotationInfo;
import gov.nasa.jpf.jvm.ElementInfo;
import gov.nasa.jpf.jvm.FieldInfo;
import gov.nasa.jpf.jvm.JVM;
import gov.nasa.jpf.jvm.MethodInfo;
import gov.nasa.jpf.jvm.StackFrame;
import gov.nasa.jpf.jvm.ThreadInfo;
import gov.nasa.jpf.jvm.bytecode.ASTORE;
import gov.nasa.jpf.jvm.bytecode.Instruction;
import gov.nasa.jpf.jvm.bytecode.InvokeInstruction;
import gov.nasa.jpf.jvm.bytecode.PUTFIELD;
import gov.nasa.jpf.jvm.bytecode.PUTSTATIC;
import gov.nasa.jpf.jvm.bytecode.ReturnInstruction;
import gov.nasa.jpf.report.ConsolePublisher;
import gov.nasa.jpf.report.Publisher;
import gov.nasa.jpf.search.Search;
import gov.nasa.jpf.test.SequenceOp;
import gov.nasa.jpf.test.SequenceProcessor;
import gov.nasa.jpf.util.Trace;
import java.io.IOException;
import java.io.PrintWriter;
import java.util.ArrayList;
import java.util.HashMap;

public class SequenceAnalyzer
extends ListenerAdapter {
    Trace<SequenceOp> sequenceOps = new Trace();
    Trace<SequenceContext> sequenceContext = new Trace();
    HashMap<MethodInfo, AnnotationInfo> annotations = new HashMap();
    Config config;
    ArrayList<String> processedSequences;
    SequenceProcessor processor;

    public SequenceAnalyzer(Config conf, JPF jpf) {
        this.config = conf;
        this.processedSequences = new ArrayList();
        try {
            this.processor = conf.getInstance("sequence.processor.class", SequenceProcessor.class);
        }
        catch (Config.Exception x) {
            throw new Exception("cannot instantiate sequence processor");
        }
        if (this.processor == null) {
            this.processor = new DefaultProcessor();
        }
        jpf.addPublisherExtension(ConsolePublisher.class, this);
    }

    boolean isActiveSequence(String id) {
        SequenceContext sc = this.sequenceContext.getLastOp();
        if (sc != null) {
            return sc.get(id) != null;
        }
        return false;
    }

    boolean isSequenceObject(String id, int ref) {
        SequenceContext sc = this.sequenceContext.getLastOp();
        if (sc != null) {
            return sc.containsRef(id, ref);
        }
        return false;
    }

    AnnotationInfo getAnnotation(MethodInfo mi, String aiName) {
        for (MethodInfo m = mi; m != null; m = m.getOverriddenMethodInfo()) {
            AnnotationInfo ai = m.getAnnotation(aiName);
            if (ai == null) continue;
            return ai;
        }
        return null;
    }

    AnnotationInfo getAnnotation(FieldInfo fi, String aiName) {
        return fi.getAnnotation(aiName);
    }

    String getEventName(SequenceInfo si, MethodInfo mi, String annotationName) {
        return mi.getName();
    }

    protected void processSequence(String id) {
        this.processor.processSequence(id, this.sequenceOps);
    }

    protected void processAllOpenSequences() {
    }

    public void sequenceProcessed(String msg) {
        this.processedSequences.add(msg);
    }

    @Override
    public void instructionExecuted(JVM vm) {
        Instruction insn = vm.getLastInstruction();
        ThreadInfo ti = vm.getLastThreadInfo();
        if (insn instanceof InvokeInstruction) {
            this.processInvokeInstruction(ti, (InvokeInstruction)insn);
        } else if (insn instanceof ASTORE) {
            this.processASTORE(ti, (ASTORE)insn);
        } else if (insn instanceof PUTFIELD) {
            this.processPUTFIELD(ti, (PUTFIELD)insn);
        } else if (!(insn instanceof PUTSTATIC) && insn instanceof ReturnInstruction) {
            this.processReturnInstruction(ti, (ReturnInstruction)insn);
        }
    }

    void processInvokeInstruction(ThreadInfo ti, InvokeInstruction insn) {
        if (insn.isCompleted(ti) && !ti.isInstructionSkipped()) {
            MethodInfo mi = insn.getInvokedMethod();
            AnnotationInfo ai = this.getAnnotation(mi, "gov.nasa.jpf.Sequence");
            if (ai != null) {
                this.updateSequenceScope(ai, ti, mi, insn);
            }
            if ((ai = this.getAnnotation(mi, "gov.nasa.jpf.SequenceMethod")) != null) {
                this.emitSequenceOp(ai, ti, mi, insn);
            }
        }
    }

    void updateSequenceScope(AnnotationInfo ai, ThreadInfo ti, MethodInfo mi, InvokeInstruction insn) {
        String id = ai.getValueAsString("id");
        String[] objects = ai.getValueAsStringArray("objects");
        SequenceContext sc = this.sequenceContext.getLastOp();
        SequenceInfo si = null;
        int idx = sc.getIdx(id);
        if (idx >= 0) {
            si = sc.get(idx);
            si = si.activate(objects, insn, ti);
            this.sequenceContext.addOp(sc.replaceSequence(idx, si));
        } else {
            si = new SequenceInfo(id, objects, insn, ti);
            this.sequenceContext.addOp(sc.addSequence(si));
        }
        SequenceOp.Start ss = new SequenceOp.Start(id);
        this.sequenceOps.addOp(ss);
    }

    int getThis(ThreadInfo ti, Instruction insn) {
        if (insn instanceof ReturnInstruction) {
            return ((ReturnInstruction)insn).getReturnFrame().getThis();
        }
        return ti.getTopFrame().getThis();
    }

    void emitSequenceOp(AnnotationInfo ai, ThreadInfo ti, MethodInfo mi, Instruction insn) {
        int refTgt;
        String id = ai.getValueAsString("id");
        SequenceContext sc = this.sequenceContext.getLastOp();
        SequenceInfo si = sc.get(id);
        if (si != null && si.isActive() && !mi.isStatic() && sc.containsRef(id, refTgt = this.getThis(ti, insn))) {
            int refSrc = this.getEventSource(ti, si);
            String e = this.getEventName(si, mi, ai.getValueAsString("name"));
            String src = si.getObjectName(refSrc);
            String tgt = si.getObjectName(refTgt);
            String ret = ai.getValueAsString("result");
            SequenceOp.Event so = null;
            if (insn instanceof InvokeInstruction) {
                so = new SequenceOp.Enter(id, e, src, tgt, ret);
            } else if (insn instanceof ReturnInstruction) {
                so = new SequenceOp.Exit(id, e, src, tgt, ret);
            }
            this.sequenceOps.addOp(so);
        }
    }

    void processASTORE(ThreadInfo ti, ASTORE insn) {
        MethodInfo mi = insn.getMethodInfo();
        AnnotationInfo ai = this.getAnnotation(mi, "gov.nasa.jpf.Sequence");
        if (ai != null) {
            String id = ai.getValueAsString("id");
            SequenceContext sc = this.sequenceContext.getLastOp();
            int idx = sc.getIdx(id);
            if (idx >= 0) {
                String vn = insn.getLocalVariableName();
                SequenceInfo si = sc.get(idx);
                if (si.containsVar(vn)) {
                    int r = ti.getLocalVariable(insn.getLocalVariableIndex());
                    si = si.setLocalVarRef(vn, r);
                    this.sequenceContext.addOp(sc.replaceSequence(idx, si));
                }
            }
        }
    }

    void processPUTFIELD(ThreadInfo ti, PUTFIELD insn) {
        AnnotationInfo ai;
        FieldInfo fi = insn.getFieldInfo();
        if (fi.isReference() && (ai = this.getAnnotation(fi, "gov.nasa.jpf.SequenceObject")) != null) {
            String id = ai.getValueAsString("id");
            String name = ai.getValueAsString("object");
            if (name.equals("<field>")) {
                name = fi.getName();
            }
            int ref = (int)insn.getLastValue();
            SequenceContext sc = this.sequenceContext.getLastOp();
            int idx = sc.getIdx(id);
            if (idx >= 0) {
                SequenceInfo si = sc.get(idx);
                si = si.addRef(name, ref);
                this.sequenceContext.addOp(sc.replaceSequence(idx, si));
            } else {
                SequenceInfo si = new SequenceInfo(id, name, ref);
                this.sequenceContext.addOp(sc.addSequence(si));
            }
        }
    }

    void processReturnInstruction(ThreadInfo ti, ReturnInstruction insn) {
        MethodInfo mi = insn.getMethodInfo();
        AnnotationInfo ai = this.getAnnotation(mi, "gov.nasa.jpf.Sequence");
        if (ai != null) {
            String id = ai.getValueAsString("id");
            SequenceContext sc = this.sequenceContext.getLastOp();
            int idx = sc.getIdx(id);
            if (idx >= 0) {
                SequenceInfo si = sc.get(idx);
                si = si.setActive(false);
                this.sequenceContext.addOp(sc.replaceSequence(idx, si));
                SequenceOp.End se = new SequenceOp.End(id);
                this.sequenceOps.addOp(se);
                this.processSequence(id);
            }
        }
        if ((ai = this.getAnnotation(mi, "gov.nasa.jpf.SequenceMethod")) != null) {
            this.emitSequenceOp(ai, ti, mi, insn);
        }
    }

    int getEventSource(ThreadInfo ti, SequenceInfo si) {
        int n = ti.getStackDepth();
        for (int i = n - 2; i >= 0; --i) {
            int thisRef;
            StackFrame frame = ti.getStackFrame(i);
            MethodInfo mi = frame.getMethodInfo();
            if (mi.isStatic() || !si.containsRef(thisRef = frame.getThis())) continue;
            return thisRef;
        }
        return -1;
    }

    @Override
    public void searchStarted(Search search) {
        this.sequenceContext.addOp(new SequenceContext(null));
    }

    @Override
    public void searchFinished(Search search) {
    }

    @Override
    public void stateAdvanced(Search search) {
        this.sequenceOps.stateAdvanced(search);
        this.sequenceContext.stateAdvanced(search);
        if (search.isEndState()) {
            this.processAllOpenSequences();
        }
    }

    @Override
    public void stateBacktracked(Search search) {
        this.sequenceOps.stateBacktracked(search);
        this.sequenceContext.stateBacktracked(search);
    }

    @Override
    public void stateStored(Search search) {
        this.sequenceOps.stateStored(search);
        this.sequenceContext.stateBacktracked(search);
    }

    @Override
    public void stateRestored(Search search) {
        this.sequenceOps.stateRestored(search);
        this.sequenceContext.stateBacktracked(search);
    }

    @Override
    public void publishFinished(Publisher publisher) {
        PrintWriter out = publisher.getOut();
        publisher.publishTopicStart("sequence analyzer");
        out.println("number of sequences processed: " + this.processedSequences.size());
        for (String msg : this.processedSequences) {
            out.print('\t');
            out.println(msg);
        }
    }

    static class SequenceContext {
        SequenceInfo[] list;

        SequenceContext(SequenceInfo[] list) {
            this.list = list;
        }

        int getIdx(String id) {
            if (this.list != null) {
                for (int i = 0; i < this.list.length; ++i) {
                    if (!this.list[i].getId().equals(id)) continue;
                    return i;
                }
            }
            return -1;
        }

        SequenceInfo get(int idx) {
            return this.list[idx];
        }

        SequenceInfo get(String id) {
            int idx = this.getIdx(id);
            if (idx >= 0) {
                return this.list[idx];
            }
            return null;
        }

        SequenceContext addSequence(SequenceInfo seq) {
            if (this.list == null) {
                SequenceInfo[] l = new SequenceInfo[]{seq};
                return new SequenceContext(l);
            }
            if (this.getIdx(seq.getId()) < 0) {
                SequenceInfo[] l = new SequenceInfo[this.list.length + 1];
                System.arraycopy(this.list, 0, l, 0, this.list.length);
                l[this.list.length] = seq;
                return new SequenceContext(l);
            }
            return this;
        }

        SequenceContext removeSequence(SequenceInfo seq) {
            int i;
            if (this.list != null && (i = this.getIdx(seq.getId())) >= 0) {
                SequenceInfo[] l = new SequenceInfo[this.list.length - 1];
                System.arraycopy(this.list, 0, l, 0, i);
                int i1 = i + 1;
                System.arraycopy(this.list, i1, l, i1, this.list.length - i1);
                return new SequenceContext(l);
            }
            return this;
        }

        SequenceContext replaceSequence(int idx, SequenceInfo seq) {
            if (this.list != null && idx >= 0 && idx < this.list.length) {
                SequenceInfo[] l = (SequenceInfo[])this.list.clone();
                l[idx] = seq;
                return new SequenceContext(l);
            }
            return this;
        }

        boolean containsRef(String id, int ref) {
            int i;
            if (this.list != null && (i = this.getIdx(id)) >= 0) {
                return this.list[i].containsRef(ref);
            }
            return false;
        }
    }

    static class SequenceInfo
    implements Cloneable {
        String id;
        boolean isActive;
        String[] objects;
        int[] objRefs;
        String[] localVars;

        void parseObjectSpecs(String[] objSpecs) {
            ArrayList<String[]> newEntries = new ArrayList<String[]>();
            for (int i = 0; i < objSpecs.length; ++i) {
                String name;
                String os = objSpecs[i];
                String var = null;
                String[] e = os.split(" *[:=] *");
                if (e.length == 1) {
                    name = e[0];
                } else if (e.length == 2) {
                    name = e[0];
                    var = e[1];
                } else {
                    throw new Exception("illegal object spec: " + os);
                }
                if (this.objects != null) {
                    for (int j = 0; j < this.objects.length; ++j) {
                        if (!this.objects[j].equals(name) || var == null) continue;
                        throw new Exception("object variable already defined: " + os);
                    }
                }
                newEntries.add(e);
            }
            if (!newEntries.isEmpty()) {
                String[] newObjects = new String[objSpecs.length];
                int[] newRefs = new int[objSpecs.length];
                String[] newLocalVars = new String[newObjects.length];
                int j = 0;
                if (this.objects != null) {
                    for (j = 0; j < this.objects.length; ++j) {
                        newObjects[j] = this.objects[j];
                        newRefs[j] = this.objRefs[j];
                    }
                }
                for (String[] e : newEntries) {
                    newObjects[j] = e[0];
                    if (e.length > 1) {
                        newLocalVars[j] = e[1];
                    }
                    ++j;
                }
                this.objects = newObjects;
                this.objRefs = newRefs;
                this.localVars = newLocalVars;
            }
        }

        public Object clone() {
            try {
                return super.clone();
            }
            catch (CloneNotSupportedException x) {
                return null;
            }
        }

        void resolveReferences(InvokeInstruction call, ThreadInfo ti) {
            if (this.localVars != null) {
                for (int i = 0; i < this.localVars.length; ++i) {
                    Object v;
                    String vn = this.localVars[i];
                    if (vn == null || (v = call.getFieldOrArgumentValue(vn, ti)) == null) continue;
                    if (v instanceof ElementInfo) {
                        this.objRefs[i] = ((ElementInfo)v).getIndex();
                        continue;
                    }
                    throw new Exception("field or argument not a reference: " + vn);
                }
            }
        }

        SequenceInfo(String id, String[] objSpecs, InvokeInstruction call, ThreadInfo ti) {
            this.id = id;
            this.init(objSpecs, call, ti);
        }

        public SequenceInfo(String id, String name, int ref) {
            this.id = id;
            this.isActive = false;
            this.objects = new String[1];
            this.objects[0] = name;
            this.objRefs = new int[1];
            this.objRefs[0] = ref;
        }

        String getId() {
            return this.id;
        }

        boolean isActive() {
            return this.isActive;
        }

        void init(String[] objSpecs, InvokeInstruction call, ThreadInfo ti) {
            this.isActive = true;
            this.parseObjectSpecs(objSpecs);
            this.resolveReferences(call, ti);
        }

        String getObjectName(int ref) {
            if (this.objRefs != null) {
                for (int i = 0; i < this.objRefs.length; ++i) {
                    if (this.objRefs[i] != ref) continue;
                    return this.objects[i];
                }
            }
            return null;
        }

        SequenceInfo setActive(boolean b) {
            if (b != this.isActive) {
                SequenceInfo si = (SequenceInfo)this.clone();
                si.isActive = b;
                return si;
            }
            return this;
        }

        SequenceInfo activate(String[] objSpecs, InvokeInstruction call, ThreadInfo ti) {
            SequenceInfo si = (SequenceInfo)this.clone();
            this.init(objSpecs, call, ti);
            return si;
        }

        SequenceInfo addRef(String name, int ref) {
            String[] vn = null;
            int[] or = null;
            if (this.objRefs != null) {
                for (int i = 0; i < this.objRefs.length; ++i) {
                    if (this.objRefs[i] != ref) continue;
                    return this;
                }
                or = new int[this.objRefs.length + 1];
                System.arraycopy(this.objRefs, 0, or, 0, this.objRefs.length);
                or[this.objRefs.length] = ref;
                vn = new String[or.length];
                System.arraycopy(this.objects, 0, vn, 0, this.objects.length);
                vn[this.objects.length] = name;
            } else {
                or = new int[]{ref};
                vn = new String[]{name};
            }
            SequenceInfo si = (SequenceInfo)this.clone();
            si.objects = vn;
            si.objRefs = or;
            return si;
        }

        SequenceInfo setLocalVarRef(String name, int ref) {
            if (this.localVars != null) {
                for (int i = 0; i < this.localVars.length; ++i) {
                    if (!this.localVars[i].equals(name)) continue;
                    SequenceInfo si = (SequenceInfo)this.clone();
                    si.objRefs = (int[])this.objRefs.clone();
                    si.objRefs[i] = ref;
                    return si;
                }
            }
            throw new Exception("trying to set reference for unknown object: " + name);
        }

        boolean containsRef(int ref) {
            if (this.objRefs != null) {
                for (int i = 0; i < this.objRefs.length; ++i) {
                    if (this.objRefs[i] != ref) continue;
                    return true;
                }
            }
            return false;
        }

        boolean containsVar(String varName) {
            if (this.localVars != null) {
                for (int i = 0; i < this.localVars.length; ++i) {
                    if (!this.localVars[i].equals(varName)) continue;
                    return true;
                }
            }
            return false;
        }
    }

    class DefaultProcessor
    extends SequenceProcessor {
        int level;
        int nSequences = 0;
        String fname;
        PrintWriter pw;

        DefaultProcessor() {
        }

        @Override
        public void processSequence(String id, Trace<SequenceOp> trace) {
            this.level = 0;
            this.fname = SequenceAnalyzer.this.config.getString("sequence.out", SequenceAnalyzer.this.config.getTargetArg() + '-' + this.nSequences + ".seq");
            try {
                this.pw = new PrintWriter(this.fname);
            }
            catch (IOException iox) {
                this.pw = new PrintWriter(System.out);
            }
            super.processSequence(id, trace);
            this.pw.close();
            ++this.nSequences;
            SequenceAnalyzer.this.sequenceProcessed("generated sequence file: " + this.fname);
        }

        @Override
        public void visit(SequenceOp.Enter s) {
            String res = s.getResult();
            if (res == null || res.length() == 0) {
                res = "' '";
            }
            this.indent();
            this.pw.println(s.getTgt() + '.' + s.getScope() + " -> " + res + " {");
            ++this.level;
        }

        @Override
        public void visit(SequenceOp.Exit s) {
            --this.level;
            this.indent();
            this.pw.println("}");
        }

        void indent() {
            for (int i = 0; i < this.level; ++i) {
                this.pw.print("  ");
            }
        }
    }

    public static class Exception
    extends RuntimeException {
        public Exception(String msg) {
            super(msg);
        }
    }
}

