/*
 * Decompiled with CFR 0.152.
 */
package gov.nasa.jpf.tools;

import gov.nasa.jpf.Config;
import gov.nasa.jpf.JPF;
import gov.nasa.jpf.PropertyListenerAdapter;
import gov.nasa.jpf.jvm.JVM;
import gov.nasa.jpf.jvm.Path;
import gov.nasa.jpf.jvm.Transition;
import gov.nasa.jpf.report.ConsolePublisher;
import gov.nasa.jpf.report.Publisher;
import gov.nasa.jpf.search.Search;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.io.PrintWriter;
import java.io.StringReader;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.logging.Logger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;

public class PathOutputMonitor
extends PropertyListenerAdapter {
    static final String SEPARATOR = "~~~~~";
    static final String ELLIPSIS = ". . .";
    static Logger log = JPF.getLogger("gov.nasa.jpf.tools.PathOutputMonitor");
    JVM vm;
    List<String[]> pathOutputs = new ArrayList<String[]>();
    Class<? extends PathOutputSpec> psClass;
    boolean printOutput;
    boolean deferOutput;
    List<PathOutputSpec> anySpecs;
    List<PathOutputSpec> allSpecs;
    List<PathOutputSpec> noneSpecs;
    String errorMsg;
    List<PathOutputSpec> violatedSpecs;
    String[] offendingOutput;

    public PathOutputMonitor(Config config, JPF jpf) {
        this.vm = jpf.getVM();
        this.vm.storePathOutput();
        jpf.addPublisherExtension(ConsolePublisher.class, this);
        this.printOutput = config.getBoolean("pom.print_output", true);
        this.deferOutput = config.getBoolean("pom.defer_output", true);
        try {
            this.psClass = config.getClass("pom.output_spec.class", PathOutputSpec.class);
        }
        catch (Config.Exception cx) {
            log.warning("error initializing pom.output_spec.class: " + cx.getMessage());
        }
        if (this.psClass == null) {
            this.psClass = RegexOutputSpec.class;
        }
        this.anySpecs = this.loadSpecs(config, "pom.any");
        this.allSpecs = this.loadSpecs(config, "pom.all");
        this.noneSpecs = this.loadSpecs(config, "pom.none");
        this.violatedSpecs = new ArrayList<PathOutputSpec>();
    }

    List<PathOutputSpec> loadSpecs(Config conf, String key) {
        String fname = conf.getString(key);
        if (fname != null) {
            File file = new File(fname);
            if (file.exists()) {
                return this.readPathPatterns(file);
            }
            log.warning("pattern file not found: " + fname);
        }
        return null;
    }

    PathOutputSpec createPathOutputSpec() {
        try {
            return this.psClass.newInstance();
        }
        catch (Throwable t) {
            log.severe("cannot instantiate PathoutputSpec class: " + t.getMessage());
            return null;
        }
    }

    List<PathOutputSpec> readPathPatterns(File f) {
        ArrayList<PathOutputSpec> results = new ArrayList<PathOutputSpec>();
        try {
            FileReader fr = new FileReader(f);
            BufferedReader br = new BufferedReader(fr);
            PathOutputSpec ps = this.createPathOutputSpec();
            String line = br.readLine();
            while (true) {
                if (line == null) break;
                if (line.startsWith(SEPARATOR)) {
                    results.add(ps);
                    ps = this.createPathOutputSpec();
                } else {
                    ps.add(line);
                }
                line = br.readLine();
            }
            results.add(ps);
            br.close();
        }
        catch (FileNotFoundException fnfx) {
            return null;
        }
        catch (IOException e) {
            e.printStackTrace();
        }
        return results;
    }

    String[] getLines(String output) {
        ArrayList<String> lines = new ArrayList<String>();
        BufferedReader br = new BufferedReader(new StringReader(output));
        try {
            String line = br.readLine();
            while (line != null) {
                lines.add(line);
                line = br.readLine();
            }
        }
        catch (IOException iox) {
            iox.printStackTrace();
        }
        return lines.toArray(new String[lines.size()]);
    }

    boolean matchesAny(List<PathOutputSpec> outputSpecs, String[] lines) {
        for (PathOutputSpec ps : outputSpecs) {
            if (!ps.matches(lines)) continue;
            return true;
        }
        this.errorMsg = "unmatched output";
        this.offendingOutput = lines;
        return false;
    }

    boolean matchesNone(List<PathOutputSpec> outputSpecs, String[] lines) {
        for (PathOutputSpec ps : outputSpecs) {
            if (!ps.matches(lines)) continue;
            this.errorMsg = "illegal output (matching inverse spec)";
            this.offendingOutput = lines;
            this.violatedSpecs.add(ps);
            return false;
        }
        return true;
    }

    boolean matchesAll(List<PathOutputSpec> outputSpecs, List<String[]> outputs) {
        HashSet<PathOutputSpec> unmatched = new HashSet<PathOutputSpec>();
        unmatched.addAll(outputSpecs);
        for (String[] lines : outputs) {
            for (PathOutputSpec ps : outputSpecs) {
                if (!ps.matches(lines)) continue;
                unmatched.remove(ps);
                if (!unmatched.isEmpty()) continue;
                return true;
            }
        }
        this.errorMsg = "unmatched specs (" + unmatched.size() + ')';
        for (PathOutputSpec ps : unmatched) {
            this.violatedSpecs.add(ps);
        }
        return false;
    }

    @Override
    public boolean check(Search search, JVM vm) {
        return this.errorMsg == null;
    }

    @Override
    public String getErrorMessage() {
        StringWriter sw = new StringWriter();
        PrintWriter pw = new PrintWriter(sw);
        pw.println(this.errorMsg);
        if (this.offendingOutput != null) {
            pw.println("offending output:");
            for (String line : this.offendingOutput) {
                pw.println(line);
            }
        }
        if (!this.violatedSpecs.isEmpty()) {
            pw.println("violated specs:");
            for (PathOutputSpec ps : this.violatedSpecs) {
                ps.printOn(pw);
                pw.println(SEPARATOR);
            }
        }
        String s = sw.toString();
        pw.close();
        return s;
    }

    @Override
    public void reset() {
        this.errorMsg = null;
        this.violatedSpecs.clear();
        this.offendingOutput = null;
    }

    @Override
    public void stateAdvanced(Search search) {
        Path path;
        if (search.isEndState() && (path = this.vm.getPath()).hasOutput()) {
            StringBuilder sb = null;
            if (this.deferOutput || this.noneSpecs != null) {
                sb = new StringBuilder();
                for (Transition t : path) {
                    String s = t.getOutput();
                    if (s == null) continue;
                    sb.append(s);
                }
            }
            String[] lines = this.getLines(sb.toString());
            if (this.deferOutput) {
                this.pathOutputs.add(lines);
            } else if (this.printOutput) {
                for (Transition t : path) {
                    String s = t.getOutput();
                    if (s == null) continue;
                    System.out.print(s);
                }
            }
            if (this.noneSpecs != null && !this.matchesNone(this.noneSpecs, lines)) {
                log.warning("pom.none violated");
            }
            if (this.anySpecs != null && !this.matchesAny(this.anySpecs, lines)) {
                log.warning("pom.any violated");
            }
        }
    }

    @Override
    public void searchFinished(Search search) {
        if (this.allSpecs != null && !this.matchesAll(this.allSpecs, this.pathOutputs)) {
            log.warning("pom.all violated");
            search.error(this);
        }
    }

    @Override
    public void publishFinished(Publisher publisher) {
        if (this.printOutput) {
            PrintWriter pw = publisher.getOut();
            publisher.publishTopicStart("path outputs");
            for (String[] output : this.pathOutputs) {
                for (String line : output) {
                    pw.println(line);
                }
                pw.println(SEPARATOR);
            }
        }
    }

    static class RegexOutputSpec
    implements PathOutputSpec {
        ArrayList<Pattern> patterns = new ArrayList();

        RegexOutputSpec() {
        }

        @Override
        public boolean add(String spec) {
            try {
                Pattern p = Pattern.compile(spec);
                this.patterns.add(p);
            }
            catch (PatternSyntaxException psx) {
                return false;
            }
            return true;
        }

        @Override
        public boolean matches(String[] output) {
            if (output != null && output.length > 0) {
                Iterator<Pattern> it = this.patterns.iterator();
                for (String line : output) {
                    if (it.hasNext()) {
                        Pattern p = it.next();
                        if (p.toString().equals(PathOutputMonitor.ELLIPSIS)) {
                            return true;
                        }
                        Matcher m = p.matcher(line);
                        if (m.matches()) continue;
                        return false;
                    }
                    return false;
                }
                return !it.hasNext() || it.next().toString().equals(PathOutputMonitor.ELLIPSIS);
            }
            return this.patterns.isEmpty();
        }

        @Override
        public void printOn(PrintWriter pw) {
            for (Pattern p : this.patterns) {
                pw.println(p.toString());
            }
        }
    }

    public static interface PathOutputSpec {
        public boolean add(String var1);

        public boolean matches(String[] var1);

        public void printOn(PrintWriter var1);
    }
}

