package main;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Vector;
import java.util.Map.Entry;

public class CThrModel {

	private Vector<ClassID> classNames;
	private Vector<MethodID> methodNames;
	private Vector<FieldID> fieldNames;
	private HashMap<FullMethodID, CircusAction[]> methods = new HashMap<FullMethodID, CircusAction[]>();
	private HashMap<FullMethodID, CircusAction> executeMethodActions = new HashMap<FullMethodID, CircusAction>();
	
	public CThrModel(
			Vector<ClassID> classNames,
			Vector<MethodID> methodNames,
			Vector<FieldID> fieldNames) {
		this.classNames = classNames;
		this.methodNames = methodNames;
		this.fieldNames = fieldNames;
	}
	
	public void addMethod(FullMethodID name, CircusAction[] actions, CircusAction executeMethodAction) {
		if (methods.containsKey(name)) {
			throw new IllegalArgumentException("This method has already been added: " + name);
		}
		
		methods.put(name, actions);
		executeMethodActions.put(name, executeMethodAction);
	}
	
	public boolean hasMethod(FullMethodID name) {
		return methods.containsKey(name);
	}
	
	public String toModelString() {
		StringBuilder buffer = new StringBuilder(1024);
	
		buffer.append("\\begin{axdef}\n");
		Iterator<ClassID> classIDIterator = classNames.iterator();
		while (classIDIterator.hasNext()) {
			buffer.append("\t" + classIDIterator.next().toModelString() + "ID : ClassID");
			if (classIDIterator.hasNext()) {
				buffer.append(" \\\\\n");
			} else {
				buffer.append("\n");
			}
		}
		buffer.append("\\end{axdef}\n\n");
		
		buffer.append("\\begin{axdef}\n");
		Iterator<MethodID> methodIDIterator = methodNames.iterator();
		while (methodIDIterator.hasNext()) {
			buffer.append("\t" + methodIDIterator.next().toModelString() + " : MethodID");
			if (methodIDIterator.hasNext()) {
				buffer.append(" \\\\\n");
			} else {
				buffer.append("\n");
			}
		}
		buffer.append("\\end{axdef}\n\n");
		
		buffer.append("\\begin{axdef}\n");
		Iterator<FieldID> fieldIDIterator = fieldNames.iterator();
		while (fieldIDIterator.hasNext()) {
			buffer.append("\t" + fieldIDIterator.next() + " : FieldID");
			if (fieldIDIterator.hasNext()) {
				buffer.append(" \\\\\n");
			} else {
				buffer.append("\n");
			}
		}
		buffer.append("\\end{axdef}\n\n");
		
		buffer.append("\\begin{circus}\n");
		buffer.append("\t\\circprocess CThr \\circdef \\\\\n");
		buffer.append("\t\\t1 thread : ThreadID \\circspot \\\\\n");
		buffer.append("\t\\t1 \\circbegin\n");
		buffer.append("\\end{circus}\n\n");
		
		buffer.append("\\begin{circusaction}\n");
		buffer.append("\tAddLong \\circdef \\circvres msb1 : Word; \\circvres lsb1 : Word; \\circval msb2 : Word; \\circval lsb2 : Word \\circspot \\\\\n");
		buffer.append("\t\t\\t1 \\circvar lsbTmp : Word \\circspot \\\\\n");
		buffer.append("\t\t\\t2 lsbTmp := lsb1 + lsb2 \\circseq \\\\\n");
		buffer.append("\t\t\\t2 \\circif lsbTmp > " + Integer.MAX_VALUE + "\\circthen lsb1 := " + Integer.MAX_VALUE + "\\circseq lsbTmp := lsbTmp - lsb1");
		buffer.append("\t\t\\t2 {} \\circelse lsbTmp < " + Integer.MIN_VALUE + "\\circthen lsb1 := " + Integer.MIN_VALUE + "\\circseq lsbTmp := lsbTmp - lsb1");
		buffer.append("\t\t\\t2 {} \\circelse (lsbTmp \\leq " + Integer.MAX_VALUE + ") \\land (lsbTmp \\geq " + Integer.MIN_VALUE + ") \\circthen lsb1 := lsbTmp \\circseq lsbTmp := 0 \\\\\n");
		buffer.append("\t\t\\t2 \\circfi \\circseq \\\\\n");
		buffer.append("\t\t\\t2 msb1 := msb1 + msb2 + lsbTmp \\\\\n");
		buffer.append("\\end{circusaction}\n\n");
		
		for (Entry<FullMethodID, CircusAction[]> method : methods.entrySet()) {
			buffer.append("\\begin{circusaction}\n");
			buffer.append("\t" + method.getKey() + " \\circdef \\\\\n");
			for (int i = 0; i < method.getValue().length; i++) {
				CircusAction action = method.getValue()[i];
				if (i < method.getValue().length-1) {
					buffer.append("\t\t\\t1 " + action.toModelString(1) + " \\circseq \\\\\n");
				} else {
					buffer.append("\t\t\\t1 " + action.toModelString(1) + " \n");
				}
			}
			buffer.append("\\end{circusaction}\n\n");
		}
		
		buffer.append("\\begin{circusaction}\n");
		buffer.append("\tMainThread \\circdef setStack?t \\prefixcolon (t = thread) ?stack \\then \\circmu X \\circspot \\\\\n");
		buffer.append("\t\t\\t1 \\circblockbegin \n");
		buffer.append("\t\t\\circvar retVal : Word \\circspot executeMethod?t \\prefixcolon (t = thread) ?c?m?a \\then {} \\\\\n");
		buffer.append("\t\t\\t1 ExecuteMethod(c,m,a,retVal) \\circseq executeMethodRet!thread!retVal \\then Poll \\circseq X \\\\\n");
		buffer.append("\t\t{} \\extchoice {} \\\\\n");
		buffer.append("\t\tCEEswitchThread?from?to \\prefixcolon (from = thread) \\then Blocked \\circseq X\n");
		buffer.append("\t\t\\circblockend\n");
		buffer.append("\\end{circusaction}\n\n");
	
		buffer.append("\\begin{circusaction}\n");
		buffer.append("\tBlocked \\circdef CEEswitchThread?from?to \\prefixcolon (to = thread) \\then \\Skip\n");
		buffer.append("\\end{circusaction}\n\n");
	
		buffer.append("\\begin{circusaction}\n");
		buffer.append("\tPoll \\circdef \\\\\n");
		buffer.append("\t\t\\t1 CEEswitchThread?from?to \\prefixcolon (from = thread) \\then Blocked \\circseq Poll \\\\\n");
		buffer.append("\t\t\\t1 {} \\extchoice {} \\\\\n");
		buffer.append("\t\t\\t1 CEEproceed?toProceed \\prefixcolon (toProceed = thread) \\then \\Skip\n");
		buffer.append("\\end{circusaction}\n\n");
	
		buffer.append("\\begin{circusaction}\n");
		buffer.append("\tExecuteMethod \\circdef \\\\\n");
		buffer.append("\t\t\\t1 \\circval classID : ClassID; \\circval methodID : MethodID; \\circval methodArgs : \\seq Word; \\circres retVal : Word \\circspot \\\\\n");
		Iterator<FullMethodID> methodNameIterator = methods.keySet().iterator();
		if (methodNameIterator.hasNext()) {
			FullMethodID firstMethod = methodNameIterator.next();
			buffer.append("\t\t\\t1 \\circif {(classID, methodID) = (" + firstMethod.classID + "ID,"
					+ firstMethod.methodID + ")} \\circthen {} \\\\\n");
			buffer.append("\t\t\t\\t2 " + executeMethodActions.get(firstMethod).toModelString(2) + " \\\\\n");
			
			while (methodNameIterator.hasNext()) {
				FullMethodID methodName = methodNameIterator.next();
				if (!methodNames.contains(methodName.methodID)) {
					throw new IllegalStateException("Method name not recognised: " + methodName.methodID);
				}
				buffer.append("\t\t\\t1 {} \\circelse {(classID, methodID) = (" + methodName.classID + "ID,"
						+ methodName.methodID + ")} \\circthen {} \\\\\n");
				buffer.append("\t\t\t\\t2 " + executeMethodActions.get(methodName).toModelString(2) + " \\\\\n");
			}
			buffer.append("\t\t\\t1\\circfi\n");
		} else {
			buffer.append("\t\t\\t1 \\Skip\n");
		}		
		buffer.append("\\end{circusaction}\n\n");
		
		buffer.append("\\begin{circusaction}\n");
		buffer.append("\tNotStarted \\circdef \\circvar methodID : MethodID; methodArgs : \\seq Word \\circspot \\\\\n");
		buffer.append("\t\t\\t1 CEEstartThread?toStart?bsid?stack?cid?mid?args \\prefixcolon (toStart = thread) \\\\\n");
		buffer.append("\t\t\\t1 {} \\then addThreadMemory!thread!bsid \\\\\n");
		buffer.append("\t\t\\t1 {} \\then methodID, methodArgs := mid, args \\circseq \\\\\n");
		buffer.append("\t\t\\t1 Blocked \\circseq runThread!thread!(head~methodArgs)!methodID \\then Started\n");
		buffer.append("\\end{circusaction}\n\n");
		
		buffer.append("\\begin{circusaction}\n");
		buffer.append("\tStarted \\circdef \\\\\n");
		buffer.append("\t\t\\t1 \\circblockbegin\n");
		buffer.append("\t\t\\circvar retVal : Word \\circspot executeMethod?t \\prefixcolon (t = thread) ?c?m?a \\then {} \\\\\n");
		buffer.append("\t\t\\t1 ExecuteMethod(c,m,a,retVal) \\circseq executeMethodRet!thread!retVal \\then Poll \\circseq \\circblockbegin\n");
		buffer.append("\t\tcontinue?t \\prefixcolon (t = thread) \\then Started \\\\\n");
		buffer.append("\t\t{} \\extchoice {} \\\\\n");
		buffer.append("\t\tendThread?t \\prefixcolon (t = thread) \\then \\Skip\n");
		buffer.append("\t\t\\circblockend \\\\\n");
		buffer.append("\t\t{} \\extchoice {} \\\\\n");
		buffer.append("\t\tCEEswitchThread?from?to \\prefixcolon (from = thread) \\then Blocked \\circseq Started \\\\\n");
		buffer.append("\t\t{} \\extchoice {} \\\\\n");
		buffer.append("\t\tendThread?t \\prefixcolon (t = thread) \\then \\Skip\n");
		buffer.append("\t\t\\circblockend \\circseq \\\\\n");
		buffer.append("\t\t\\t1 removeThreadMemory!thread \\then SendThread \\\\\n");
		buffer.append("\t\t\\t1 {} \\then CEEswitchThread?from?to \\prefixcolon (from = thread) \\then NotStarted\n");
		buffer.append("\\end{circusaction}\n\n");

		buffer.append("\\begin{circusaction}\n");
		buffer.append("\t\\circspot \\\\\n");
		buffer.append("\t\t\\t1 \\lcircguard thread = main \\rcircguard \\circguard MainThread \\\\\n");
		buffer.append("\t\t\\t1 {} \\extchoice {} \\\\\n");
		buffer.append("\t\t\\t1 \\lcircguard thread \\neq main \\rcircguard \\circguard NotStarted\n");
		buffer.append("\\end{circusaction}\n");
		buffer.append("\\begin{circus}\n");
		buffer.append("\t\\circend");
		buffer.append("\\end{circus}\n\n");
		
		return buffer.toString();
	}
	
	public String toCCode() {
		StringBuilder builder = new StringBuilder();
		
		builder.append("staticClassFieldsStruct * staticClassFields = NULL;\n\n");
		
		builder.append("void addLong(int32_t * msb1, int32_t * lsb1, int32_t msb2, int32_t lsb2);\n\n");
		
		// build function prototypes
		for (Entry<FullMethodID, CircusAction[]> method : methods.entrySet()) {
			//System.out.println("Building C code for " + method.getKey());
//			if (method.getKey().methodID.hasReturnValue()) {
//				builder.append("int32_t ");
//			} else {
//				builder.append("void ");
//			}
			builder.append("void ");
			builder.append(method.getKey());
			builder.append("(");
			
			if (method.getValue()[0] instanceof CircusAction.ParametrisedBlock) {
				CircusAction.ParametrisedBlock paramBlock = (CircusAction.ParametrisedBlock) method.getValue()[0];
				// make returns into pointer values
				int numArgs = paramBlock.getParameterNames().length;
				for (int i = 0; i < numArgs; i++) {
					builder.append("int32_t ");
					if (paramBlock.getParameterIsRet()[i]) {
						builder.append("* ");
					}
					builder.append(paramBlock.getParameterNames()[i]);
					if (i < numArgs-1) {
						builder.append(", ");
					}
				}
				builder.append(");\n");
//				// Assume last parameter will be return, if present
//				int numArgs = paramBlock.getParameterNames().length;
//				if (paramBlock.getParameterIsRet()[numArgs-1]) {
//					numArgs--;
//				}
//				
//				for (int i = 0; i < numArgs; i++) {
//					builder.append("int32_t ");
//					builder.append(paramBlock.getParameterNames()[i]);
//					if (i < numArgs-1) {
//						builder.append(", ");
//					}
//				}
			} else {
				builder.append("void);\n");
			}
		}
		builder.append("\n");
		
		
		builder.append("void addLong(int32_t * msb1, int32_t * lsb1, int32_t msb2, int32_t lsb2) {\n");
		builder.append("\tint64_t lsbTmp;\n");
		builder.append("\tlsbTmp = *lsb1 + lsb2;\n");
		builder.append("\t if(lsbTmp > " + Integer.MAX_VALUE + ") {\n");
		builder.append("\t\t*lsb1 = " + Integer.MAX_VALUE + ";\n");
		builder.append("\t\tlsbTmp = lsbTmp - *lsb1;\n");
		builder.append("\t} else if (lsbTmp < " + Integer.MIN_VALUE + ") {\n");
		builder.append("\t\t*lsb1 = " + Integer.MIN_VALUE + ";\n");
		builder.append("\t\tlsbTmp = lsbTmp - *lsb1;\n");
		builder.append("\t} else {\n");
		builder.append("\t\t*lsb1 = lsbTmp;\n");
		builder.append("\t\tlsbTmp = 0;\n");
		builder.append("\t}\n");
		builder.append("\t\t*msb1 = *msb1 + msb2 + lsbTmp;\n");
		builder.append("}\n\n");
		
		
		for (Entry<FullMethodID, CircusAction[]> method : methods.entrySet()) {
			//System.out.println("Building C code for " + method.getKey());
			builder.append("void ");
			builder.append(method.getKey());
			builder.append("(");
			
			if (method.getValue()[0] instanceof CircusAction.ParametrisedBlock) {
				CircusAction.ParametrisedBlock paramBlock = (CircusAction.ParametrisedBlock) method.getValue()[0];
				int numArgs = paramBlock.getParameterNames().length;
				for (int i = 0; i < numArgs; i++) {
					builder.append("int32_t ");
					if (paramBlock.getParameterIsRet()[i]) {
						builder.append("* ");
					}
					builder.append(paramBlock.getParameterNames()[i]);
					if (i < numArgs-1) {
						builder.append(", ");
					}
				}
				builder.append(") {\n\t");
				builder.append(CircusAction.actionsToCCode(paramBlock.getActions(), 1));
				builder.append("\n}\n\n");
			} else {
				builder.append(") {\n\t");
				builder.append(CircusAction.actionsToCCode(method.getValue(), 1));
				builder.append("\n}\n\n");
			}
			
		}		
		
		return builder.toString().replace("\\_", "_");
	}
}
