package main;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map.Entry;
import java.util.Vector;

import main.CircusAction.Assignment;
import main.CircusAction.ChannelComm;
import main.CircusAction.Conditional;
import main.CircusAction.MethodActionRef;
import main.CircusAction.PoppedArgsAssignment;
import main.CircusAction.VarBlock;

public class ThrCFModel {
	
	private Vector<ClassID> classNames;
	private Vector<MethodID> methodNames;
	private Vector<FieldID> fieldNames;
	private HashMap<ClassID,ClassModel> classes;
	private HashMap<FullMethodID, CircusAction[]> methods = new HashMap<FullMethodID, CircusAction[]>();
	

	public ThrCFModel(
			Vector<ClassID> classNames,
			Vector<MethodID> methodNames,
			Vector<FieldID> fieldNames,
			HashMap<ClassID,ClassModel> classes) {
		this.classNames = classNames;
		this.methodNames = methodNames;
		this.fieldNames = fieldNames;
		this.classes = classes;
	}
	
	public void addMethod(FullMethodID name, CircusAction[] actions) {
		if (methods.containsKey(name)) {
			throw new IllegalArgumentException("This method has already been added: " + name);
		}
		
		methods.put(name, removePCAssignments(actions));
	}
	
	private CircusAction[] removePCAssignments(CircusAction[] oldActions) {
		Vector<CircusAction> newActions = new Vector<CircusAction>();
	
		for (CircusAction action : oldActions) {
			if (action instanceof CircusAction.ChannelComm) {
				CircusAction newAction = removePCAssignments(new CircusAction[]{((CircusAction.ChannelComm) action).getAction()})[0];
				newActions.add(new CircusAction.ChannelComm(
						((CircusAction.ChannelComm) action).getChannelName(),
						((CircusAction.ChannelComm) action).getChannelArgs(),
						((CircusAction.ChannelComm) action).getArgIsInputArray(),
						newAction));
			} else if (action instanceof CircusAction.Conditional) {
				CircusAction[][] oldConditionActions = ((CircusAction.Conditional) action).getConditionActions();
				CircusAction[][] newConditionActions = new CircusAction[oldConditionActions.length][];
				for (int i = 0; i < oldConditionActions.length; i++) {
					newConditionActions[i] = removePCAssignments(oldConditionActions[i]);
				}
				newActions.add(new CircusAction.Conditional(((CircusAction.Conditional) action).getConditions(), newConditionActions));
			} else if (action instanceof CircusAction.PCAssignment) {
				continue;
			} else if (action instanceof CircusAction.VarBlock) {
				newActions.add(new CircusAction.VarBlock(
						((CircusAction.VarBlock) action).getVarName(),
						removePCAssignments(((CircusAction.VarBlock) action).getActions())));
			} else if (action instanceof CircusAction.ConditionalPCAssignment) {
				continue;
			} else if (action instanceof CircusAction.Recursion) {
				newActions.add(new CircusAction.Recursion(
						((CircusAction.Recursion) action).getRecVarName(),
						removePCAssignments(((CircusAction.Recursion) action).getActions())));
			} else if (action instanceof CircusAction.PCAssumption) {
				continue;
			} else {
				// just copy everything else across unchanged
				newActions.add(action);
			}
		}
		
		// make sure the sequence of actions is non-empty
		if (newActions.isEmpty()) {
			newActions.add(new CircusAction.Skip());
		}
		
		return newActions.toArray(new CircusAction[] {});
	}
	
	public boolean hasMethod(FullMethodID name) {
		return methods.containsKey(name);
	}
	
	public String toModelString() {
		StringBuilder buffer = new StringBuilder(1024);
	
		buffer.append("\\begin{axdef}\n");
		Iterator<ClassID> classIDIterator = classNames.iterator();
		while (classIDIterator.hasNext()) {
			buffer.append("\t" + classIDIterator.next().toModelString() + "ID : ClassID");
			if (classIDIterator.hasNext()) {
				buffer.append(" \\\\\n");
			} else {
				buffer.append("\n");
			}
		}
		buffer.append("\\end{axdef}\n\n");
		
		buffer.append("\\begin{axdef}\n");
		Iterator<MethodID> methodIDIterator = methodNames.iterator();
		while (methodIDIterator.hasNext()) {
			buffer.append("\t" + methodIDIterator.next().toModelString() + " : MethodID");
			if (methodIDIterator.hasNext()) {
				buffer.append(" \\\\\n");
			} else {
				buffer.append("\n");
			}
		}
		buffer.append("\\end{axdef}\n\n");
		
		buffer.append("\\begin{zed}\n");
		buffer.append("\tBytecode ::= \\\\\n");
		buffer.append("\t\\t1 aconst\\_null | dup | areturn | return | iadd | ineg \\\\\n");
		buffer.append("\t\\t1 | new \\ldata CPIndex \\rdata | iconst \\ldata \\nat \\rdata \\\\\n");
		buffer.append("\t\\t1 | aload \\ldata \\nat \\rdata | astore \\ldata \\nat \\rdata \\\\\n");
		buffer.append("\t\\t1 | getfield \\ldata CPIndex \\rdata | putfield \\ldata CPIndex \\rdata \\\\\n");
		buffer.append("\t\\t1 | getstatic \\ldata CPIndex \\rdata | putstatic \\ldata CPIndex \\rdata \\\\\n");
		buffer.append("\t\\t1 | invokespecial \\ldata CPIndex \\rdata | invokevirtual \\ldata CPIndex \\rdata \\\\\n");
		buffer.append("\t\\t1 | invokestatic \\ldata CPIndex \\rdata \\\\\n");
		buffer.append("\t\\t1 | if\\_icmple \\ldata ProgramAddress \\rdata | goto \\ldata ProgramAddress \\rdata\n");
		buffer.append("\\end{zed}\n\n");
		
		buffer.append("\\begin{circus}\n");
		buffer.append("\t\\circprocess ThrCF \\circdef \\\\\n");
		buffer.append("\t\\t1 cs : ClassID \\pfun Class; thread : ThreadID \\circspot \\\\\n");
		buffer.append("\t\\t1 \\circbegin\n");
		buffer.append("\\end{circus}\n\n");
		
		buffer.append("\\begin{schema}{InterpreterStateEPC}\n");
		buffer.append("\tframeStack : \\seq StackFrame \\\\\n");
		buffer.append("\tpc : ProgramAddress \\\\\n");
		buffer.append("\tcurrentClass : Class\n");
		buffer.append("\\where\n");
		buffer.append("\tframeStack \\neq \\emptyset \\implies currentClass = (last~frameStack).frameClass \\\\\n");
		buffer.append("\tcurrentClass = (\\mu c : \\ran cs | \\\\\n");
		buffer.append("\t\\t1 (\\exists_1 m : MethodID | m \\in \\dom c.methodEntry @  \\\\\n");
		buffer.append("\t\\t2 pc \\in c.methodEntry~m \\upto c.methodEnd~m)) \\\\\n");
		buffer.append("\t\\forall f : \\ran frameStack @ f.frameClass = (\\mu c :\\ran cs | \\\\\n");
		buffer.append("\t\\t1 (\\exists_1 m : MethodID | m \\in \\dom c.methodEntry @  \\\\\n");
		buffer.append("\t\\t2 f.storedPC \\in c.methodEntry~m \\upto c.methodEnd~m))\n");
		buffer.append("\\end{schema}\n");
		buffer.append("\\begin{circusaction}\n");
		buffer.append("\t\\circstate InterpreterStateEPC\n");
		buffer.append("\\end{circusaction}\n\n");
		
		buffer.append("\\begin{schema}{InterpreterInit}\n");
		buffer.append("\tInterpreterStateEPC~'\n");
		buffer.append("\\where\n");
		buffer.append("\tframeStack' = \\langle\\rangle\n");
		buffer.append("\\end{schema}\n\n");
		
		buffer.append("\\begin{schema}{InterpreterNewStackFrame}\n");
		buffer.append("\t\\Delta InterpreterStateEPC \\\\\n"); 
		buffer.append("\tclass? : Class \\\\\n");
		buffer.append("\tmethodID? : MethodID \\\\\n");
		buffer.append("\tmethodArgs? : \\seq Word\n");
		buffer.append("\\where\n");
		buffer.append("\t\\exists numLocals? : \\nat | numLocals? = class?.methodLocals~methodID? @ \\\\\n");
		buffer.append("\t\\exists stackSize? : \\nat | stackSize? = class?.methodStackSize~methodID? @ \\\\\n");
		buffer.append("\t\\exists storedPC? : ProgramAddress | storedPC? = class?.methodEntry~methodID? @ \\\\\n");
		buffer.append("\t\\exists StackFrame~' | StackFrameInit[methodArgs?/initLocals?] @ \\\\\n");
		buffer.append("\tframeStack' = frameStack \\cat \\langle \\theta StackFrame~' \\rangle \\land \\\\\n");
		buffer.append("\tcurrentClass' = class?\n");
		buffer.append("\t\\end{schema}\n\n");
		
		buffer.append("\\begin{schema}{InterpreterReturn}\n");
		buffer.append("\t\\Delta InterpreterStateEPC\n");
		buffer.append("\\where\n");
		buffer.append("\tframeStack \\neq \\emptyset \\\\\n");
		buffer.append("\tframeStack' = (front~frameStack) \\\\\n");
		buffer.append("\tframeStack' \\neq \\emptyset \\implies \\\\\n");
		buffer.append("\t\\t1 currentClass' = (last~frameStack').frameClass\n");
		buffer.append("\\end{schema}\n\n");
		
		buffer.append("\\begin{schema}{InterpreterAreturn1}\n");
		buffer.append("\t\\Delta InterpreterStateEPC\n");
		buffer.append("\\where\n");
		buffer.append("\tframeStack \\neq \\emptyset \\land front~frameStack \\neq \\emptyset \\\\\n");
		buffer.append("\t\\exists returnValue : Word @ \\\\\n");
		buffer.append("\t(\\exists \\Delta StackFrame | StackFramePop[returnValue/value!] @ \\\\\n");
		buffer.append("\t\t\\t1 \\theta StackFrame = last~frameStack) \\land \\\\\n");
		buffer.append("\t(\\exists \\Delta StackFrame | StackFramePush[returnValue/value?] @ \\\\\n");
		buffer.append("\t\t\\t1 \\theta StackFrame = last~(front~frameStack) \\land \\\\\n");
		buffer.append("\t\t\\t1 frameStack' = (front~(front~frameStack)) \\cat \\langle \\theta StackFrame~' \\rangle \\land \\\\\n");
		buffer.append("\t\t\\t1 currentClass' = (last~frameStack').frameClass)\n");
		buffer.append("\\end{schema}\n\n");
		
		buffer.append("\\begin{schema}{InterpreterAreturn2}");
		buffer.append("\\Delta InterpreterStateEPC \\\\\n");
		buffer.append("returnValue! : Word \\\\\n");
		buffer.append("\\where\n");
		buffer.append("\tframeStack \\neq \\emptyset \\\\\n");
		buffer.append("\t(\\exists \\Delta StackFrame | StackFramePop[returnValue!/value!] @ \\\\\n");
		buffer.append("\t\t\\t1 \\theta StackFrame = last~frameStack) \\\\\n");
		buffer.append("\tframeStack' = front~frameStack \\\\\n");
		buffer.append("\tframeStack' \\neq \\emptyset \\implies \\\\\n");
		buffer.append("\t\t\\t1 currentClass' = (last~frameStack').frameClass\n");
		buffer.append("\\end{schema}\n\n");
		
		buffer.append("\\begin{zed}\n");
		buffer.append("\tInterpreterAreturn == InterpreterAreturn1 \\lor InterpreterAreturn2\n");
		buffer.append("\\end{zed}\n\n");
		
		buffer.append("\\begin{schema}{PromoteStackFrameOpEPC}\n");
		buffer.append("\t\\Delta InterpreterStateEPC \\\\\n");
		buffer.append("\t\\Delta StackFrame\n");
		buffer.append("\\where\n");
		buffer.append("\t\\theta StackFrame = last~frameStack \\\\\n");
		buffer.append("\tframeStack' = (front~frameStack) \\cat \\langle \\theta StackFrame~' \\rangle \\\\\n");
		buffer.append("\tcurrentClass' = currentClass\n");
		buffer.append("\\end{schema}\n\n");
		
		buffer.append("\\begin{zed}\n");
		buffer.append("\tInterpreterAconst\\_nullEPC == \\\\\n");
		buffer.append("\t\t\\t1 \\exists \\Delta StackFrame @ StackFrameACONST\\_NULL \\land PromoteStackFrameOpEPC \\\\\n");
		buffer.append("\tInterpreterAloadEPC == \\exists \\Delta StackFrame @ StackFrameALOAD \\land PromoteStackFrameOpEPC \\\\\n");
		buffer.append("\tInterpreterAstoreEPC == \\exists \\Delta StackFrame @ StackFrameASTORE \\land PromoteStackFrameOpEPC \\\\\n");
		buffer.append("\tInterpreterDupEPC == \\exists \\Delta StackFrame @ StackFrameDUP \\land PromoteStackFrameOpEPC \\\\\n");
		buffer.append("\tInterpreterPopEPC == \\exists \\Delta StackFrame @ StackFramePop \\land PromoteStackFrameOpEPC \\\\\n");
		buffer.append("\tInterpreterPushEPC == \\exists \\Delta StackFrame @ StackFramePush \\land PromoteStackFrameOpEPC \\\\\n");
		buffer.append("\tInterpreterIaddEPC == \\exists \\Delta StackFrame @ StackFrameIADD \\land PromoteStackFrameOpEPC \\\\\n");
		buffer.append("\tInterpreterInegEPC == \\exists \\Delta StackFrame @ StackFrameINEG \\land PromoteStackFrameOpEPC \\\\\n");
		buffer.append("\tInterpreterStackFrameInvokeEPC == [\\Delta InterpreterStateEPC; argsToPop? : \\nat; poppedArgs! : \\seq Word | \\\\\n");
		buffer.append("\t\t\\t1 \\exists \\Delta StackFrame @ \\exists returnAddress? == pc + 1 @ StackFrameInvoke \\land PromoteStackFrameOpEPC]\n");
		buffer.append("\\end{zed}\n\n");
		
		buffer.append("\\begin{circusaction}\n");
		buffer.append("\tHandleAconst\\_nullEPC \\circdef \\lschexpract InterpreterAconst\\_nullEPC \\rschexpract\n");
		buffer.append("\\end{circusaction}\n\n");
		
		buffer.append("\\begin{circusaction}\n");
		buffer.append("\tHandleDupEPC \\circdef \\lschexpract InterpreterDupEPC \\rschexpract\n");
		buffer.append("\\end{circusaction}\n\n");
	
		buffer.append("\\begin{circusaction}\n");
		buffer.append("\tHandleIaddEPC \\circdef \\lschexpract InterpreterIaddEPC \\rschexpract\n");
		buffer.append("\\end{circusaction}\n\n");
	
		buffer.append("\\begin{circusaction}\n");
		buffer.append("\tHandleInegEPC \\circdef \\lschexpract InterpreterInegEPC \\rschexpract\n");
		buffer.append("\\end{circusaction}\n\n");
		
		buffer.append("\\begin{circusaction}\n");
		buffer.append("\tHandleAloadEPC \\circdef \\circval variableIndex : \\nat \\circspot \\lschexpract InterpreterAloadEPC \\rschexpract\n");
		buffer.append("\\end{circusaction}\n\n");
	
		buffer.append("\\begin{circusaction}\n");
		buffer.append("\tHandleAstoreEPC \\circdef \\circval variableIndex : \\nat \\circspot \\lschexpract InterpreterAstoreEPC \\rschexpract\n");
		buffer.append("\\end{circusaction}\n\n");
		
		buffer.append("\\begin{circusaction}\n");
		buffer.append("\tHandleIconstEPC \\circdef \\circval value : \\nat \\circspot \\lschexpract InterpreterPushEPC \\rschexpract\n");
		buffer.append("\\end{circusaction}\n\n");
		
		buffer.append("\\begin{circusaction}\n");
		buffer.append("\tCheckLauncherReturn \\circdef \\circval returnValue : Word \\circspot \\\\\n");
		buffer.append("\t\t\\t1 \\circif frameStack = \\emptyset \\circthen \\\\\n");
		buffer.append("\t\t\t\\t2 executeMethodRet!thread!returnValue \\then \\Skip \\\\\n");
		buffer.append("\t\t\\t1 {} \\circelse frameStack \\neq \\emptyset \\circthen \\Skip \\\\\n");
		buffer.append("\t\t\\t1 \\circfi\n");
		buffer.append("\\end{circusaction}\n");
		buffer.append("\\begin{circusaction}\n");
		buffer.append("\tHandleAreturnEPC \\circdef \\circvar returnValue : Word \\circspot \\\\\n");
		buffer.append("\t\t\\t1 \\lschexpract InterpreterAreturn \\rschexpract \\circseq CheckLauncherReturn(returnValue)\n");
		buffer.append("\\end{circusaction}\n");
		buffer.append("\\begin{circusaction}\n");
		buffer.append("\tHandleReturn \\circdef \\circvar returnValue : Word \\circspot \\\\\n");
		buffer.append("\t\t\\t1 \\lschexpract InterpreterReturn \\rschexpract \\circseq CheckLauncherReturn(returnValue)\n");
		buffer.append("\\end{circusaction}\n\n");
		
		buffer.append("\\begin{circusaction}\n");
		buffer.append("\tHandleNewEPC \\circdef \\circval cpIndex : \\nat \\circspot \\\\\n");
		buffer.append("\t\t\\t1 \\circif cpIndex \\in classRefIndices~currentClass \\circthen {} \\\\\n");
		buffer.append("\t\t\t\\t2 newObject!thread!(classOf~cpIndex) \\then newObjectRet?oid \\then {} \\\\\n");
		buffer.append("\t\t\t\\t2 \\lschexpract InterpreterPushEPC[oid/value?] \\rschexpract \\\\\n");
		buffer.append("\t\t\\t1 {} \\circelse cpIndex \\notin classRefIndices~currentClass \\circthen \\Chaos \\\\\n");
		buffer.append("\t\\t1 \\circfi\n");
		buffer.append("\\end{circusaction}\n\n");
		
		buffer.append("\\begin{circusaction}\n");
		buffer.append("\tHandleGetfieldEPC \\circdef \\circval cpIndex : \\nat \\circspot \\\\\n");
		buffer.append("\t\t\\t1 \\circif cpIndex \\in fieldRefIndices~currentClass \\circthen {} \\\\\n");
		buffer.append("\t\t\t\\t2 \\circvar oid : ObjectID \\circspot \\lschexpract InterpreterPopEPC[oid!/value!] \\rschexpract \\circseq \\\\\n");
		buffer.append("\t\t\t\\t2 \\circvar fid : FieldID \\circspot fid := fieldOf~currentClass~cpIndex \\circseq \\\\\n");
		buffer.append("\t\t\t\\t2 \\circvar cid : ClassID \\circspot cid := classOf~currentClass~cpIndex \\circseq \\\\\n");
		buffer.append("\t\t\t\\t2 getField!oid!cid!fid \\then getFieldRet?value \\then \\lschexpract InterpreterPushEPC \\rschexpract \\\\\n");
		buffer.append("\t\t\\t1 {} \\circelse cpIndex \\notin fieldRefIndices~currentClass \\circthen \\Chaos \\\\\n");
		buffer.append("\t\t\\t1 \\circfi\n");
		buffer.append("\\end{circusaction}\n\n");
		
		buffer.append("\\begin{circusaction}\n");
		buffer.append("\tHandlePutfieldEPC \\circdef \\circval cpIndex : \\nat \\circspot \\\\\n");
		buffer.append("\t\t\\t1 \\circif cpIndex \\in fieldRefIndices~currentClass \\circthen {} \\\\\n");
		buffer.append("\t\t\t\\t2 \\circvar fid : FieldID \\circspot fid := fieldOf~currentClass~cpIndex \\circseq \\\\\n");
		buffer.append("\t\t\t\\t2 \\circvar cid : ClassID \\circspot cid := classOf~currentClass~cpIndex \\circseq \\\\\n");
		buffer.append("\t\t\t\\t2 \\circvar oid : ObjectID; value : Word \\circspot \\lschexpract InterpreterPopEPC \\rschexpract \\circseq \\lschexpract InterpreterPopEPC[oid!/value!] \\rschexpract \\circseq \\\\\n");
		buffer.append("\t\t\t\\t2 putField!oid!cid!fid!value \\then \\Skip \\\\\n");
		buffer.append("\t\t\\t1 {} \\circelse cpIndex \\notin fieldRefIndices~currentClass \\circthen \\Chaos \\\\\n");
		buffer.append("\t\t\\t1 \\circfi\n");
		buffer.append("\\end{circusaction}\n\n");
		
		buffer.append("\\begin{circusaction}\n");
		buffer.append("\tHandleGetstaticEPC \\circdef \\circval cpIndex : \\nat \\circspot \\\\\n");
		buffer.append("\t\t\\t1 \\circif cpIndex \\in fieldRefIndices~currentClass \\circthen {} \\\\\n");
		buffer.append("\t\t\t\\t2 \\circvar cid : ClassID \\circspot cid := classOf~currentClass~cpIndex \\circseq \\\\\n");
		buffer.append("\t\t\t\\t2 \\circvar fid : FieldID \\circspot fid := fieldOf~currentClass~cpIndex \\circseq \\\\\n");
		buffer.append("\t\t\t\\t2 getStatic!cid!fid \\then getFieldRet?value \\then \\lschexpract InterpreterPushEPC \\rschexpract \\\\\n");
		buffer.append("\t\t\\t1 {} \\circelse cpIndex \\notin fieldRefIndices~currentClass \\circthen \\Chaos \\\\\n");
		buffer.append("\t\t\\t1 \\circfi\n");
		buffer.append("\\end{circusaction}\n\n");
		
		buffer.append("\\begin{circusaction}\n");
		buffer.append("\tHandlePutstaticEPC \\circdef \\circval cpIndex : \\nat \\circspot \\\\\n");
		buffer.append("\t\t\\t1 \\circif cpIndex \\in fieldRefIndices~currentClass \\circthen {} \\\\\n");
		buffer.append("\t\t\t\\t2 \\circvar cid : ClassID \\circspot cid := classOf~currentClass~cpIndex \\circseq \\\\\n");
		buffer.append("\t\t\t\\t2 \\circvar fid : FieldID \\circspot fid := fieldOf~currentClass~cpIndex \\circseq \\\\\n");
		buffer.append("\t\t\t\\t2 \\circvar value : Word \\circspot \\lschexpract InterpreterPopEPC \\rschexpract \\circseq \\\\\n");
		buffer.append("\t\t\t\\t2 putStatic!cid!fid!value \\then \\Skip \\\\\n");
		buffer.append("\t\t\\t1 {} \\circelse cpIndex \\notin fieldRefIndices~currentClass \\circthen \\Chaos \\\\\n");
		buffer.append("\t\t\\t1 \\circfi\n");
		buffer.append("\\end{circusaction}\n\n");
		
		for (Entry<FullMethodID, CircusAction[]> method : methods.entrySet()) {
			buffer.append("\\begin{circusaction}\n");
			buffer.append("\t" + method.getKey() + " \\circdef \\\\\n");
			for (int i = 0; i < method.getValue().length; i++) {
				CircusAction action = method.getValue()[i];
				if (i < method.getValue().length-1) {
					buffer.append("\t\t\\t1 " + action.toModelString(1) + " \\circseq \\\\\n");
				} else {
					buffer.append("\t\t\\t1 " + action.toModelString(1) + " \n");
				}
			}
			buffer.append("\\end{circusaction}\n\n");
		}
		
		buffer.append("\\begin{circusaction}\n");
		buffer.append("\tMainThread \\circdef setStack?t \\prefixcolon (t = thread) ?stack \\then \\circmu X \\circspot \\\\\n");
		buffer.append("\t\t\\t1 \\circblockbegin \n");
		buffer.append("\t\texecuteMethod?t \\prefixcolon (t = thread) ?c?m?a \\then ExecuteMethod(c,m,a) \\circseq X \\\\\n");
		buffer.append("\t\t{} \\extchoice {} \\\\\n");
		buffer.append("\t\tCEEswitchThread?from?to \\prefixcolon (from = thread) \\then Blocked \\circseq X\n");
		buffer.append("\t\t\\circblockend\n");
		buffer.append("\\end{circusaction}\n\n");
	
		buffer.append("\\begin{circusaction}\n");
		buffer.append("\tBlocked \\circdef CEEswitchThread?from?to \\prefixcolon (to = thread) \\then \\Skip\n");
		buffer.append("\\end{circusaction}\n\n");
	
		buffer.append("\\begin{circusaction}\n");
		buffer.append("\tPoll \\circdef \\\\\n");
		buffer.append("\t\t\\t1 CEEswitchThread?from?to \\prefixcolon (from = thread) \\then Blocked \\circseq Poll \\\\\n");
		buffer.append("\t\t\\t1 {} \\extchoice {} \\\\\n");
		buffer.append("\t\t\\t1 CEEproceed?toProceed \\prefixcolon (toProceed = thread) \\then \\Skip\n");
		buffer.append("\\end{circusaction}\n\n");
	
		buffer.append("\\begin{circusaction}\n");
		buffer.append("\tExecuteMethod \\circdef \\\\\n");
		buffer.append("\t\t\\t1\\circval classID : ClassID; \\circval methodID : MethodID; \\circval methodArgs : \\seq Word \\circspot \\\\\n");
		Iterator<FullMethodID> methodNameIterator = methods.keySet().iterator();
		if (methodNameIterator.hasNext()) {
			FullMethodID firstMethod = methodNameIterator.next();
			buffer.append("\t\t\\t1 \\circif {(classID, methodID) = (" + firstMethod.classID + "ClassID,"
					+ firstMethod.methodID + ")} \\circthen {} \\\\\n");
			buffer.append("\t\t\t\\t2 \\lschexpract InterpreterNewStackFrame[" + firstMethod.classID
					+ "/class?, " + firstMethod.methodID + "/methodID?] \\rschexpract \\circseq \\\\\n");
			buffer.append("\t\t\t\\t2 " + firstMethod + " \\\\\n");
			
			while (methodNameIterator.hasNext()) {
				FullMethodID methodName = methodNameIterator.next();
				buffer.append("\t\t\\t1 {} \\circelse {(classID, methodID) = (" + methodName.classID + "ClassID,"
						+ methodName.methodID + ")} \\circthen {} \\\\\n");
				buffer.append("\t\t\t\\t2 \\lschexpract InterpreterNewStackFrame[" + methodName.classID
						+ "/class?, " + methodName.methodID + "/methodID?] \\rschexpract \\circseq \\\\\n");
				buffer.append("\t\t\t\\t2 " + methodName + " \\\\\n");
			}
			buffer.append("\t1\\circfi\n");
		} else {
			buffer.append("\t\t\\t1 \\Skip\n");
		}		
		buffer.append("\\end{circusaction}\n\n");
		
		buffer.append("\\begin{circusaction}\n");
		buffer.append("\tNotStarted \\circdef \\circvar methodID : MethodID; methodArgs : \\seq Word \\circspot \\\\\n");
		buffer.append("\t\t\\t1 CEEstartThread?toStart?bsid?stack?cid?mid?args \\prefixcolon (toStart = thread) \\\\\n");
		buffer.append("\t\t\\t1 {} \\then addThreadMemory!thread!bsid \\\\\n");
		buffer.append("\t\t\\t1 {} \\then methodID, methodArgs := mid, args \\circseq \\\\\n");
		buffer.append("\t\t\\t1 Blocked \\circseq runThread!thread!(head~methodArgs)!methodID \\then Started\n");
		buffer.append("\\end{circusaction}\n\n");
		
		buffer.append("\\begin{circusaction}\n");
		buffer.append("\tStarted \\circdef \\\\\n");
		buffer.append("\t\t\\t1 \\circblockbegin\n");
		buffer.append("\t\texecuteMethod?t \\prefixcolon (t = thread) ?c?m?a \\then ExecuteMethod(c,m,a) \\circseq \\circblockbegin\n");
		buffer.append("\t\tcontinue?t \\prefixcolon (t = thread) \\then Started \\\\\n");
		buffer.append("\t\t{} \\extchoice {} \\\\\n");
		buffer.append("\t\tendThread?t \\prefixcolon (t = thread) \\then \\Skip\n");
		buffer.append("\t\t\\circblockend \\\\\n");
		buffer.append("\t\t{} \\extchoice {} \\\\\n");
		buffer.append("\t\tCEEswitchThread?from?to \\prefixcolon (from = thread) \\then Blocked \\circseq Started \\\\\n");
		buffer.append("\t\t{} \\extchoice {} \\\\\n");
		buffer.append("\t\tendThread?t \\prefixcolon (t = thread) \\then \\Skip\n");
		buffer.append("\t\t\\circblockend \\circseq \\\\\n");
		buffer.append("\t\t\\t1 removeThreadMemory!thread \\then SendThread \\\\\n");
		buffer.append("\t\t\\t1 {} \\then CEEswitchThread?from?to \\prefixcolon (from = thread) \\then NotStarted\n");
		buffer.append("\\end{circusaction}\n\n");

		buffer.append("\\begin{circusaction}\n");
		buffer.append("\t\\circspot \\lschexpract InterpreterInit \\rschexpract \\circseq \\\\\n");
		buffer.append("\t\t\\t1 \\lcircguard thread = main \\rcircguard \\circguard MainThread \\\\\n");
		buffer.append("\t\t\\t1 {} \\extchoice {} \\\\\n");
		buffer.append("\t\t\\t1 \\lcircguard thread \\neq main \\rcircguard \\circguard NotStarted\n");
		buffer.append("\\end{circusaction}\n");
		buffer.append("\\begin{circus}\n");
		buffer.append("\t\\circend");
		buffer.append("\\end{circus}\n\n");
		
		return buffer.toString();
	}
	
	public CThrModel doEliminationOfFrameStack() {
		CThrModel newModel = new CThrModel(classNames, methodNames, fieldNames);
		
		// remove launcher returns
		System.out.println("Removing launcher returns...");
//		long startTime = System.currentTimeMillis();
		
		// - move return action to the end of the method
		for (FullMethodID methodName : methods.keySet()) {
			CircusAction[] methodActions = methods.get(methodName);
			CircusAction returnAction = getReturnAction(methodActions);
			if (returnAction == null) {
				returnAction = new CircusAction.HandleReturnEPC();
			}
			
			
			methodActions = introduceReturnActions(methodActions, returnAction);
			
			methodActions = returnActionDist(methodActions);
			
			methods.put(methodName, methodActions);
			
//			System.out.println("Return action distributed: " + methodName);
//			System.out.println(CircusAction.actionsToString(methods.get(methodName), 1));
		}
		
//		long endTime = System.currentTimeMillis();
//		System.out.println("time elapsed: " + (endTime-startTime));
		
		// - remainder of remove launcher returns is implicit, we assume no launcher returns are present in the body of methods
		
		// localise stack frames
		System.out.println("Localising stack frames...");
//		startTime = System.currentTimeMillis();
		
		// - data refinement to eliminate currentClass is implicit
				
		// - redefine method actions to include arguments and insert stack frame variable
		HashMap<FullMethodID,Boolean> methodReturnsValue = new HashMap<FullMethodID,Boolean>();
		HashMap<FullMethodID,Boolean> methodReturnsLong = new HashMap<FullMethodID,Boolean>();
		HashMap<FullMethodID, CircusAction> executeMethodActions = new HashMap<FullMethodID, CircusAction>();
		for (FullMethodID methodName : methods.keySet()) {
			CircusAction[] oldMethodActions = methods.get(methodName);
			// -- figure out the number of method arguments
			int numArgs = methodName.methodID.getNumArguments();
			if (!classes.get(methodName.classID).isStatic(methodName.methodID)) {
				numArgs = numArgs + 1;
			}
			
			// -- replace return action with actions to pop return values
			Vector<CircusAction> bodyActions = new Vector<CircusAction>();
			bodyActions.add(new CircusAction.InitSF(methodName, numArgs));
			bodyActions.add(new CircusAction.Poll());
			bodyActions.addAll(Arrays.asList(Arrays.copyOf(oldMethodActions, oldMethodActions.length-1)));
			CircusAction returnAction = oldMethodActions[oldMethodActions.length-1];
			if (returnAction instanceof CircusAction.HandleReturnEPC) {
				// delete the return action
				methodReturnsValue.put(methodName, false);
				methodReturnsLong.put(methodName, false);
			} else if (returnAction instanceof CircusAction.HandleAreturnEPC) {
				bodyActions.add(new CircusAction.InterpreterPopEPC("retVal"));
				methodReturnsValue.put(methodName, true);
				methodReturnsLong.put(methodName, false);
			} else if (returnAction instanceof CircusAction.HandleLreturnEPC) {
				bodyActions.add(new CircusAction.InterpreterPopEPC("retVal\\_lsb"));
				bodyActions.add(new CircusAction.InterpreterPopEPC("retVal\\_msb"));
				methodReturnsValue.put(methodName, true);
				methodReturnsLong.put(methodName, true);
			} else {
				System.out.println(methodName + ": \n");
				System.out.println(CircusAction.actionsToString(oldMethodActions,1));
				System.out.flush();
				throw new IllegalStateException(methodName + " doesn't end in a return action");
			}
			
			// -- if all the method did was return then we replace it with Skip
			if (bodyActions.isEmpty()) {
				bodyActions.add(new CircusAction.Skip());
			}
			
			// -- construct the parameter list for the method
			String[] parameterNames; 
			boolean[] parameterIsRet;			
			if (methodReturnsValue.get(methodName)) {
				if (methodReturnsLong.get(methodName)) {
					parameterNames = new String[numArgs+2];
					parameterIsRet = new boolean[numArgs+2];
					for (int i = 0; i < numArgs; i++) {
						parameterNames[i] = "arg" + (i+1);
						parameterIsRet[i] = false;
					}
					parameterNames[numArgs] = "retVal\\_msb";
					parameterIsRet[numArgs] = true;
					parameterNames[numArgs+1] = "retVal\\_lsb";
					parameterIsRet[numArgs+1] = true;
				} else {
					parameterNames = new String[numArgs+1];
					parameterIsRet = new boolean[numArgs+1];
					for (int i = 0; i < numArgs; i++) {
						parameterNames[i] = "arg" + (i+1);
						parameterIsRet[i] = false;
					}
					parameterNames[numArgs] = "retVal";
					parameterIsRet[numArgs] = true;
				}
			} else {
				parameterNames = new String[numArgs];
				parameterIsRet = new boolean[numArgs];
				for (int i = 0; i < numArgs; i++) {
					parameterNames[i] = "arg" + (i+1);
					parameterIsRet[i] = false;
				}
			}
			
			// -- wrap the method in its variable and parameter blocks 
			int numVars = classes.get(methodName.classID).getMethodLocals(methodName.methodID);
			int stackSize = classes.get(methodName.classID).getMethodStackSize(methodName.methodID);
			methods.put(methodName, 
					new CircusAction[] {
							new CircusAction.ParametrisedBlock(parameterNames, parameterIsRet,
									new CircusAction[] {
											new CircusAction.StackFrameVarBlock(bodyActions.toArray(new CircusAction[] {}), numVars, stackSize)
									})
					});
			
			
			// -- update ExecuteMethod to use new parameters
			String[] executeMethodParameterNames; 
			if (methodReturnsValue.get(methodName)) {
				if (methodReturnsLong.get(methodName)) {
					executeMethodParameterNames = new String[numArgs+1];
					for (int i = 0; i < numArgs; i++) {
						executeMethodParameterNames[i] = "methodArgs~" + (i+1);
					}
					executeMethodParameterNames[numArgs] = "retVal";
					executeMethodActions.put(methodName,
							new CircusAction.VarBlock("retval\\_msb, retVal\\_lsb", new CircusAction[]{
									new CircusAction.MethodActionRef(methodName, executeMethodParameterNames),
									new CircusAction.Assignment("retVal", "(retVal\\_msb * " + (1 << 32) + ") + retVal\\_lsb")
								})
							);
				} else {
					executeMethodParameterNames = new String[numArgs+1];
					for (int i = 0; i < numArgs; i++) {
						executeMethodParameterNames[i] = "methodArgs~" + (i+1);
					}
					executeMethodParameterNames[numArgs] = "retVal";
					executeMethodActions.put(methodName, new CircusAction.MethodActionRef(methodName, executeMethodParameterNames));
				}
			} else {
				executeMethodParameterNames = new String[numArgs];
				for (int i = 0; i < numArgs; i++) {
					executeMethodParameterNames[i] = "methodArgs~" + (i+1);
				}
				executeMethodActions.put(methodName, new CircusAction.MethodActionRef(methodName, executeMethodParameterNames));
			}
			
		}
		
		// -- eliminate NewStackFrame and update method calls to use parameters
		for (FullMethodID methodName : methods.keySet()) {
			methods.put(methodName, eliminateNewStackFrame(methods.get(methodName), methodReturnsValue, methodReturnsLong));
		}
		
//		endTime = System.currentTimeMillis();
//		System.out.println("time elapsed: " + (endTime-startTime));
		
		// introduce variables
		System.out.println("Introducing variables...");
//		startTime = System.currentTimeMillis();
		
		// - refine putfield, getfield, putstatic, getstatic, and new using class information
		for (FullMethodID method : methods.keySet()) {
			//System.out.println("Refining with class information: " + method);
			//System.out.println(CircusAction.actionsToString(methods.get(method), 1));
			CircusAction[] oldActions = methods.get(method);
			Vector<CircusAction> newActions = new Vector<CircusAction>();
			for (int i = 0; i < oldActions.length; i++) {
				newActions.addAll(Arrays.asList(oldActions[i].expandWithClassInfo(classes.get(method.classID))));
			}
			methods.put(method, newActions.toArray(new CircusAction[]{}));
		}
		
		// - data refinement proper
		for (FullMethodID method : methods.keySet()) {
			CircusAction[] methodActions = methods.get(method);
			ArrayList<CircusAction> newActions = new ArrayList<CircusAction>();
			int stackDepth = 0;
			for (CircusAction action : methodActions) {
				newActions.addAll(Arrays.asList(action.doEFSDataRefinement(stackDepth)));
				stackDepth += action.stackDepthChange();
			}
			methods.put(method, newActions.toArray(new CircusAction[] {}));
		}
		
		// - eliminate unnecessary variable blocks
		for (FullMethodID method : methods.keySet()) {
			CircusAction[] methodActions = methods.get(method);
			//if (method.classID.equals(new ClassID("java/lang/LongArray"))) {
				//System.out.println("Eliminating var blocks: " + method);
				//System.out.println("\t\t\\t1 " + CircusAction.actionsToString(methods.get(method), 1));
			//}
			CircusAction[] newActions = eliminateVarBlocks(methodActions);
			newModel.addMethod(method, newActions, executeMethodActions.get(method));
		}
		
//		endTime = System.currentTimeMillis();
//		System.out.println("time elapsed: " + (endTime-startTime));
		
		return newModel;
	}

	private CircusAction[] eliminateVarBlocks(CircusAction[] actions) {
		Vector<CircusAction> newActions = new Vector<CircusAction>();
		
		for (int i = 0; i < actions.length; i++) {
			if (actions[i] instanceof CircusAction.VarBlock) {
				String varName = ((VarBlock) actions[i]).getVarName();
				CircusAction[] varBlockActions = ((VarBlock) actions[i]).getActions();
				if (varName.equals("value1, value2")) {
					// Rule [cond-value1-value2-elim]
					if (varBlockActions[0] instanceof CircusAction.Assignment) {
						if (varBlockActions[1] instanceof CircusAction.Assignment) {
							CircusAction.Assignment assign1 = (Assignment) varBlockActions[0];
							CircusAction.Assignment assign2 = (Assignment) varBlockActions[1];
							String assign1Var = assign1.getVar();
							String assign2Var = assign2.getVar();
							String assign1Expr = assign1.getExpr();
							String assign2Expr = assign2.getExpr();
							if (varBlockActions[2] instanceof CircusAction.Conditional) {
								String[] conditions = ((Conditional) varBlockActions[2]).getConditions();
								CircusAction[][] branches = ((Conditional) varBlockActions[2]).getConditionActions();
								String[] newConditions = new String[conditions.length];
								CircusAction[][] newBranches = new CircusAction[branches.length][];
								for (int j = 0; j < conditions.length; j++) {
									newConditions[j] = conditions[j].replace(assign1Var, assign1Expr).replace(assign2Var, assign2Expr);
									newBranches[j] = eliminateVarBlocks(branches[j]);
								}
								newActions.add(new CircusAction.Conditional(newConditions, newBranches));
							}
						}
					}
				} else if (varName.equals("oid")) {
					// Rule [getField-oid-elim]
					//first, do we have a normal getfield or a long getfield?
					if (varBlockActions.length == 2) {
						// normal getfield - apply rule as written 
						if (varBlockActions[0] instanceof CircusAction.Assignment) {
							CircusAction.Assignment assign = (Assignment) varBlockActions[0];
							String assignExpr = assign.getExpr();
							if (varBlockActions[1] instanceof CircusAction.ChannelComm) {
								CircusAction.ChannelComm comm = (ChannelComm) varBlockActions[1];
								String[] commArgs = comm.getChannelArgs();
								String[] newCommArgs = new String[commArgs.length];
								newCommArgs[0] = assignExpr;
								for (int j = 1; j < commArgs.length; j++) {
									newCommArgs[j] = commArgs[j];
								}
								newActions.add(new CircusAction.ChannelComm(
										comm.getChannelName(), newCommArgs, comm.getArgIsInputArray(), comm.getAction()));
							}
						}
					} else {
						// long getfield - copy oid to both communications
						if (varBlockActions[0] instanceof CircusAction.Assignment) {
							CircusAction.Assignment assign = (Assignment) varBlockActions[0];
							String assignExpr = assign.getExpr();
							if (varBlockActions[1] instanceof CircusAction.ChannelComm) {
								CircusAction.ChannelComm comm = (ChannelComm) varBlockActions[1];
								String[] commArgs = comm.getChannelArgs();
								String[] newCommArgs = new String[commArgs.length];
								newCommArgs[0] = assignExpr;
								for (int j = 1; j < commArgs.length; j++) {
									newCommArgs[j] = commArgs[j];
								}
								newActions.add(new CircusAction.ChannelComm(
										comm.getChannelName(), newCommArgs, comm.getArgIsInputArray(), comm.getAction()));
							}
							if (varBlockActions[2] instanceof CircusAction.ChannelComm) {
								CircusAction.ChannelComm comm = (ChannelComm) varBlockActions[2];
								String[] commArgs = comm.getChannelArgs();
								String[] newCommArgs = new String[commArgs.length];
								newCommArgs[0] = assignExpr;
								for (int j = 1; j < commArgs.length; j++) {
									newCommArgs[j] = commArgs[j];
								}
								newActions.add(new CircusAction.ChannelComm(
										comm.getChannelName(), newCommArgs, comm.getArgIsInputArray(), comm.getAction()));
							}
						}
					}
				} else if (varName.equals("oid, value")) {
					// Rule [putField-oid-value-elim]
					if (varBlockActions[0] instanceof CircusAction.Assignment) {
						if (varBlockActions[1] instanceof CircusAction.Assignment) {
							CircusAction.Assignment assign1 = (Assignment) varBlockActions[0];
							CircusAction.Assignment assign2 = (Assignment) varBlockActions[1];
							//String assign1Var = assign1.getVar();
							//String assign2Var = assign2.getVar();
							String assign1Expr = assign1.getExpr();
							String assign2Expr = assign2.getExpr();
							if (varBlockActions[2] instanceof CircusAction.ChannelComm) {
								CircusAction.ChannelComm comm = (ChannelComm) varBlockActions[2];
								String[] commArgs = comm.getChannelArgs();
								String[] newCommArgs = new String[commArgs.length];
								newCommArgs[0] = assign2Expr;
								newCommArgs[1] = commArgs[1];
								newCommArgs[2] = commArgs[2];
								newCommArgs[3] = assign1Expr;
								newActions.add(new CircusAction.ChannelComm(
										comm.getChannelName(), newCommArgs, comm.getArgIsInputArray(), comm.getAction()));
							}
						}
					}
				} else if (varName.equals("oid, lsb, msb")) {
					// adapt Rule [putField-oid-value-elim] to long value
					if (varBlockActions[0] instanceof CircusAction.Assignment) {
						if (varBlockActions[1] instanceof CircusAction.Assignment) {
							if (varBlockActions[2] instanceof CircusAction.Assignment) {
								CircusAction.Assignment lsbAssign = (Assignment) varBlockActions[0];
								CircusAction.Assignment msbAssign = (Assignment) varBlockActions[1];
								CircusAction.Assignment oidAssign = (Assignment) varBlockActions[2];
								String lsb = lsbAssign.getExpr();
								String msb = msbAssign.getExpr();
								String oid = oidAssign.getExpr();
								if (varBlockActions[3] instanceof CircusAction.ChannelComm) {
									CircusAction.ChannelComm comm = (ChannelComm) varBlockActions[3];
									String[] commArgs = comm.getChannelArgs();
									String[] newCommArgs = new String[commArgs.length];
									newCommArgs[0] = oid;
									newCommArgs[1] = commArgs[1];
									newCommArgs[2] = commArgs[2];
									newCommArgs[3] = lsb;
									newActions.add(new CircusAction.ChannelComm(
											comm.getChannelName(), newCommArgs, comm.getArgIsInputArray(), comm.getAction()));
								}
								if (varBlockActions[4] instanceof CircusAction.ChannelComm) {
									CircusAction.ChannelComm comm = (ChannelComm) varBlockActions[4];
									String[] commArgs = comm.getChannelArgs();
									String[] newCommArgs = new String[commArgs.length];
									newCommArgs[0] = oid;
									newCommArgs[1] = commArgs[1];
									newCommArgs[2] = commArgs[2];
									newCommArgs[3] = msb;
									newActions.add(new CircusAction.ChannelComm(
											comm.getChannelName(), newCommArgs, comm.getArgIsInputArray(), comm.getAction()));
								}
							}
						}
					}
				} else if (varName.equals("value")) {
					if (varBlockActions[0] instanceof CircusAction.Assignment) {
						CircusAction.Assignment assign = (Assignment) varBlockActions[0];
						String assignVar = assign.getVar();
						String assignExpr = assign.getExpr();
						if (varBlockActions[1] instanceof CircusAction.ChannelComm) {
							// Rule [putStatic-value-elim]
							CircusAction.ChannelComm comm = (ChannelComm) varBlockActions[1];
							String[] commArgs = comm.getChannelArgs();
							String[] newCommArgs = new String[commArgs.length];
							for (int j = 0; j < commArgs.length-1; j++) {
								newCommArgs[j] = commArgs[j];
							}
							newCommArgs[commArgs.length-1] = assignExpr;
							newActions.add(new CircusAction.ChannelComm(
									comm.getChannelName(), newCommArgs, comm.getArgIsInputArray(), comm.getAction()));
						} else if (varBlockActions[1] instanceof CircusAction.Conditional) {
							// Rule [cond-value-elim] (not in thesis but similar to Rule [cond-value1-value2-elim])
							String[] conditions = ((Conditional) varBlockActions[1]).getConditions();
							CircusAction[][] branches = ((Conditional) varBlockActions[1]).getConditionActions();
							String[] newConditions = new String[conditions.length];
							CircusAction[][] newBranches = new CircusAction[branches.length][];
							for (int j = 0; j < conditions.length; j++) {
								newConditions[j] = conditions[j].replace(assignVar, assignExpr);
								newBranches[j] = eliminateVarBlocks(branches[j]);
							}
							newActions.add(new CircusAction.Conditional(newConditions, newBranches));
						}
					}
				} else if (varName.equals("lsb, msb")) {
					// adapt Rule [putStatic-value-elim] to long value
					if (varBlockActions[0] instanceof CircusAction.Assignment) {
						CircusAction.Assignment lsbAssign = (Assignment) varBlockActions[0];
						String lsb = lsbAssign.getExpr();
						if (varBlockActions[1] instanceof CircusAction.Assignment) {
							CircusAction.Assignment msbAssign = (Assignment) varBlockActions[1];
							String msb = msbAssign.getExpr();
							if (varBlockActions[2] instanceof CircusAction.ChannelComm) {
								CircusAction.ChannelComm comm = (ChannelComm) varBlockActions[2];
								String[] commArgs = comm.getChannelArgs();
								String[] newCommArgs = new String[commArgs.length];
								for (int j = 0; j < commArgs.length-1; j++) {
									newCommArgs[j] = commArgs[j];
								}
								newCommArgs[commArgs.length-1] = lsb;
								newActions.add(new CircusAction.ChannelComm(
										comm.getChannelName(), newCommArgs, comm.getArgIsInputArray(), comm.getAction()));
							}
							if (varBlockActions[3] instanceof CircusAction.ChannelComm) {
								// Rule [putStatic-value-elim]
								CircusAction.ChannelComm comm = (ChannelComm) varBlockActions[3];
								String[] commArgs = comm.getChannelArgs();
								String[] newCommArgs = new String[commArgs.length];
								for (int j = 0; j < commArgs.length-1; j++) {
									newCommArgs[j] = commArgs[j];
								}
								newCommArgs[commArgs.length-1] = lsb;
								newActions.add(new CircusAction.ChannelComm(
										comm.getChannelName(), newCommArgs, comm.getArgIsInputArray(), comm.getAction()));
							}
						}
					}
				} else if (varName.equals("poppedArgs")) {
					if (varBlockActions[0] instanceof CircusAction.PoppedArgsAssignment) {
						String[] args = ((PoppedArgsAssignment) varBlockActions[0]).getArgs();
						if (varBlockActions[1] instanceof CircusAction.MethodActionRef) {
							// Rule [poppedArgs-elim]
							CircusAction.MethodActionRef methodCall = (MethodActionRef) varBlockActions[1];
							FullMethodID methodName = methodCall.getMethodName();
							String[] methodArgs = methodCall.getMethodArgs();
							int numMethodArgs = methodArgs.length;
							String[] newMethodArgs = new String[numMethodArgs];
							
							for (int j = 0; j < args.length; j++) {
								newMethodArgs[j] = args[j];
							}
							for (int j = args.length; j < methodArgs.length; j++) {
								newMethodArgs[j] = methodArgs[j];
							}
							newActions.add(new CircusAction.MethodActionRef(methodName, newMethodArgs, methodCall.getIsReturnArray()));
						} else if (varBlockActions[1] instanceof CircusAction.ChannelComm) {
							CircusAction.ChannelComm comm = (CircusAction.ChannelComm) varBlockActions[1];
							if (comm.getChannelName().equals("takeLock")) {
								// Rule [poppedArgs-sync-elim]
								if (varBlockActions[2] instanceof CircusAction.MethodActionRef) {
									CircusAction.MethodActionRef methodCall = (MethodActionRef) varBlockActions[2];
									FullMethodID methodName = methodCall.getMethodName();
									String[] methodArgs = methodCall.getMethodArgs();
									int numMethodArgs = methodArgs.length;
									String[] newMethodArgs = new String[numMethodArgs];
									
									for (int j = 0; j < args.length; j++) {
										newMethodArgs[j] = args[j];
									}
									for (int j = args.length; j < methodArgs.length; j++) {
										newMethodArgs[j] = methodArgs[j];
									}
									newActions.add(new CircusAction.ChannelComm(
											comm.getChannelName(), new String[] {args[0]}, new boolean[] {false}, comm.getAction()));
									newActions.add(new CircusAction.MethodActionRef(methodName, newMethodArgs, methodCall.getIsReturnArray()));
								}
							} else if (comm.getChannelName().equals("getClassIDOf")) {
								// Rule [invokevirtual-poppedArgs-elim]
								if (comm.getAction() instanceof CircusAction.Conditional) {
									CircusAction.Conditional conditional = (CircusAction.Conditional) comm.getAction();
									CircusAction[][] branches = conditional.getConditionActions();
									CircusAction[][] newBranches = new CircusAction[branches.length][];
									
									for (int j = 0; j < branches.length; j++) {
										if (branches[j][0] instanceof CircusAction.MethodActionRef) {
											CircusAction.MethodActionRef methodCall = (MethodActionRef) branches[j][0];
											FullMethodID methodName = methodCall.getMethodName();
											String[] methodArgs = methodCall.getMethodArgs();
											int numMethodArgs = methodArgs.length;
											String[] newMethodArgs = new String[numMethodArgs];
											
											for (int k = 0; k < args.length; k++) {
												newMethodArgs[k] = args[k];
											}
											for (int k = args.length; k < methodArgs.length; k++) {
												newMethodArgs[k] = methodArgs[k];
											}
											newBranches[j] = new CircusAction[] {
													new CircusAction.MethodActionRef(methodName, newMethodArgs, methodCall.getIsReturnArray())
											};
										} else if (branches[j][0] instanceof CircusAction.ChannelComm) {
											CircusAction.ChannelComm takeLockComm = (ChannelComm) branches[j][0];
											CircusAction.MethodActionRef methodCall = (MethodActionRef) branches[j][1];
											FullMethodID methodName = methodCall.getMethodName();
											String[] methodArgs = methodCall.getMethodArgs();
											int numMethodArgs = methodArgs.length;
											String[] newMethodArgs = new String[numMethodArgs];
											
											for (int k = 0; k < args.length; k++) {
												newMethodArgs[k] = args[k];
											}
											for (int k = args.length; k < methodArgs.length; k++) {
												newMethodArgs[k] = methodArgs[k];
											}
											newBranches[j] = new CircusAction[] {
													new CircusAction.ChannelComm(
															takeLockComm.getChannelName(), new String[] {args[0]}, new boolean[] {false}, takeLockComm.getAction()),
													new CircusAction.MethodActionRef(methodName, newMethodArgs, methodCall.getIsReturnArray())
											};
										}
									}
									
									newActions.add(new CircusAction.ChannelComm(
											comm.getChannelName(), 
											new String[] {args[0], comm.getChannelArgs()[1]}, 
											comm.getArgIsInputArray(), 
											new CircusAction.Conditional(conditional.getConditions(), newBranches)));
								}
							} else if (comm.getChannelName().equals("setPriorityCeiling")) {
								// setPriorityCeiling takes two arguments
								newActions.add(new CircusAction.ChannelComm(
										comm.getChannelName(), 
										new String[] {args[0], args[1]}, 
										comm.getArgIsInputArray(), 
										comm.getAction()));
							} else if (comm.getChannelName().equals("output")) {
								// output takes one argument
								newActions.add(new CircusAction.ChannelComm(
										comm.getChannelName(), 
										new String[] {args[0]}, 
										comm.getArgIsInputArray(), 
										comm.getAction()));
							} else if (comm.getChannelName().equals("input")) {
								// input takes no arguments
								newActions.add(comm);
							} else if (comm.getChannelName().equals("register")) {
								// register takes a ThreadID and one argument
								newActions.add(new CircusAction.ChannelComm(
										comm.getChannelName(), 
										new String[] {comm.getChannelArgs()[0], args[0]}, 
										comm.getArgIsInputArray(), 
										comm.getAction()));
							} else if (comm.getChannelName().equals("enterPrivateMemory")) {
								// enterPrivateMemory takes a ThreadID and one argument
								newActions.add(new CircusAction.ChannelComm(
										comm.getChannelName(), 
										new String[] {comm.getChannelArgs()[0], args[0]}, 
										comm.getArgIsInputArray(), 
										comm.getAction()));
							} else if (comm.getChannelName().equals("executeInAreaOf")) {
								// executeInAreaOf takes a ThreadID and one argument
								newActions.add(new CircusAction.ChannelComm(
										comm.getChannelName(), 
										new String[] {comm.getChannelArgs()[0], args[0]}, 
										comm.getArgIsInputArray(), 
										comm.getAction()));
							} else if (comm.getChannelName().equals("executeInOuterArea")) {
								// executeInOuterArea takes a ThreadID and no arguments
								newActions.add(comm);
							} else if (comm.getChannelName().equals("exitMemory")) {
								// exitMemory takes a ThreadID and no arguments
								newActions.add(comm);
							} else if (comm.getChannelName().equals("initAPEH")) {
								// initAPEH takes a ThreadID and five arguments - argIsInput needs updating
								newActions.add(new CircusAction.ChannelComm(
										comm.getChannelName(), 
										new String[] {comm.getChannelArgs()[0], args[0], args[1], args[2], args[3], args[4]}, 
										new boolean[]{false, false, false, false, false, false},
										comm.getAction()));
							} else if (comm.getChannelName().equals("initPEH")) {
								// initPEH takes a ThreadID and seven arguments - argIsInput needs updating
								newActions.add(new CircusAction.ChannelComm(
										comm.getChannelName(), 
										new String[] {comm.getChannelArgs()[0], args[0], args[1], args[2], args[3], args[4], args[5], args[6]}, 
										new boolean[]{false, false, false, false, false, false, false, false}, 
										comm.getAction()));
							} else if (comm.getChannelName().equals("initOSEHAbs")) {
								// initOSEHAbs takes a ThreadID and six arguments - argIsInput needs updating
								newActions.add(new CircusAction.ChannelComm(
										comm.getChannelName(), 
										new String[] {comm.getChannelArgs()[0], args[0], args[1], args[2], args[3], args[4], args[5]}, 
										new boolean[]{false, false, false, false, false, false, false}, 
										comm.getAction()));
							} else if (comm.getChannelName().equals("initOSEHRel")) {
								// initOSEHRel takes a ThreadID and six arguments - argIsInput needs updating
								newActions.add(new CircusAction.ChannelComm(
										comm.getChannelName(), 
										new String[] {comm.getChannelArgs()[0], args[0], args[1], args[2], args[3], args[4], args[5]}, 
										new boolean[]{false, false, false, false, false, false, false},
										comm.getAction()));
							} else if (comm.getChannelName().equals("releaseAperiodic")) {
								// releaseAperiodic takes a ThreadID and one argument
								newActions.add(new CircusAction.ChannelComm(
										comm.getChannelName(), 
										new String[] {comm.getChannelArgs()[0], args[0]}, 
										comm.getArgIsInputArray(), 
										comm.getAction()));
							} else {
								// for other communications, do nothing
								newActions.add(actions[i]);
							}
						}
					} else {
						throw new IllegalStateException("poppedArgs variable block can't be eliminated");
					}
				} else {
					throw new IllegalStateException("variable block isn't of a recognised type");
				}
			} else if (actions[i] instanceof CircusAction.ParametrisedBlock) {
				// Rule [var-parameter-conversion] and Rule [argument-variable-elimination]
				CircusAction.ParametrisedBlock paramBlock = (CircusAction.ParametrisedBlock) actions[i];
				String[] params = paramBlock.getParameterNames();
				boolean[] paramIsRet = paramBlock.getParameterIsRet();
				String[] newParams = new String[params.length];
				
				int argCount = 0;
				for (int j = 0; j < params.length; j++) {
					if (paramIsRet[j]) {
						newParams[j] = params[j];
					} else {
						newParams[j] = "var" + (j+1);
						argCount++;
					}
				}
				
				if (paramBlock.getActions()[0] instanceof CircusAction.MethodVarBlock) {
					CircusAction.MethodVarBlock methodBlock = (CircusAction.MethodVarBlock) paramBlock.getActions()[0];
					int numVars = methodBlock.getNumVars();
					int stackSize = methodBlock.getStackSize();
					CircusAction[] blockActions = methodBlock.getActions();
					// remove the first argCount actions (since they are assignments to the variables made into parameters
					CircusAction[] newBlockActions = eliminateVarBlocks(Arrays.copyOfRange(blockActions, argCount, blockActions.length));
					
					newActions.add(new CircusAction.ParametrisedBlock(newParams, paramIsRet,
							new CircusAction[] {
									new CircusAction.MethodVarBlock(newBlockActions, numVars, stackSize, argCount+1)
							}));
				}
				
			} else if (actions[i] instanceof CircusAction.Conditional) {
				String[] conditions = ((Conditional) actions[i]).getConditions();
				CircusAction[][] branchActions = ((Conditional) actions[i]).getConditionActions();
				CircusAction[][] newBranchActions = new CircusAction[branchActions.length][];
				
				for (int j = 0; j < branchActions.length; j++) {
					newBranchActions[i] = eliminateVarBlocks(branchActions[i]);
				}
				
				newActions.add(new CircusAction.Conditional(conditions, newBranchActions));
			} else if (actions[i] instanceof CircusAction.ChannelComm) {
				CircusAction oldAction = ((ChannelComm) actions[i]).getAction();
				CircusAction newAction = eliminateVarBlocks(new CircusAction[] {oldAction})[0];
						
				newActions.add(new CircusAction.ChannelComm(
						((ChannelComm) actions[i]).getChannelName(), 
						((ChannelComm) actions[i]).getChannelArgs(), 
						((ChannelComm) actions[i]).getArgIsInputArray(), 
						newAction));
			} else if (actions[i] instanceof CircusAction.Recursion) {
				String recVar = ((CircusAction.Recursion) actions[i]).getRecVarName();
				CircusAction[] recBody = ((CircusAction.Recursion) actions[i]).getActions();
				// apply variable elimination to body of recursion first
				CircusAction[] newRecBody = eliminateVarBlocks(recBody);
				
				// then apply Law [rec-rolling-rule] if it applies
				CircusAction lastAction = newRecBody[newRecBody.length-1];
				if (lastAction instanceof CircusAction.Conditional) {
					String[] conditions = ((CircusAction.Conditional) lastAction).getConditions();
					CircusAction[][] conditionalBranches = ((CircusAction.Conditional) lastAction).getConditionActions();
					CircusAction[][] newConditionalBranches = new CircusAction[conditionalBranches.length][];
					boolean rollingRuleApplies = false;
					for (int branchIndex = 0; branchIndex < conditionalBranches.length; branchIndex++) {
						CircusAction[] branch = conditionalBranches[branchIndex];
						CircusAction branchLastAction = branch[branch.length-1];
						if (branchLastAction instanceof CircusAction.ActionRef) {
							String actionRefName = ((CircusAction.ActionRef) branchLastAction).getActionName();
							if (actionRefName.equals(recVar)) {
								rollingRuleApplies = true;
								Vector<CircusAction> newBranch = new Vector<CircusAction>();
								for (int j = 0; j < branch.length-1; j++) {
									newBranch.add(branch[j]);
								}
								for (int j = 0; j < newRecBody.length-1; j++) {
									newBranch.add(newRecBody[j]);
								}
								newBranch.add(branchLastAction);
								newConditionalBranches[branchIndex] = newBranch.toArray(new CircusAction[] {});
							}
						} else {
							newConditionalBranches[branchIndex] = branch;
						}
					}
					
					if (rollingRuleApplies) {
						for (int j = 0; j < newRecBody.length-1; j++) {
							newActions.add(newRecBody[j]);
						}
						newActions.add(new CircusAction.Recursion(recVar, 
								new CircusAction[] {
										new CircusAction.Conditional(conditions, newConditionalBranches)
								}));
					} else {
						newActions.add(new CircusAction.Recursion(recVar, newRecBody));
					}
				}
				
			} else {
				// everything else requires no changes
				newActions.add(actions[i]);
			}
		}
		
		return newActions.toArray(new CircusAction[] {});
	}

	private CircusAction[] eliminateNewStackFrame(CircusAction[] actions, HashMap<FullMethodID, Boolean> methodReturnsValue, HashMap<FullMethodID, Boolean> methodReturnsLong) {
		Vector<CircusAction> newActions = new Vector<CircusAction>();
		
		for (int i = 0; i < actions.length; i++) {
			if (actions[i] instanceof CircusAction.VarBlock) {
				CircusAction[] varBlockActions = ((VarBlock) actions[i]).getActions();
				String varName = ((VarBlock) actions[i]).getVarName();
				newActions.add(new CircusAction.VarBlock(varName, eliminateNewStackFrame(varBlockActions, methodReturnsValue, methodReturnsLong)));
			} else if (actions[i] instanceof CircusAction.Recursion) {
				CircusAction[] recActions = ((CircusAction.Recursion) actions[i]).getActions();
				String recVar = ((CircusAction.Recursion) actions[i]).getRecVarName();
				newActions.add(new CircusAction.Recursion(recVar, eliminateNewStackFrame(recActions, methodReturnsValue, methodReturnsLong)));
			} else if (actions[i] instanceof CircusAction.Conditional) {
				String[] conditions = ((Conditional) actions[i]).getConditions();
				CircusAction[][] branchActions = ((Conditional) actions[i]).getConditionActions();
				CircusAction[][] newBranchActions = new CircusAction[branchActions.length][];
				
				for (int j = 0; j < branchActions.length; j++) {
					newBranchActions[j] = eliminateNewStackFrame(branchActions[j], methodReturnsValue, methodReturnsLong);
				}
				
				newActions.add(new CircusAction.Conditional(conditions, newBranchActions));
			} else if (actions[i] instanceof CircusAction.ChannelComm) {
				CircusAction oldAction = ((ChannelComm) actions[i]).getAction();
				CircusAction newAction = eliminateNewStackFrame(new CircusAction[] {oldAction}, methodReturnsValue, methodReturnsLong)[0];
						
				newActions.add(new CircusAction.ChannelComm(
						((ChannelComm) actions[i]).getChannelName(), 
						((ChannelComm) actions[i]).getChannelArgs(), 
						((ChannelComm) actions[i]).getArgIsInputArray(), 
						newAction));
			} else if (actions[i] instanceof CircusAction.MethodActionRef) {
				FullMethodID methodName = ((MethodActionRef) actions[i]).getMethodName();
				int numArgs = methodName.methodID.getNumArguments();
				if (!classes.get(methodName.classID).isStatic(methodName.methodID)) {
					numArgs = numArgs + 1;
				}
				
				String[] parameterNames; 
				
				if (methodReturnsValue.get(methodName)) {
					if (methodReturnsLong.get(methodName)) {
						parameterNames = new String[numArgs+2];
						for (int j = 0; j < numArgs; j++) {
							parameterNames[j] = "poppedArgs~" + (j+1);
						}
						parameterNames[numArgs] = "retVal\\_msb";
						parameterNames[numArgs+1] = "retVal\\_lsb";
					} else {
						parameterNames = new String[numArgs+1];
						for (int j = 0; j < numArgs; j++) {
							parameterNames[j] = "poppedArgs~" + (j+1);
						}
						parameterNames[numArgs] = "retVal";
					}
				} else {
					parameterNames = new String[numArgs];
					for (int j = 0; j < numArgs; j++) {
						parameterNames[j] = "poppedArgs~" + (j+1);
					}
				}
				
				newActions.add(new CircusAction.MethodActionRef(methodName, parameterNames));
			} else if (actions[i] instanceof CircusAction.InterpreterNewStackFrame) {
				// eliminate this
				
				// and the following Poll action
				i++;
			} else if (actions[i] instanceof CircusAction.ParametrisedBlock) {
				CircusAction.ParametrisedBlock block = (CircusAction.ParametrisedBlock) actions[i];
				CircusAction[] blockActions = block.getActions();
				CircusAction[] newBlockActions = eliminateNewStackFrame(blockActions, methodReturnsValue, methodReturnsLong);
				newActions.add(new CircusAction.ParametrisedBlock(block.getParameterNames(), block.getParameterIsRet(), newBlockActions));
			} else if (actions[i] instanceof CircusAction.StackFrameVarBlock) {
				CircusAction.StackFrameVarBlock block = (CircusAction.StackFrameVarBlock) actions[i];
				CircusAction[] blockActions = block.getActions();
				CircusAction[] newBlockActions = eliminateNewStackFrame(blockActions, methodReturnsValue, methodReturnsLong);
				newActions.add(new CircusAction.StackFrameVarBlock(newBlockActions, block.getNumVars(), block.getStackSize()));
			} else {
				// everything else requires no changes
				newActions.add(actions[i]);
			}
		}
		
		return newActions.toArray(new CircusAction[] {});
	}

	private CircusAction[] returnActionDist(CircusAction[] actions) {
		CircusAction lastAction = actions[actions.length-1];
		if (lastAction instanceof CircusAction.VarBlock) {
			String varName = ((CircusAction.VarBlock) lastAction).getVarName();
			CircusAction[] varBlockActions = ((CircusAction.VarBlock) lastAction).getActions();
			
			
			CircusAction lastAction2 = varBlockActions[varBlockActions.length-1];
			if (lastAction2 instanceof CircusAction.Conditional) {
				String[] conditions = ((CircusAction.Conditional) lastAction2).getConditions();
				CircusAction[][] branchActions = ((CircusAction.Conditional) lastAction2).getConditionActions();
				
				CircusAction[][] newBranchActions = new CircusAction[branchActions.length][];
				CircusAction distributedAction = null;
				
				for (int i = 0; i < branchActions.length; i++) {
					CircusAction[] branch = returnActionDist(branchActions[i]);
					newBranchActions[i] = Arrays.copyOf(branch, branch.length-1);
					distributedAction = branch[branch.length-1];
				}
				
				CircusAction[] newVarBlockActions = new CircusAction[varBlockActions.length];
				for (int i = 0; i < varBlockActions.length-1; i++) {
					newVarBlockActions[i] = varBlockActions[i];
				}
				newVarBlockActions[varBlockActions.length-1] = new CircusAction.Conditional(conditions, newBranchActions);
				
				
				CircusAction[] newActions = new CircusAction[actions.length+1];
				for (int i = 0; i < actions.length-1; i++) {
					newActions[i] = actions[i];
				}
				newActions[actions.length-1] = new CircusAction.VarBlock(varName, newVarBlockActions);
				newActions[actions.length] = distributedAction;
				
				
				return newActions;
			} else {
				return actions;
			}
		} else {
			return actions;
		}
	}

	private CircusAction[] introduceReturnActions(CircusAction[] actions, CircusAction returnAction) {
		CircusAction lastAction = actions[actions.length-1];
		if (lastAction instanceof CircusAction.Recursion) {
			CircusAction[] newActions = new CircusAction[actions.length + 1];
			for (int i = 0; i < actions.length; i++) {
				newActions[i] = actions[i];
			}
			newActions[actions.length] = returnAction;
			return newActions;
		} else if (lastAction instanceof CircusAction.VarBlock) {
			String varName = ((CircusAction.VarBlock) lastAction).getVarName();
			CircusAction[] varBlockActions = ((CircusAction.VarBlock) lastAction).getActions();
			
			CircusAction[] newActions = new CircusAction[actions.length];
			for (int i = 0; i < actions.length-1; i++) {
				newActions[i] = actions[i];
			}
			newActions[actions.length-1] = new CircusAction.VarBlock(varName, introduceReturnActions(varBlockActions, returnAction));
		
			return newActions;
		} else if (lastAction instanceof CircusAction.Conditional) {
			String[] conditions = ((CircusAction.Conditional) lastAction).getConditions();
			CircusAction[][] branchActions = ((CircusAction.Conditional) lastAction).getConditionActions();
			
			CircusAction[][] newBranchActions = new CircusAction[branchActions.length][];
			
			for (int i = 0; i < branchActions.length; i++) {
				newBranchActions[i] = introduceReturnActions(branchActions[i], returnAction);
			}
			
			
			CircusAction[] newActions = new CircusAction[actions.length];
			for (int i = 0; i < actions.length-1; i++) {
				newActions[i] = actions[i];
			}
			newActions[actions.length-1] = new CircusAction.Conditional(conditions, newBranchActions);
		
			return newActions;
		} else {
			return actions;
		}
	}

	private CircusAction getReturnAction(CircusAction[] actions) {
		CircusAction lastAction = actions[actions.length-1];
		if (lastAction instanceof CircusAction.HandleReturnEPC) {
			return lastAction;
		} else if (lastAction instanceof CircusAction.HandleAreturnEPC) {
			return lastAction;
		} else if (lastAction instanceof CircusAction.HandleLreturnEPC) {
			return lastAction;
		} else if (lastAction instanceof CircusAction.VarBlock) {
			return getReturnAction(((CircusAction.VarBlock) lastAction).getActions());
		} else if (lastAction instanceof CircusAction.Conditional) {
			for (CircusAction[] branch : ((CircusAction.Conditional) lastAction).getConditionActions()) {
				CircusAction returnAction = getReturnAction(branch);
				if (returnAction != null) {
					return returnAction;
				}
			}
			return null;
		}
		
		return null;
	}
}
