package main;

import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Vector;

import main.CPEntry.MethodRef;
import main.CircusAction.PCAssumption;

public class Model {
	
	private HashSet<ClassID> classNames = new HashSet<ClassID>(); // class names used as identifiers in the model
	private Vector<ClassID> classNameOrdering = new Vector<ClassID>();
	private HashSet<MethodID> methodNames = new HashSet<MethodID>(); // method names used as identifiers in the model
	private Vector<MethodID> methodNameOrdering = new Vector<MethodID>();
	private HashSet<FieldID> fieldNames = new HashSet<FieldID>(); // field names used as identifiers in the model
	private Vector<FieldID> fieldNameOrdering = new Vector<FieldID>();
	private Vector<ClassModel> classes = new Vector<ClassModel>(); // class information from classes present in the model
	private Vector<BytecodeModel> bytecodes = new Vector<BytecodeModel>();
	
	private HashSet<FullMethodID> excludedMethods = new HashSet<FullMethodID>();
	
	public void addClass(ClassModel model) {
		classes.add(model);
	}
	
	public void addClassName(ClassID className) {
		if (!classNames.contains(className)) {
			classNames.add(className);
			classNameOrdering.add(className);
		}
	}
	
	public void addMethodName(MethodID methodName) {
		if (!methodNames.contains(methodName)) {
			methodNames.add(methodName);
			methodNameOrdering.add(methodName);
		}
	}
	
	public void addFieldName(FieldID fieldName) {
		if (!fieldNames.contains(fieldName)) {
			fieldNames.add(fieldName);
			fieldNameOrdering.add(fieldName);
		}
	}
	
	public void addBytecode(BytecodeModel bytecode) {
		this.bytecodes.add(bytecode);
	}
	
	public int getNextBytecodeIndex() {
		return this.bytecodes.size();
	}
	
	public String toModelString() {
		StringBuilder buffer = new StringBuilder(1024);
		
		buffer.append("\\begin{axdef}\n");
		Iterator<ClassID> classIDIterator = classNameOrdering.iterator();
		while (classIDIterator.hasNext()) {
			buffer.append("\t" + classIDIterator.next().toModelString() + "ID : ClassID");
			if (classIDIterator.hasNext()) {
				buffer.append(" \\\\\n");
			} else {
				buffer.append("\n");
			}
		}
		buffer.append("\\end{axdef}\n\n");
		
		buffer.append("\\begin{axdef}\n");
		Iterator<MethodID> methodIDIterator = methodNameOrdering.iterator();
		while (methodIDIterator.hasNext()) {
			buffer.append("\t" + methodIDIterator.next().toModelString() + " : MethodID");
			if (methodIDIterator.hasNext()) {
				buffer.append(" \\\\\n");
			} else {
				buffer.append("\n");
			}
		}
		buffer.append("\\end{axdef}\n\n");
		
		buffer.append("\\begin{axdef}\n");
		Iterator<FieldID> fieldIDIterator = fieldNameOrdering.iterator();
		while (fieldIDIterator.hasNext()) {
			buffer.append("\t" + fieldIDIterator.next() + " : FieldID");
			if (fieldIDIterator.hasNext()) {
				buffer.append(" \\\\\n");
			} else {
				buffer.append("\n");
			}
		}
		buffer.append("\\end{axdef}\n\n");
		
		for (ClassModel cm : classes) {
			buffer.append(cm.toModelString() + "\n");
		}
		
		buffer.append("\\begin{axdef}\n");
		buffer.append("\tprogramMemory : ProgramAddress \\pfun Bytecode\n");
		buffer.append("\\where\n");
		buffer.append("\tprogramMemory = \\{ \\\\\n");
		for (int i = 0; i < bytecodes.size(); i++) {
			if (bytecodes.elementAt(i) instanceof BytecodeModel.NonSubsetInstruction) {
				buffer.append("%"); // we don't want to add anything incompatible with the model
			}
			buffer.append("\t\t\\t1 " + i + " \\mapsto " + bytecodes.elementAt(i).toModelString());
			if (i < bytecodes.size()-1) {
				buffer.append(", \\\\\n");
			} else {
				buffer.append(" \\\\\n");
			}
		}
		buffer.append("\t\\} \n");
		buffer.append("\\end{axdef}\n\n");
		
		buffer.append("\\begin{axdef}\n");
		buffer.append("\tclasses : ClassID \\pfun Class\n");
		buffer.append("\\where\n");
		buffer.append("\tclasses = \\{ \\\\\n");
		Iterator<ClassModel> classIterator = classes.iterator();
		while (classIterator.hasNext()) {
			ClassID className = classIterator.next().getClassName();
			buffer.append("\t\t\\t1 " + className + "ID \\mapsto " + className);
			
			if (classIterator.hasNext()) {
				buffer.append(", \\\\\n");
			} else {
				buffer.append(" \\\\\n");
			}
		}
		buffer.append("\t\\} \n");
		buffer.append("\\end{axdef}\n");
		
		
		return buffer.toString();
	}
	
	//public static final String resumeThreadClass = "ManagedSchedulable";
		//public static final String suspendClass = "ManagedSchedulable";
		public static final ClassID setPriorityCeilingClass = new ClassID("javax.safetycritical.Services");
		public static final ClassID writeClass = new ClassID("javax.safetycritical.io.ConsoleOutput");
		public static final ClassID readClass = new ClassID("javax.safetycritical.io.ConsoleInput");
		public static final ClassID managedSchedulableClass = new ClassID("javax.safetycritical.ManagedEventHandler");
		public static final ClassID managedMemoryClass = new ClassID("javax.safetycritical.ManagedMemory");
		public static final ClassID aperiodicEventHandlerClass = new ClassID("javax.safetycritical.AperiodicEventHandler");
		public static final ClassID periodicEventHandlerClass = new ClassID("javax.safetycritical.PeriodicEventHandler");
		public static final ClassID oneShotEventHandlerClass = new ClassID("javax.safetycritical.OneShotEventHandler");

		//public static final String resumeThreadID = "";
		//public static final String suspendID = "";
		public static final MethodID setPriorityCeilingID = new MethodID("setPriorityCeiling", "(Ljava/lang/Object;I)V");
		public static final MethodID writeID = new MethodID("writeNative", "(I)V");
		public static final MethodID readID = new MethodID("readNative", "()I");
		public static final MethodID registerID = new MethodID("register", "(Ljavax/safetycritical/ManagedEventHandler;)V");
		public static final MethodID enterPrivateMemoryHelperID = new MethodID("enterPrivateMemory", "(I)V");
		public static final MethodID executeInAreaOfHelperID = new MethodID("executeInAreaOf", "(Ljava/lang/Object;)V");
		public static final MethodID executeInOuterAreaHelperID = new MethodID("executeInOuterArea", "()V");
		public static final MethodID exitMemoryID = new MethodID("exitMemory", "()V");
		public static final MethodID releaseAperiodicID = new MethodID("releaseAperiodic", "(Ljavax/safetycritical/AperiodicEventHandler;)V");
		public static final MethodID initAPEHID = new MethodID("initAPEH", "(Ljavax/safetycritical/AperiodicEventHandler;IIII)V");
		public static final MethodID initPEHID = new MethodID("initPEH", "(Ljavax/safetycritical/PeriodicEventHandler;IIIIII)V");
		public static final MethodID initOSEHAbsID = new MethodID("initOSEHAbs", "(Ljavax/safetycritical/OneShotEventHandler;IIIII)V");
		public static final MethodID initOSEHRelID = new MethodID("initOSEHRel", "(Ljavax/safetycritical/OneShotEventHandler;IIIII)V");
	
	public ThrCFModel doEliminationOfProgramCounter() {
		HashMap<ClassID,ClassModel> classMap = new HashMap<ClassID,ClassModel>();
		for (ClassModel classModel : classes) {
			classMap.put(classModel.getClassName(), classModel);
		}
		
		ThrCFModel newModel = new ThrCFModel(classNameOrdering, methodNameOrdering, fieldNameOrdering, classMap);
		
		// set up a dependency checker for the program
		MethodDependencyChecker dependencyChecker = new MethodDependencyChecker(classMap);
		for (ClassModel classModel : classes) {
			for (MethodID methodID : classModel.getMethods()) {
				BytecodeModel[] methodBytecodes = bytecodes.subList(
						classModel.getMethodEntry(methodID),
						classModel.getMethodEnd(methodID)).toArray(new BytecodeModel[] {});
				dependencyChecker.addMethod(classModel.getClassName(), methodID, methodBytecodes);
			}
		}
		
		// check the program isn't using recursion
		if (dependencyChecker.hasDependencyLoop()) {
			throw new IllegalArgumentException("The program must not use recursion (direct or indirect)");
		}
		
		HashSet<ClassID> instantiatedClasses = new HashSet<ClassID>();
		
		// bytecode expansion
		System.out.println("Performing bytecode expansion...");
//		long startTime = System.currentTimeMillis();
		CircusAction[][] runningChoices = new CircusAction[bytecodes.size()][];
		for (int pc = 0; pc < bytecodes.size(); pc++) {
			BytecodeModel bytecode = bytecodes.get(pc);
			if (bytecode instanceof BytecodeModel.ACONST_NULL) {
				runningChoices[pc] = new CircusAction[2];
				runningChoices[pc][0] = new CircusAction.HandleAconst_nullEPC();
				runningChoices[pc][1] = new CircusAction.PCAssignment(pc+1);
			} else if (bytecode instanceof BytecodeModel.DUP) {
				runningChoices[pc] = new CircusAction[2];
				runningChoices[pc][0] = new CircusAction.HandleDupEPC();
				runningChoices[pc][1] = new CircusAction.PCAssignment(pc+1);
			} else if (bytecode instanceof BytecodeModel.ARETURN) {
				ClassModel currentClass = getClassFromPC(pc);
				if (currentClass.isSynchronized(getClassMethodIDFromPC(currentClass, pc))) {
					runningChoices[pc] = new CircusAction[2];
					runningChoices[pc][0] = 
							new CircusAction.ChannelComm("releaseLock", new String[] {"((last~frameStack).localVariables~1)"}, new boolean[] {false}, 
							new CircusAction.ChannelComm("releaseLockRet", null, null, new CircusAction.Skip()));
					runningChoices[pc][1] = new CircusAction.HandleAreturnEPC();
				} else {
					runningChoices[pc] = new CircusAction[1];
					runningChoices[pc][0] = new CircusAction.HandleAreturnEPC();
				}
			} else if (bytecode instanceof BytecodeModel.RETURN) {
				ClassModel currentClass = getClassFromPC(pc);
				if (currentClass.isSynchronized(getClassMethodIDFromPC(currentClass, pc))) {
					runningChoices[pc] = new CircusAction[2];
					runningChoices[pc][0] = 
							new CircusAction.ChannelComm("releaseLock", new String[] {"((last~frameStack).localVariables~1)"}, new boolean[] {false}, 
							new CircusAction.ChannelComm("releaseLockRet", null, null, new CircusAction.Skip()));
					runningChoices[pc][1] = new CircusAction.HandleReturnEPC();
				} else {
					runningChoices[pc] = new CircusAction[1];
					runningChoices[pc][0] = new CircusAction.HandleReturnEPC();
				}
			} else if (bytecode instanceof BytecodeModel.IADD) {
				runningChoices[pc] = new CircusAction[2];
				runningChoices[pc][0] = new CircusAction.HandleIaddEPC();
				runningChoices[pc][1] = new CircusAction.PCAssignment(pc+1);
			} else if (bytecode instanceof BytecodeModel.INEG) {
				runningChoices[pc] = new CircusAction[2];
				runningChoices[pc][0] = new CircusAction.HandleInegEPC();
				runningChoices[pc][1] = new CircusAction.PCAssignment(pc+1);
			} else if (bytecode instanceof BytecodeModel.NEW) {
				int classIndex = ((BytecodeModel.NEW) bytecode).getClassIndex();
				runningChoices[pc] = new CircusAction[2];
				runningChoices[pc][0] = new CircusAction.HandleNewEPC(classIndex);
				runningChoices[pc][1] = new CircusAction.PCAssignment(pc+1);
				ClassID classID = ((CPEntry.ClassRef) getClassFromPC(pc).getCPEntry(classIndex)).getClassID();
				if (!classMap.get(classID).isAbstract()) {
					instantiatedClasses.add(classID);
				}
			} else if (bytecode instanceof BytecodeModel.ICONST) {
				runningChoices[pc] = new CircusAction[2];
				runningChoices[pc][0] = new CircusAction.HandleIconstEPC(((BytecodeModel.ICONST) bytecode).getConstantValue());
				runningChoices[pc][1] = new CircusAction.PCAssignment(pc+1);
			} else if (bytecode instanceof BytecodeModel.ALOAD) {
				runningChoices[pc] = new CircusAction[2];
				runningChoices[pc][0] = new CircusAction.HandleAloadEPC(((BytecodeModel.ALOAD) bytecode).getVariableIndex());
				runningChoices[pc][1] = new CircusAction.PCAssignment(pc+1);
			} else if (bytecode instanceof BytecodeModel.ASTORE) {
				runningChoices[pc] = new CircusAction[2];
				runningChoices[pc][0] = new CircusAction.HandleAstoreEPC(((BytecodeModel.ASTORE) bytecode).getVariableIndex());
				runningChoices[pc][1] = new CircusAction.PCAssignment(pc+1);
			} else if (bytecode instanceof BytecodeModel.GETFIELD) {
				runningChoices[pc] = new CircusAction[2];
				runningChoices[pc][0] = new CircusAction.HandleGetfieldEPC(((BytecodeModel.GETFIELD) bytecode).getFieldIndex());
				runningChoices[pc][1] = new CircusAction.PCAssignment(pc+1);
			} else if (bytecode instanceof BytecodeModel.PUTFIELD) {
				runningChoices[pc] = new CircusAction[2];
				runningChoices[pc][0] = new CircusAction.HandlePutfieldEPC(((BytecodeModel.PUTFIELD) bytecode).getFieldIndex());
				runningChoices[pc][1] = new CircusAction.PCAssignment(pc+1);
			} else if (bytecode instanceof BytecodeModel.GETSTATIC) {
				runningChoices[pc] = new CircusAction[2];
				runningChoices[pc][0] = new CircusAction.HandleGetstaticEPC(((BytecodeModel.GETSTATIC) bytecode).getFieldIndex());
				runningChoices[pc][1] = new CircusAction.PCAssignment(pc+1);
			} else if (bytecode instanceof BytecodeModel.PUTSTATIC) {
				runningChoices[pc] = new CircusAction[2];
				runningChoices[pc][0] = new CircusAction.HandlePutstaticEPC(((BytecodeModel.PUTSTATIC) bytecode).getFieldIndex());
				runningChoices[pc][1] = new CircusAction.PCAssignment(pc+1);
			} else if (bytecode instanceof BytecodeModel.INVOKESPECIAL) {
				runningChoices[pc] = new CircusAction[2];
				runningChoices[pc][0] = new CircusAction.PCAssumption(pc);
				runningChoices[pc][1] = new CircusAction.HandleInvokespecialEPC(((BytecodeModel.INVOKESPECIAL) bytecode).getMethodIndex());
			} else if (bytecode instanceof BytecodeModel.INVOKESTATIC) {
				runningChoices[pc] = new CircusAction[2];
				runningChoices[pc][0] = new CircusAction.PCAssumption(pc);
				runningChoices[pc][1] = new CircusAction.HandleInvokestaticEPC(((BytecodeModel.INVOKESTATIC) bytecode).getMethodIndex());
			} else if (bytecode instanceof BytecodeModel.INVOKEVIRTUAL) {
				runningChoices[pc] = new CircusAction[2];
				runningChoices[pc][0] = new CircusAction.PCAssumption(pc);
				runningChoices[pc][1] = new CircusAction.HandleInvokevirtualEPC(((BytecodeModel.INVOKEVIRTUAL) bytecode).getMethodIndex());
			} else if (bytecode instanceof BytecodeModel.IF_ICMPLE) {
				runningChoices[pc] = new CircusAction[1];
				CircusAction[] varBlockActions = new CircusAction[3];
				varBlockActions[0] = new CircusAction.InterpreterPopEPC("value2");
				varBlockActions[1] = new CircusAction.InterpreterPopEPC("value1");
				varBlockActions[2] = new CircusAction.ConditionalPCAssignment(
						"value1 \\leq value2",
						pc + (((BytecodeModel.IF_ICMPLE) bytecode).getOffset()),
						pc + 1);
				runningChoices[pc][0] = new CircusAction.VarBlock("value1, value2", varBlockActions); 
			} else if (bytecode instanceof BytecodeModel.GOTO) {
				runningChoices[pc] = new CircusAction[1];
				runningChoices[pc][0] = new CircusAction.PCAssignment(pc + ((BytecodeModel.GOTO) bytecode).getOffset());
			} else if (bytecode instanceof BytecodeModel.LLOAD) {
				runningChoices[pc] = new CircusAction[2];
				runningChoices[pc][0] = new CircusAction.HandleLloadEPC(((BytecodeModel.LLOAD) bytecode).getVariableIndex());
				runningChoices[pc][1] = new CircusAction.PCAssignment(pc+1);
			} else if (bytecode instanceof BytecodeModel.LSTORE) {
				runningChoices[pc] = new CircusAction[2];
				runningChoices[pc][0] = new CircusAction.HandleLstoreEPC(((BytecodeModel.LSTORE) bytecode).getVariableIndex());
				runningChoices[pc][1] = new CircusAction.PCAssignment(pc+1);
			} else if (bytecode instanceof BytecodeModel.LADD) {
				runningChoices[pc] = new CircusAction[2];
				runningChoices[pc][0] = new CircusAction.HandleLaddEPC();
				runningChoices[pc][1] = new CircusAction.PCAssignment(pc+1);
			} else if (bytecode instanceof BytecodeModel.L2I) {
				runningChoices[pc] = new CircusAction[2];
				runningChoices[pc][0] = new CircusAction.HandleLtoIEPC();
				runningChoices[pc][1] = new CircusAction.PCAssignment(pc+1);
			} else if (bytecode instanceof BytecodeModel.I2L) {
				// push a zero lsb byte onto the stack
				runningChoices[pc] = new CircusAction[2];
				runningChoices[pc][0] = new CircusAction.HandleIconstEPC(0);
				runningChoices[pc][1] = new CircusAction.PCAssignment(pc+1);
			} else if (bytecode instanceof BytecodeModel.IINC) {
				BytecodeModel.IINC instr = (BytecodeModel.IINC) bytecode;
				runningChoices[pc] = new CircusAction[2];
				runningChoices[pc][0] = new CircusAction.HandleIincEPC(instr.getVariableIndex(), instr.getConstantValue());
				runningChoices[pc][1] = new CircusAction.PCAssignment(pc+1);
			} else if (bytecode instanceof BytecodeModel.LCONST) {
				runningChoices[pc] = new CircusAction[2];
				runningChoices[pc][0] = new CircusAction.HandleLconstEPC(((BytecodeModel.LCONST) bytecode).getConstantValue());
				runningChoices[pc][1] = new CircusAction.PCAssignment(pc+1);
			} else if (bytecode instanceof BytecodeModel.LRETURN) {
				ClassModel currentClass = getClassFromPC(pc);
				if (currentClass.isSynchronized(getClassMethodIDFromPC(currentClass, pc))) {
					runningChoices[pc] = new CircusAction[2];
					runningChoices[pc][0] = 
							new CircusAction.ChannelComm("releaseLock", new String[] {"((last~frameStack).localVariables~1)"}, new boolean[] {false}, 
							new CircusAction.ChannelComm("releaseLockRet", null, null, new CircusAction.Skip()));
					runningChoices[pc][1] = new CircusAction.HandleLreturnEPC();
				} else {
					runningChoices[pc] = new CircusAction[1];
					runningChoices[pc][0] = new CircusAction.HandleLreturnEPC();
				}
			} else if (bytecode instanceof BytecodeModel.If_icmpInstruction) {
				runningChoices[pc] = new CircusAction[1];
				CircusAction[] varBlockActions = new CircusAction[3];
				varBlockActions[0] = new CircusAction.InterpreterPopEPC("value2");
				varBlockActions[1] = new CircusAction.InterpreterPopEPC("value1");
				varBlockActions[2] = new CircusAction.ConditionalPCAssignment(
						"value1 " + ((BytecodeModel.If_icmpInstruction) bytecode).comparisonOperator() +" value2",
						pc +  ((BytecodeModel.If_icmpInstruction) bytecode).getOffset(),
						pc + 1);
				runningChoices[pc][0] = new CircusAction.VarBlock("value1, value2", varBlockActions);
			} else if (bytecode instanceof BytecodeModel.IfInstruction) {
				runningChoices[pc] = new CircusAction[1];
				CircusAction[] varBlockActions = new CircusAction[2];
				varBlockActions[0] = new CircusAction.InterpreterPopEPC("value");
				varBlockActions[1] = new CircusAction.ConditionalPCAssignment(
						"value " + ((BytecodeModel.IfInstruction) bytecode).comparisonOperator() +" 0",
						pc +  ((BytecodeModel.IfInstruction) bytecode).getOffset(),
						pc + 1);
				runningChoices[pc][0] = new CircusAction.VarBlock("value", varBlockActions);
			} else if (bytecode instanceof BytecodeModel.BinOpInstruction) {
				runningChoices[pc] = new CircusAction[2];
				runningChoices[pc][0] = new CircusAction.HandleBinOpEPC(((BytecodeModel.BinOpInstruction) bytecode).operator());
				runningChoices[pc][1] = new CircusAction.PCAssignment(pc+1);
			} else {
				throw new UnsupportedOperationException("Unhandled ModelByteCode: " + bytecode + " at " + pc);
			}
		}
//		long endTime = System.currentTimeMillis();
//		System.out.println("time elapsed: " + (endTime-startTime));
		
		//System.out.println("Instantiated classes: " + instantiatedClasses);
		
		System.out.println("Performing sequential composition introduction...");
//		startTime = System.currentTimeMillis();
		// sequential composition introduction
		ControlFlowGraph cfg = new ControlFlowGraph(runningChoices, classMap);
		for (int node = 0; node < runningChoices.length; node++) {
			while (cfg.hasSimpleSequence(node)) {
				CircusAction[] actions = runningChoices[node];
				CircusAction lastAction = actions[actions.length-1];
				if (lastAction instanceof CircusAction.PCAssignment) {
					int nextPC = ((CircusAction.PCAssignment) lastAction).getTarget();
					CircusAction[] nextActions = runningChoices[nextPC];
					Vector<CircusAction> newActions = new Vector<CircusAction>();
					newActions.addAll(Arrays.asList(actions));
					newActions.add(new CircusAction.Poll());
					newActions.addAll(Arrays.asList(nextActions));
					runningChoices[node] = newActions.toArray(new CircusAction[] {});
					cfg = new ControlFlowGraph(runningChoices, classMap);
					//System.out.println("Sequential composition introduced at pc = " + node);
				} else {
					break;
				}
			}
		}
//		endTime = System.currentTimeMillis();
//		System.out.println("time elapsed: " + (endTime-startTime));
		//System.out.println("Sequential composition introduction complete!");
		
		// iterate in dependency order
		//FullMethodID[] methodOrder = dependencyChecker.dependencyOrder();
		int methodLoopCount = 0;
		while (!allMethodsSeparated(newModel)) {
			methodLoopCount++;
			// introduce loops and conditionals
			System.out.println("Introducing loops and conditionals(pass " + methodLoopCount + ")...");
//			startTime = System.currentTimeMillis();
			for (FullMethodID currentMethod : methods()) {
				int entryPoint = classMap.get(currentMethod.classID).getMethodEntry(currentMethod.methodID);
				if (hasUnresolvedCalls(currentMethod, classMap, cfg, runningChoices) || cfg.isComplete(entryPoint)) {
					continue;
				}
				//System.out.println("Introducing loops and conditionals to " + currentMethod + "...");
				int recVarID = 0;
				CircusAction lastActionAtCurrentNode = runningChoices[entryPoint][runningChoices[entryPoint].length-1];
				//System.out.println("Last action at " + entryPoint + ": " + lastActionAtCurrentNode);
				while (lastActionAtCurrentNode instanceof CircusAction.VarBlock || lastActionAtCurrentNode instanceof CircusAction.ChannelComm) {
					if (lastActionAtCurrentNode instanceof CircusAction.VarBlock) {
						CircusAction[] actions = ((CircusAction.VarBlock) lastActionAtCurrentNode).getActions();
						lastActionAtCurrentNode = actions[actions.length-1];
					} else if (lastActionAtCurrentNode instanceof CircusAction.ChannelComm) {
						lastActionAtCurrentNode = ((CircusAction.ChannelComm) lastActionAtCurrentNode).getAction();
					}
				}
				Integer[] iterationOrder = cfg.getReverseNodeTraversalOrder(entryPoint);
				for (int node : iterationOrder) {
					//System.out.println("Introducing loops and conditionals to node pc = " + node + "...");
//					System.out.println("node before:\n" + CircusAction.actionsToString(runningChoices[node], 0));
					
					// apply Rule [if-conditional-intro]
					CircusAction[] actions = runningChoices[node];
					CircusAction lastAction = actions[actions.length-1];
					if (lastAction instanceof CircusAction.VarBlock) {
						CircusAction[] varBlockActions = ((CircusAction.VarBlock) lastAction).getActions();
						String vars = ((CircusAction.VarBlock) lastAction).getVarName();
						lastAction = varBlockActions[varBlockActions.length-1];
						if (lastAction instanceof CircusAction.ConditionalPCAssignment) {
							int j = ((CircusAction.ConditionalPCAssignment) lastAction).getTrueTarget();
							int k = ((CircusAction.ConditionalPCAssignment) lastAction).getFalseTarget();
							if (node != j && node != k) {
								CircusAction[] branchActions = runningChoices[k];
								CircusAction branchLastAction = branchActions[branchActions.length-1];
								if (branchLastAction instanceof CircusAction.PCAssignment) {
									if (((CircusAction.PCAssignment) branchLastAction).getTarget() == j) {
										CircusAction[] trueActions = new CircusAction[1];
										trueActions[0] = new CircusAction.Skip();
										CircusAction[] falseActions = new CircusAction[2 + branchActions.length-1];
										falseActions[0] = new CircusAction.PCAssignment(k);
										falseActions[1] = new CircusAction.Poll();
										for (int x = 0; x < branchActions.length-1; x++) {
											falseActions[2+x] = branchActions[x];
										}
									
										// build LHS of rule
										Vector<CircusAction> newActions = new Vector<CircusAction>();
										for (int x = 0; x < actions.length-1; x++) {
											newActions.add(actions[x]);
										}
										String trueCondition = ((CircusAction.ConditionalPCAssignment) lastAction).getCondition();
										String falseCondition = new StringBuilder().append("\\lnot ").append(trueCondition).toString();
										varBlockActions[varBlockActions.length-1] = new CircusAction.Conditional(
												new String[] {trueCondition, falseCondition}, 
												new CircusAction[][] {trueActions, falseActions});
										newActions.add(new CircusAction.VarBlock(vars, varBlockActions));
//										newActions.add(new CircusAction.Conditional(
//												new String[] {trueCondition, falseCondition}, 
//												new CircusAction[][] {trueActions, falseActions}));
										newActions.add(new CircusAction.PCAssignment(j));
										runningChoices[node] = newActions.toArray(new CircusAction[] {});
										cfg = new ControlFlowGraph(runningChoices, classMap);
										//System.out.println("Rule [if-conditional-intro] applied at pc = " + node);
									}	
								}
							}
						}
					}
				
					// apply Rule [if-else-conditional-intro]
					actions = runningChoices[node];
					lastAction = actions[actions.length-1];
					if (lastAction instanceof CircusAction.VarBlock) {
						CircusAction[] varBlockActions = ((CircusAction.VarBlock) lastAction).getActions();
						String vars = ((CircusAction.VarBlock) lastAction).getVarName();
						lastAction = varBlockActions[varBlockActions.length-1];
						if (lastAction instanceof CircusAction.ConditionalPCAssignment) {
							int j = ((CircusAction.ConditionalPCAssignment) lastAction).getTrueTarget();
							int k = ((CircusAction.ConditionalPCAssignment) lastAction).getFalseTarget();
							if (node != j && node != k) {
								CircusAction[] branch1Actions = runningChoices[j];
								CircusAction branch1LastAction = branch1Actions[branch1Actions.length-1];
								if (branch1LastAction instanceof CircusAction.PCAssignment) {
									CircusAction[] branch2Actions = runningChoices[k];
									CircusAction branch2LastAction = branch2Actions[branch2Actions.length-1];
									int x = ((CircusAction.PCAssignment) branch1LastAction).getTarget();
									if (branch2LastAction instanceof CircusAction.PCAssignment) {
										if (((CircusAction.PCAssignment) branch2LastAction).getTarget() == x) {
											CircusAction[] trueActions = new CircusAction[2 + branch1Actions.length-1];
											trueActions[0] = new CircusAction.PCAssignment(j);
											trueActions[1] = new CircusAction.Poll();
											for (int y = 0; y < branch1Actions.length-1; y++) {
												trueActions[2+y] = branch1Actions[y];
											}
											CircusAction[] falseActions = new CircusAction[2 + branch2Actions.length-1];
											falseActions[0] = new CircusAction.PCAssignment(k);
											falseActions[1] = new CircusAction.Poll();
											for (int y = 0; y < branch2Actions.length-1; y++) {
												falseActions[2+y] = branch2Actions[y];
											}
										
											// build LHS of rule
											Vector<CircusAction> newActions = new Vector<CircusAction>();
											for (int y = 0; y < actions.length-1; y++) {
												newActions.add(actions[y]);
											}
											String trueCondition = ((CircusAction.ConditionalPCAssignment) lastAction).getCondition();
											String falseCondition = new StringBuilder().append("\\lnot ").append(trueCondition).toString();
											varBlockActions[varBlockActions.length-1] = new CircusAction.Conditional(
													new String[] {trueCondition, falseCondition}, 
													new CircusAction[][] {trueActions, falseActions});
											newActions.add(new CircusAction.VarBlock(vars, varBlockActions));
											newActions.add(new CircusAction.PCAssignment(x));
											runningChoices[node] = newActions.toArray(new CircusAction[] {});
											cfg = new ControlFlowGraph(runningChoices, classMap);
											//System.out.println("Rule [if-else-conditional-intro] applied at pc = " + node);
										}
									}
								}
							}
						}
					}
					
					// apply Rule [conditional-intro] if simple conditional
					if (cfg.hasSimpleConditional(node)) {
						actions = runningChoices[node];
						lastAction = actions[actions.length-1];
						if (lastAction instanceof CircusAction.VarBlock) {
							CircusAction[] varBlockActions = ((CircusAction.VarBlock) lastAction).getActions();
							String vars = ((CircusAction.VarBlock) lastAction).getVarName();
							lastAction = varBlockActions[varBlockActions.length-1];
							if (lastAction instanceof CircusAction.ConditionalPCAssignment) {
								int j = ((CircusAction.ConditionalPCAssignment) lastAction).getTrueTarget();
								int k = ((CircusAction.ConditionalPCAssignment) lastAction).getFalseTarget();
								if (node != j && node != k) {
									CircusAction[] branch1Actions = runningChoices[j];
									CircusAction[] branch2Actions = runningChoices[k];
									CircusAction[] trueActions = new CircusAction[2 + branch1Actions.length];
									trueActions[0] = new CircusAction.PCAssignment(j);
									trueActions[1] = new CircusAction.Poll();
									for (int y = 0; y < branch1Actions.length; y++) {
										trueActions[2+y] = branch1Actions[y];
									}
									CircusAction[] falseActions = new CircusAction[2 + branch2Actions.length];
									falseActions[0] = new CircusAction.PCAssignment(k);
									falseActions[1] = new CircusAction.Poll();
									for (int y = 0; y < branch2Actions.length; y++) {
										falseActions[2+y] = branch2Actions[y];
									}
									
									// build LHS of rule
									Vector<CircusAction> newActions = new Vector<CircusAction>();
									for (int y = 0; y < actions.length-1; y++) {
										newActions.add(actions[y]);
									}
									String trueCondition = ((CircusAction.ConditionalPCAssignment) lastAction).getCondition();
									String falseCondition = new StringBuilder().append("\\lnot ").append(trueCondition).toString();
									varBlockActions[varBlockActions.length-1] = new CircusAction.Conditional(
											new String[] {trueCondition, falseCondition}, 
											new CircusAction[][] {trueActions, falseActions});
									newActions.add(new CircusAction.VarBlock(vars, varBlockActions));
									runningChoices[node] = newActions.toArray(new CircusAction[] {});
									cfg = new ControlFlowGraph(runningChoices, classMap);
									//System.out.println("Rule [conditional-intro] applied at pc = " + node);
								}
							}
						}
					}
					
					// apply Rule [while-loop-intro1]
					actions = runningChoices[node];
					lastAction = actions[actions.length-1];
					if (lastAction instanceof CircusAction.VarBlock) {
						CircusAction[] varBlockActions = ((CircusAction.VarBlock) lastAction).getActions();
						String vars = ((CircusAction.VarBlock) lastAction).getVarName();
						lastAction = varBlockActions[varBlockActions.length-1];
						if (lastAction instanceof CircusAction.ConditionalPCAssignment) {
							int j = ((CircusAction.ConditionalPCAssignment) lastAction).getTrueTarget();
							int k = ((CircusAction.ConditionalPCAssignment) lastAction).getFalseTarget();
							if (node != j && node != k) {
								CircusAction[] branchActions = runningChoices[k];
								CircusAction branchLastAction = branchActions[branchActions.length-1];
								if (branchLastAction instanceof CircusAction.PCAssignment) {
									if (((CircusAction.PCAssignment) branchLastAction).getTarget() == node) {
										CircusAction[] trueActions = new CircusAction[1];
										trueActions[0] = new CircusAction.Skip();
										CircusAction[] falseActions = new CircusAction[5 + branchActions.length-1];
										falseActions[0] = new CircusAction.PCAssignment(k);
										falseActions[1] = new CircusAction.Poll();
										for (int x = 0; x < branchActions.length-1; x++) {
											falseActions[2+x] = branchActions[x];
										}
										falseActions[2+branchActions.length-1] = new CircusAction.PCAssignment(node);
										falseActions[2+branchActions.length-1+1] = new CircusAction.Poll();
										falseActions[2+branchActions.length-1+2] = new CircusAction.ActionRef("Y"+recVarID);
										
										// build LHS of rule
										Vector<CircusAction> recursionActions = new Vector<CircusAction>();
										for (int x = 0; x < actions.length-1; x++) {
											recursionActions.add(actions[x]);
										}
										
										String trueCondition = ((CircusAction.ConditionalPCAssignment) lastAction).getCondition();
										String falseCondition = new StringBuilder().append("\\lnot ").append(trueCondition).toString();
										varBlockActions[varBlockActions.length-1] = new CircusAction.Conditional(
												new String[] {trueCondition, falseCondition}, 
												new CircusAction[][] {trueActions, falseActions});
										recursionActions.add(new CircusAction.VarBlock(vars, varBlockActions));
										Vector<CircusAction> newActions = new Vector<CircusAction>();
										newActions.add(new CircusAction.Recursion("Y"+recVarID, recursionActions.toArray(new CircusAction[] {})));
										newActions.add(new CircusAction.PCAssignment(j));
										runningChoices[node] = newActions.toArray(new CircusAction[] {});
										cfg = new ControlFlowGraph(runningChoices, classMap);
										recVarID++;
										//System.out.println("Rule [while-loop-intro1] applied at pc = " + node);
									}
								}
							}
						}
					}
					
					// apply Rule [while-loop-intro2]
					actions = runningChoices[node];
					lastAction = actions[actions.length-1];
					if (lastAction instanceof CircusAction.VarBlock) {
						CircusAction[] varBlockActions = ((CircusAction.VarBlock) lastAction).getActions();
						String vars = ((CircusAction.VarBlock) lastAction).getVarName();
						lastAction = varBlockActions[varBlockActions.length-1];
						if (lastAction instanceof CircusAction.ConditionalPCAssignment) {
							int j = ((CircusAction.ConditionalPCAssignment) lastAction).getTrueTarget();
							int k = ((CircusAction.ConditionalPCAssignment) lastAction).getFalseTarget();
							if (node != j && node != k) {
								CircusAction[] branchActions = runningChoices[j];
								CircusAction branchLastAction = branchActions[branchActions.length-1];
								if (branchLastAction instanceof CircusAction.PCAssignment) {
									if (((CircusAction.PCAssignment) branchLastAction).getTarget() == node) {
										CircusAction[] trueActions = new CircusAction[5 + branchActions.length-1];
										trueActions[0] = new CircusAction.PCAssignment(j);
										trueActions[1] = new CircusAction.Poll();
										for (int x = 0; x < branchActions.length-1; x++) {
											trueActions[2+x] = branchActions[x];
										}
										trueActions[2+branchActions.length-1] = new CircusAction.PCAssignment(node);
										trueActions[2+branchActions.length-1+1] = new CircusAction.Poll();
										trueActions[2+branchActions.length-1+2] = new CircusAction.ActionRef("Y"+recVarID);
										CircusAction[] falseActions = new CircusAction[1];
										falseActions[0] = new CircusAction.Skip();
										
										
										// build LHS of rule
										Vector<CircusAction> recursionActions = new Vector<CircusAction>();
										for (int x = 0; x < actions.length-1; x++) {
											recursionActions.add(actions[x]);
										}
										String trueCondition = ((CircusAction.ConditionalPCAssignment) lastAction).getCondition();
										String falseCondition = new StringBuilder().append("\\lnot ").append(trueCondition).toString();
										varBlockActions[varBlockActions.length-1] = new CircusAction.Conditional(
												new String[] {trueCondition, falseCondition}, 
												new CircusAction[][] {trueActions, falseActions});
										recursionActions.add(new CircusAction.VarBlock(vars, varBlockActions));
										Vector<CircusAction> newActions = new Vector<CircusAction>();
										newActions.add(new CircusAction.Recursion("Y"+recVarID, recursionActions.toArray(new CircusAction[] {})));
										newActions.add(new CircusAction.PCAssignment(k));
										runningChoices[node] = newActions.toArray(new CircusAction[] {});
										cfg = new ControlFlowGraph(runningChoices, classMap);
										recVarID++;
										//System.out.println("Rule [while-loop-intro2] applied at pc = " + node);
									}
								}
							}
						}
					}
					
					// apply Rule [do-while-loop-intro]
					actions = runningChoices[node];
					lastAction = actions[actions.length-1];
					if (lastAction instanceof CircusAction.VarBlock) {
						CircusAction[] varBlockActions = ((CircusAction.VarBlock) lastAction).getActions();
						String vars = ((CircusAction.VarBlock) lastAction).getVarName();
						lastAction = varBlockActions[varBlockActions.length-1];
						if (lastAction instanceof CircusAction.ConditionalPCAssignment) {
							int i = ((CircusAction.ConditionalPCAssignment) lastAction).getTrueTarget();
							int j = ((CircusAction.ConditionalPCAssignment) lastAction).getFalseTarget();
							if (node == i && node != j) {
								CircusAction[] trueActions = new CircusAction[3];
								trueActions[0] = new CircusAction.PCAssignment(node);
								trueActions[1] = new CircusAction.Poll();
								trueActions[2] = new CircusAction.ActionRef("Y"+recVarID);
								CircusAction[] falseActions = new CircusAction[1];
								falseActions[0] = new CircusAction.Skip();
								
								// build LHS of rule
								Vector<CircusAction> recursionActions = new Vector<CircusAction>();
								for (int x = 0; x < actions.length-1; x++) {
									recursionActions.add(actions[x]);
								}
								String trueCondition = ((CircusAction.ConditionalPCAssignment) lastAction).getCondition();
								String falseCondition = new StringBuilder().append("\\lnot ").append(trueCondition).toString();
								varBlockActions[varBlockActions.length-1] = new CircusAction.Conditional(
										new String[] {trueCondition, falseCondition}, 
										new CircusAction[][] {trueActions, falseActions});
								recursionActions.add(new CircusAction.VarBlock(vars, varBlockActions));
								Vector<CircusAction> newActions = new Vector<CircusAction>();
								newActions.add(new CircusAction.Recursion("Y"+recVarID, recursionActions.toArray(new CircusAction[] {})));
								newActions.add(new CircusAction.PCAssignment(j));
								runningChoices[node] = newActions.toArray(new CircusAction[] {});
								cfg = new ControlFlowGraph(runningChoices, classMap);
								recVarID++;
								//System.out.println("Rule [do-while-loop-intro] applied at pc = " + node);
							}
						}
					}
					
					// apply Rule [infinite-loop-intro]
					actions = runningChoices[node];
					lastAction = actions[actions.length-1];
					if (lastAction instanceof CircusAction.PCAssignment) {
						int i = ((CircusAction.PCAssignment) lastAction).getTarget();
						if (node == i) {
							
							
							// build LHS of rule
							Vector<CircusAction> recursionActions = new Vector<CircusAction>();
							for (int x = 0; x < actions.length-1; x++) {
								recursionActions.add(actions[x]);
							}
							recursionActions.add(new CircusAction.PCAssignment(node));
							recursionActions.add(new CircusAction.Poll());
							recursionActions.add(new CircusAction.ActionRef("Y"+recVarID));
							runningChoices[node] = new CircusAction[] {
									new CircusAction.Recursion("Y"+recVarID, recursionActions.toArray(new CircusAction[] {}))
							};
							cfg = new ControlFlowGraph(runningChoices, classMap);
							recVarID++;
							//System.out.println("Rule [infinite-loop-intro] applied at pc = " + node);
						}
					}
			
					// apply Rule [sequence-intro] if simple sequence
					if (cfg.hasSimpleSequence(node)) {
						actions = runningChoices[node];
						lastAction = actions[actions.length-1];
						if (lastAction instanceof CircusAction.PCAssignment) {
							int nextPC = ((CircusAction.PCAssignment) lastAction).getTarget();
							CircusAction[] nextActions = runningChoices[nextPC];
							Vector<CircusAction> newActions = new Vector<CircusAction>();
							newActions.addAll(Arrays.asList(actions));
							newActions.add(new CircusAction.Poll());
							newActions.addAll(Arrays.asList(nextActions));
							runningChoices[node] = newActions.toArray(new CircusAction[] {});
							cfg = new ControlFlowGraph(runningChoices, classMap);
//							System.out.println("Rule [sequence-intro] applied at pc = " + node);
						}
					}
					//System.out.println("node after:\n" + CircusAction.actionsToString(runningChoices[node], 0));
				}
//				if (currentMethod.classID.equals(new ClassID("main.Producer"))) {
//					try {
//						System.in.read();
//					} catch (IOException e) {
//						e.printStackTrace();
//					}
//				}
			}
//			endTime = System.currentTimeMillis();
//			System.out.println("time elapsed: " + (endTime-startTime));
			
			System.out.println("Separating methods(pass " + methodLoopCount + ")...");
//			startTime = System.currentTimeMillis();
			// method separation
			for (FullMethodID currentMethod : methods()) {
				int entryPoint = classMap.get(currentMethod.classID).getMethodEntry(currentMethod.methodID);
				if (cfg.isComplete(entryPoint) && !newModel.hasMethod(currentMethod)) {
					newModel.addMethod(currentMethod, runningChoices[entryPoint]);
					runningChoices[entryPoint] = new CircusAction[] {
						new CircusAction.MethodActionRef(currentMethod, null)
					};
//					System.out.println(currentMethod + " separated");
				}
			}
//			endTime = System.currentTimeMillis();
//			System.out.println("time elapsed: " + (endTime-startTime));
			
			// method call resolution
			System.out.println("Resolving method calls(pass " + methodLoopCount + ")...");
//			startTime = System.currentTimeMillis();
			outerResolutionLoop:
			for (FullMethodID method : methods()) {
				int entryPoint = classMap.get(method.classID).getMethodEntry(method.methodID);
				Integer[] iterationOrder = cfg.getReverseNodeTraversalOrder(entryPoint);
//				System.out.println("Resolving method calls in " + method + "...");
				resolutionLoop:
				for (int node : iterationOrder) {
					CircusAction[] actions = runningChoices[node];
					CircusAction lastAction = actions[actions.length-1];
					if (lastAction instanceof CircusAction.HandleInvokeEPC) {
						// we are dealing with an unresolved method call
						// the second-to-last action should be a pc assumption
						int prePC = ((PCAssumption) actions[actions.length-2]).getPCValue();
						CPEntry.MethodRef methodRef = (MethodRef) classMap.get(method.classID).getCPEntry(
								((CircusAction.HandleInvokeEPC) lastAction).getConstantPoolIndex());
						if (lastAction instanceof CircusAction.HandleInvokevirtualEPC) {
							// resolve each target
								
							MethodID methodName = methodRef.getMethodID();
							ClassID className = methodRef.getClassID();
//							System.out.println("Resolving virtual method call: " + methodRef.getFullID());
							
							// build set of target classes
							HashSet<ClassID> targets = new HashSet<ClassID>();
							for (ClassID target : classMap.keySet()) {
//								System.out.println("target: " + target);
//								System.out.println("isSubclass: " + isSubclass(classMap, target, className));
//								System.out.println("instantiatedClasses.contains(target): " + instantiatedClasses.contains(target));
//								System.out.println("implemented interfaces: " + Arrays.toString(classMap.get(target).getInterfaceNames()));
//								System.out.println("");
//								if (className.equals(new ClassID("java.lang.Runnable"))){
//									try {
//										System.in.read();
//									} catch (IOException e) {
//										e.printStackTrace();
//									}
//								}
								if (isSubclass(classMap, target, className) && instantiatedClasses.contains(target)) {
									targets.add(target);
								}
							}
							
							String[] dispatchConditions = new String[targets.size()];
							CircusAction[][] dispatchActions = new CircusAction[targets.size()][];
							
							if (targets.isEmpty()) {
								// this is an erroneous method call
								// the method that contains it should be excluded from the set of methods
								excludedMethods.add(method);
								continue outerResolutionLoop;
							}
							
							int targetIndex = 0;
							for (ClassID target : targets) {
								// resolve normal methods by checking superclasses (special methods can't be virtual)
								ClassID resolvedClassName = target;
								boolean methodResolved = classMap.get(resolvedClassName).getMethods().contains(methodName);
								while (!methodResolved && classMap.get(resolvedClassName).hasSuper()) {
									ClassModel resolvedClass = classMap.get(resolvedClassName);
									if (resolvedClass.getMethods().contains(methodName)) {
										methodResolved = true;
									} else {
										resolvedClassName = resolvedClass.getSuperName();
									}
								}
								// one last check in case the method is in object
								ClassModel resolvedClass = classMap.get(resolvedClassName);
								if (resolvedClass.getMethods().contains(methodName)) {
									methodResolved = true;
								}
							
								if (methodResolved) {
									// successful normal method resolution
									if (!newModel.hasMethod(new FullMethodID(resolvedClassName,methodName))) {
										// can't be resolved now, continue outer loop
										// can't be resolved now, continue the outer loop
										continue resolutionLoop;
									}
									
									dispatchConditions[targetIndex] = "cid = " + target + "ID";
									
									FullMethodID resolvedActionName = new FullMethodID(resolvedClassName,methodName);
									if (classMap.get(resolvedClassName).isSynchronized(methodName)) {
										dispatchActions[targetIndex] = new CircusAction[4];
										dispatchActions[targetIndex][0] = 
												new CircusAction.ChannelComm("takeLock", new String[] {"(head~methodArgs)"}, new boolean[] {false}, 
												new CircusAction.ChannelComm("takeLockRet", null, null, new CircusAction.Skip()));
										dispatchActions[targetIndex][1] = new CircusAction.InterpreterNewStackFrame(resolvedClassName, methodName);
										dispatchActions[targetIndex][2] = new CircusAction.Poll();
										dispatchActions[targetIndex][3] = new CircusAction.MethodActionRef(resolvedActionName, null);
									} else {
										dispatchActions[targetIndex] = new CircusAction[3];
										dispatchActions[targetIndex][0] = new CircusAction.InterpreterNewStackFrame(resolvedClassName, methodName);
										dispatchActions[targetIndex][1] = new CircusAction.Poll();
										dispatchActions[targetIndex][2] = new CircusAction.MethodActionRef(resolvedActionName, null);
									}
//									dispatchActions[targetIndex] = new CircusAction[3];
//									dispatchActions[targetIndex][0] = new CircusAction.InterpreterNewStackFrame(resolvedClassName, methodName);
//									dispatchActions[targetIndex][1] = new CircusAction.Poll();
//									dispatchActions[targetIndex][2] = new CircusAction.MethodActionRef(resolvedActionName, null);
								} else {
									// can't be resolved now, continue the outer loop
									System.out.println("method resolution failure");
									continue resolutionLoop;
								}
								targetIndex++;
							}
							
							CircusAction[] argumentPoppingActions = new CircusAction[2];
							int numArgs = methodName.getNumArguments() + 1; // virtual method call so add one for this
							argumentPoppingActions[0] = new CircusAction.InterpreterStackFrameInvokeEPC(numArgs);
							argumentPoppingActions[1] = new CircusAction.ChannelComm("getClassIDOf", new String[] {"head~poppedArgs", "cid"}, new boolean[] {false, true}, 
									new CircusAction.Conditional(dispatchConditions, dispatchActions));
							
							CircusAction[] newActions = new CircusAction[2 + actions.length-2];
							for (int i = 0; i < actions.length-2; i++) {
								newActions[i] = actions[i];
							}
							newActions[actions.length-2] = new CircusAction.VarBlock("poppedArgs", argumentPoppingActions);
							newActions[actions.length-2+1] = new CircusAction.PCAssignment(prePC+1);
							runningChoices[node] = newActions;
							
						} else {
							// just one target to resolve
							MethodID methodName = methodRef.getMethodID();
							ClassID className = methodRef.getClassID();
							if (lastAction instanceof CircusAction.HandleInvokespecialEPC) {
								// if className is a superclass of current and not an init, 
								// it is the direct superclass
								if (isSubclass(classMap, className, method.classID)
										&& !className.equals(method.classID) && !methodName.isInit()) {
									className = classMap.get(method.classID).getSuperName();
								}
							}
							
//							System.out.println("Resolving nonvirtual method call: " + methodRef.getFullID());
							
							//check if we're doing a static invocation
							boolean isStatic = lastAction instanceof CircusAction.HandleInvokestaticEPC;
							//System.out.println("isStatic = " + isStatic);
							
							// resolve the method by first checking if it is a special method
							if (isSubclass(classMap, className, setPriorityCeilingClass)
									&& methodName.equals(setPriorityCeilingID)
									&& isStatic) {
								CircusAction[] newActions = new CircusAction[2 + actions.length-2];
								for (int i = 0; i < actions.length-2; i++) {
									newActions[i] = actions[i];
								}
								CircusAction[] argumentPoppingActions = new CircusAction[2];
								int numArgs = methodName.getNumArguments();
								argumentPoppingActions[0] = new CircusAction.InterpreterStackFrameInvokeEPC(numArgs);
								argumentPoppingActions[1] =
										new CircusAction.ChannelComm("setPriorityCeiling", new String[] {"methodArgs~1", "methodArgs~2"}, new boolean[] {false, false},
										new CircusAction.ChannelComm("setPriorityCeilingRet", null, null, new CircusAction.Skip()));
								newActions[actions.length-2] = new CircusAction.VarBlock("poppedArgs", argumentPoppingActions);
								newActions[actions.length-2+1] = new CircusAction.PCAssignment(prePC+1);
								runningChoices[node] = newActions;
							} else if (isSubclass(classMap, className, writeClass)
									&& methodName.equals(writeID)
									&& isStatic) {
								CircusAction[] newActions = new CircusAction[2 + actions.length-2];
								for (int i = 0; i < actions.length-2; i++) {
									newActions[i] = actions[i];
								}
								CircusAction[] argumentPoppingActions = new CircusAction[2];
								int numArgs = methodName.getNumArguments();
								argumentPoppingActions[0] = new CircusAction.InterpreterStackFrameInvokeEPC(numArgs);
								argumentPoppingActions[1] = new CircusAction.ChannelComm("output", new String[] {"methodArgs~1"}, new boolean[] {false}, new CircusAction.Skip());
								newActions[actions.length-2] = new CircusAction.VarBlock("poppedArgs", argumentPoppingActions);
								newActions[actions.length-2+1] = new CircusAction.PCAssignment(prePC+1);
								runningChoices[node] = newActions;
							} else if (isSubclass(classMap, className, readClass)
									&& methodName.equals(readID)
									&& isStatic) {
								CircusAction[] newActions = new CircusAction[2 + actions.length-2];
								for (int i = 0; i < actions.length-2; i++) {
									newActions[i] = actions[i];
								}
								CircusAction[] argumentPoppingActions = new CircusAction[2];
								int numArgs = methodName.getNumArguments();
								argumentPoppingActions[0] = new CircusAction.InterpreterStackFrameInvokeEPC(numArgs);
								argumentPoppingActions[1] = new CircusAction.ChannelComm("input", new String[] {"value"}, new boolean[] {true},
										new CircusAction.InterpreterPushEPC(null));
								newActions[actions.length-2] = new CircusAction.VarBlock("poppedArgs", argumentPoppingActions);
								newActions[actions.length-2+1] = new CircusAction.PCAssignment(prePC+1);
								runningChoices[node] = newActions;
							} else if (isSubclass(classMap, className, managedSchedulableClass)
									&& methodName.equals(registerID)
									&& isStatic) {
								CircusAction[] newActions = new CircusAction[2 + actions.length-2];
								for (int i = 0; i < actions.length-2; i++) {
									newActions[i] = actions[i];
								}
								CircusAction[] argumentPoppingActions = new CircusAction[2];
								int numArgs = methodName.getNumArguments();
								argumentPoppingActions[0] = new CircusAction.InterpreterStackFrameInvokeEPC(numArgs);
								argumentPoppingActions[1] = new CircusAction.ChannelComm("register", new String[] {"thread", "methodArgs~1"}, new boolean[] {false, false},
										new CircusAction.ChannelComm("registerRet", null, null, new CircusAction.Skip()));
								newActions[actions.length-2] = new CircusAction.VarBlock("poppedArgs", argumentPoppingActions);
								newActions[actions.length-2+1] = new CircusAction.PCAssignment(prePC+1);
								runningChoices[node] = newActions;
							} else if (isSubclass(classMap, className, managedMemoryClass)
									&& methodName.equals(enterPrivateMemoryHelperID)
									&& isStatic) {
								CircusAction[] newActions = new CircusAction[2 + actions.length-2];
								for (int i = 0; i < actions.length-2; i++) {
									newActions[i] = actions[i];
								}
								CircusAction[] argumentPoppingActions = new CircusAction[2];
								int numArgs = methodName.getNumArguments();
								argumentPoppingActions[0] = new CircusAction.InterpreterStackFrameInvokeEPC(numArgs);
								argumentPoppingActions[1] =
										new CircusAction.ChannelComm("enterPrivateMemory", new String[] {"thread", "methodArgs~1"}, new boolean[] {false, false},
										new CircusAction.ChannelComm("enterPrivateMemoryRet", null, null, new CircusAction.Skip()));
								newActions[actions.length-2] = new CircusAction.VarBlock("poppedArgs", argumentPoppingActions);
								newActions[actions.length-2+1] = new CircusAction.PCAssignment(prePC+1);
								runningChoices[node] = newActions;
							} else if (isSubclass(classMap, className, managedMemoryClass)
									&& methodName.equals(executeInAreaOfHelperID)
									&& isStatic) {
								CircusAction[] newActions = new CircusAction[2 + actions.length-2];
								for (int i = 0; i < actions.length-2; i++) {
									newActions[i] = actions[i];
								}
								CircusAction[] argumentPoppingActions = new CircusAction[2];
								int numArgs = methodName.getNumArguments();
								argumentPoppingActions[0] = new CircusAction.InterpreterStackFrameInvokeEPC(numArgs);
								argumentPoppingActions[1] =
										new CircusAction.ChannelComm("executeInAreaOf", new String[] {"thread", "methodArgs~1"}, new boolean[] {false, false},
										new CircusAction.ChannelComm("executeInAreaOfRet", null, null, new CircusAction.Skip()));
								newActions[actions.length-2] = new CircusAction.VarBlock("poppedArgs", argumentPoppingActions);
								newActions[actions.length-2+1] = new CircusAction.PCAssignment(prePC+1);
								runningChoices[node] = newActions;
							} else if (isSubclass(classMap, className, managedMemoryClass)
									&& methodName.equals(executeInOuterAreaHelperID)
									&& isStatic) {
								CircusAction[] newActions = new CircusAction[2 + actions.length-2];
								for (int i = 0; i < actions.length-2; i++) {
									newActions[i] = actions[i];
								}
								CircusAction[] argumentPoppingActions = new CircusAction[2];
								int numArgs = methodName.getNumArguments();
								argumentPoppingActions[0] = new CircusAction.InterpreterStackFrameInvokeEPC(numArgs);
								argumentPoppingActions[1] =
										new CircusAction.ChannelComm("executeInOuterArea", new String[] {"thread"}, new boolean[] {false},
										new CircusAction.ChannelComm("executeInOuterAreaRet", null, null, new CircusAction.Skip()));
								newActions[actions.length-2] = new CircusAction.VarBlock("poppedArgs", argumentPoppingActions);
								newActions[actions.length-2+1] = new CircusAction.PCAssignment(prePC+1);
								runningChoices[node] = newActions;
							} else if (isSubclass(classMap, className, managedMemoryClass)
									&& methodName.equals(exitMemoryID)
									&& isStatic) {
								CircusAction[] newActions = new CircusAction[2 + actions.length-2];
								for (int i = 0; i < actions.length-2; i++) {
									newActions[i] = actions[i];
								}
								CircusAction[] argumentPoppingActions = new CircusAction[2];
								int numArgs = methodName.getNumArguments();
								argumentPoppingActions[0] = new CircusAction.InterpreterStackFrameInvokeEPC(numArgs);
								argumentPoppingActions[1] =
										new CircusAction.ChannelComm("exitMemory", new String[] {"thread"}, new boolean[] {false},
										new CircusAction.ChannelComm("exitMemoryRet", null, null, new CircusAction.Skip()));
								newActions[actions.length-2] = new CircusAction.VarBlock("poppedArgs", argumentPoppingActions);
								newActions[actions.length-2+1] = new CircusAction.PCAssignment(prePC+1);
								runningChoices[node] = newActions;
							} else if (isSubclass(classMap, className, aperiodicEventHandlerClass)
									&& methodName.equals(initAPEHID)
									&& isStatic) {
								CircusAction[] newActions = new CircusAction[2 + actions.length-2];
								for (int i = 0; i < actions.length-2; i++) {
									newActions[i] = actions[i];
								}
								CircusAction[] argumentPoppingActions = new CircusAction[2];
								int numArgs = methodName.getNumArguments();
								argumentPoppingActions[0] = new CircusAction.InterpreterStackFrameInvokeEPC(numArgs);
								argumentPoppingActions[1] =
										new CircusAction.ChannelComm("initAPEH", new String[] {"thread", "seqTo5Tuple~methodArgs"}, new boolean[] {false, false},
										new CircusAction.ChannelComm("initAPEHRet", null, null, new CircusAction.Skip()));
								newActions[actions.length-2] = new CircusAction.VarBlock("poppedArgs", argumentPoppingActions);
								newActions[actions.length-2+1] = new CircusAction.PCAssignment(prePC+1);
								runningChoices[node] = newActions;
							} else if (isSubclass(classMap, className, periodicEventHandlerClass)
									&& methodName.equals(initPEHID)
									&& isStatic) {
								CircusAction[] newActions = new CircusAction[2 + actions.length-2];
								for (int i = 0; i < actions.length-2; i++) {
									newActions[i] = actions[i];
								}
								CircusAction[] argumentPoppingActions = new CircusAction[2];
								int numArgs = methodName.getNumArguments();
								argumentPoppingActions[0] = new CircusAction.InterpreterStackFrameInvokeEPC(numArgs);
								argumentPoppingActions[1] =
										new CircusAction.ChannelComm("initPEH", new String[] {"thread", "seqTo7Tuple~methodArgs"}, new boolean[] {false, false},
										new CircusAction.ChannelComm("initPEHRet", null, null, new CircusAction.Skip()));
								newActions[actions.length-2] = new CircusAction.VarBlock("poppedArgs", argumentPoppingActions);
								newActions[actions.length-2+1] = new CircusAction.PCAssignment(prePC+1);
								runningChoices[node] = newActions;
							} else if (isSubclass(classMap, className, oneShotEventHandlerClass)
									&& methodName.equals(initOSEHAbsID)
									&& isStatic) {
								CircusAction[] newActions = new CircusAction[2 + actions.length-2];
								for (int i = 0; i < actions.length-2; i++) {
									newActions[i] = actions[i];
								}
								CircusAction[] argumentPoppingActions = new CircusAction[2];
								int numArgs = methodName.getNumArguments();
								argumentPoppingActions[0] = new CircusAction.InterpreterStackFrameInvokeEPC(numArgs);
								argumentPoppingActions[1] =
										new CircusAction.ChannelComm("initOSEHAbs", new String[] {"thread", "seqTo6Tuple~methodArgs"}, new boolean[] {false, false},
										new CircusAction.ChannelComm("initOSEHAbsRet", null, null, new CircusAction.Skip()));
								newActions[actions.length-2] = new CircusAction.VarBlock("poppedArgs", argumentPoppingActions);
								newActions[actions.length-2+1] = new CircusAction.PCAssignment(prePC+1);
								runningChoices[node] = newActions;
							} else if (isSubclass(classMap, className, oneShotEventHandlerClass)
									&& methodName.equals(initOSEHRelID)
									&& isStatic) {
								CircusAction[] newActions = new CircusAction[2 + actions.length-2];
								for (int i = 0; i < actions.length-2; i++) {
									newActions[i] = actions[i];
								}
								CircusAction[] argumentPoppingActions = new CircusAction[2];
								int numArgs = methodName.getNumArguments();
								argumentPoppingActions[0] = new CircusAction.InterpreterStackFrameInvokeEPC(numArgs);
								argumentPoppingActions[1] =
										new CircusAction.ChannelComm("initOSEHRel", new String[] {"thread", "seqTo6Tuple~methodArgs"}, new boolean[] {false, false},
										new CircusAction.ChannelComm("initOSEHRelRet", null, null, new CircusAction.Skip()));
								newActions[actions.length-2] = new CircusAction.VarBlock("poppedArgs", argumentPoppingActions);
								newActions[actions.length-2+1] = new CircusAction.PCAssignment(prePC+1);
								runningChoices[node] = newActions;
							} else if (isSubclass(classMap, className, aperiodicEventHandlerClass)
									&& methodName.equals(releaseAperiodicID)
									&& isStatic) {
								CircusAction[] newActions = new CircusAction[2 + actions.length-2];
								for (int i = 0; i < actions.length-2; i++) {
									newActions[i] = actions[i];
								}
								CircusAction[] argumentPoppingActions = new CircusAction[2];
								int numArgs = methodName.getNumArguments();
								argumentPoppingActions[0] = new CircusAction.InterpreterStackFrameInvokeEPC(numArgs);
								argumentPoppingActions[1] =
										new CircusAction.ChannelComm("releaseAperiodic", new String[] {"thread", "methodArgs~1"}, new boolean[] {false, false},
										new CircusAction.ChannelComm("releaseAperiodicRet", null, null, new CircusAction.Skip()));
								newActions[actions.length-2] = new CircusAction.VarBlock("poppedArgs", argumentPoppingActions);
								newActions[actions.length-2+1] = new CircusAction.PCAssignment(prePC+1);
								runningChoices[node] = newActions;
							} else {
								// resolve normal methods by checking superclasses
								ClassID resolvedClassName = className;
//								System.out.println("To resolve: " + className);
								boolean methodResolved = classMap.get(resolvedClassName).getMethods().contains(methodName);
								while (!methodResolved && classMap.get(resolvedClassName).hasSuper()) {
//									System.out.println("Trying: " + resolvedClassName);
									ClassModel resolvedClass = classMap.get(resolvedClassName);
									if (resolvedClass.getMethods().contains(methodName)) {
										methodResolved = true;
									} else {
										resolvedClassName = resolvedClass.getSuperName();
									}
								}
								// one last check in case the method is in object
								ClassModel resolvedClass = classMap.get(resolvedClassName);
								if (resolvedClass.getMethods().contains(methodName)) {
									methodResolved = true;
								}
							
								if (methodResolved) {
									// successful normal method resolution
									FullMethodID resolvedActionName = new FullMethodID(resolvedClassName,methodName);
									if (!newModel.hasMethod(resolvedActionName)) {
										// can't be resolved now, continue the outer loop
//										System.out.println("Method hasn't been separated yet");
										continue resolutionLoop;
									}
									
									CircusAction[] newActions = new CircusAction[2 + actions.length-2];
									for (int i = 0; i < actions.length-2; i++) {
										newActions[i] = actions[i];
									}
									CircusAction[] argumentPoppingActions;
									int numArgs = methodName.getNumArguments();
									if (!isStatic) {
										numArgs++;
									}
									if (classMap.get(resolvedClassName).isSynchronized(methodName)) {
										argumentPoppingActions = new CircusAction[5];
										argumentPoppingActions[0] = new CircusAction.InterpreterStackFrameInvokeEPC(numArgs);
										argumentPoppingActions[1] = 
												new CircusAction.ChannelComm("takeLock", new String[] {"(head~methodArgs)"}, new boolean[] {false}, 
												new CircusAction.ChannelComm("takeLockRet", null, null, new CircusAction.Skip()));
										argumentPoppingActions[2] = new CircusAction.InterpreterNewStackFrame(resolvedClassName, methodName);
										argumentPoppingActions[3] = new CircusAction.Poll();
										argumentPoppingActions[4] = new CircusAction.MethodActionRef(resolvedActionName, null);
									} else {
										argumentPoppingActions = new CircusAction[4];
										argumentPoppingActions[0] = new CircusAction.InterpreterStackFrameInvokeEPC(numArgs);
										argumentPoppingActions[1] = new CircusAction.InterpreterNewStackFrame(resolvedClassName, methodName);
										argumentPoppingActions[2] = new CircusAction.Poll();
										argumentPoppingActions[3] = new CircusAction.MethodActionRef(resolvedActionName, null);
									}
									newActions[actions.length-2] = new CircusAction.VarBlock("poppedArgs", argumentPoppingActions);
									//newActions[actions.length-2+1] = new CircusAction.Poll();
									//newActions[actions.length-2+2] = new CircusAction.MethodActionRef(resolvedActionName, null);
									newActions[actions.length-2+1] = new CircusAction.PCAssignment(prePC+1);
									runningChoices[node] = newActions;
								} else {
									// can't be resolved now, continue the outer loop
//									System.out.println("Unresolved method:" + methodRef.getFullID());
//									try {
//										System.in.read();
//									} catch (IOException e) {
//										e.printStackTrace();
//									}
									continue resolutionLoop;
								}
									
							}
						}
						
						// update CFG in case it changed (should not affect iteration order)
						cfg = new ControlFlowGraph(runningChoices, classMap);
						
						// apply Rule [sequence-intro] if simple sequence
						if (cfg.hasSimpleSequence(node)) {
							actions = runningChoices[node];
							lastAction = actions[actions.length-1];
							if (lastAction instanceof CircusAction.PCAssignment) {
								int nextPC = ((CircusAction.PCAssignment) lastAction).getTarget();
								CircusAction[] nextActions = runningChoices[nextPC];
								Vector<CircusAction> newActions = new Vector<CircusAction>();
								newActions.addAll(Arrays.asList(actions));
								newActions.add(new CircusAction.Poll());
								newActions.addAll(Arrays.asList(nextActions));
								runningChoices[node] = newActions.toArray(new CircusAction[] {});
								cfg = new ControlFlowGraph(runningChoices, classMap);
							}
						}
					}
				}
			}
//			endTime = System.currentTimeMillis();
//			System.out.println("time elapsed: " + (endTime-startTime));
			
//			System.out.println("Unseparated methods after pass:");
//			for (FullMethodID method : methods()) {
//				if (!newModel.hasMethod(method)) {
//					System.out.println("\t" + method + ":");
//					//System.out.println("\t\t\\t1 " + CircusAction.actionsToString(runningChoices[classMap.get(method.classID).getMethodEntry(method.methodID)], 1));
//				}
//			}
		}
		
		// all methods separated, refine main actions can be done when epc model string is created
		
		// remove pc assignments from each method in EPC model addMethod() method
		
		return newModel;
	}
	
	private HashSet<FullMethodID> methods() {
		HashSet<FullMethodID> methodNames = new HashSet<FullMethodID>();
		for (ClassModel c : classes) {
			for (MethodID m : c.getMethods()) {
				methodNames.add(new FullMethodID(c.getClassName(), m));
			}
		}
		
		methodNames.removeAll(excludedMethods);
		
		return methodNames;
	}

	private boolean allMethodsSeparated(ThrCFModel newModel) {
		HashSet<FullMethodID> methodNames = methods();
		
		for (FullMethodID methodName : methodNames) {
			if (!newModel.hasMethod(methodName)) {
				return false;
			}
		}
		
		return true;
	}

	private ClassModel getClassFromPC(int pc) {
		for (ClassModel c : classes) {
			for (MethodID m : c.getMethods()) {
				if (pc >= c.getMethodEntry(m) && pc <= c.getMethodEnd(m)) {
					return c;
				}
			}
		}
		return null;
	}
	
	private MethodID getClassMethodIDFromPC(ClassModel c, int pc) {
		for (MethodID m : c.getMethods()) {
			if (pc >= c.getMethodEntry(m) && pc <= c.getMethodEnd(m)) {
				return m;
			}
		}
		return null;
	}
	
	private boolean isSubclass(HashMap<ClassID, ClassModel> classMap, ClassID childID, ClassID parentID) {
		if (childID.equals(parentID)) {
			return true;
		}
		if (classMap.containsKey(childID) && classMap.get(childID).hasSuper()) {
			if (isSubclass(classMap, classMap.get(childID).getSuperName(), parentID)) {
				return true;
			}
		}
		if (classMap.containsKey(childID)) {
			for (ClassID interfaceID : classMap.get(childID).getInterfaceNames()) {
				if (isSubclass(classMap, interfaceID, parentID)) {
					return true;
				}
			}
		}
		return false;
	}
	
	
	private boolean hasUnresolvedCalls(FullMethodID method, HashMap<ClassID, ClassModel> classMap, ControlFlowGraph cfg, CircusAction[][] runningChoices) {
			int entryPoint = classMap.get(method.classID).getMethodEntry(method.methodID);
			Integer[] iterationOrder = cfg.getReverseNodeTraversalOrder(entryPoint);
			for (int node : iterationOrder) {
				CircusAction[] actions = runningChoices[node];
				CircusAction lastAction = actions[actions.length-1];
				if (lastAction instanceof CircusAction.HandleInvokeEPC) {
					return true;
				}
			}
			return false;
		
	}
	
	public String generateHeader(String headerName) {
		StringBuilder builder = new StringBuilder();
		
		// require stdint.h
		builder.append("#include <stdlib.h>\n\n");
		builder.append("#include <stdint.h>\n\n");
		
		// output prototypes for services
		builder.append("void setPriorityCeiling(int32_t obj, int32_t ceiling);\n");
		builder.append("void output(int32_t value);\n");
		builder.append("int32_t input(void);\n");
		builder.append("void registerSchedulable(int32_t obj);\n");
		builder.append("void enterPrivateMemory(int32_t size);\n");
		builder.append("void executeInAreaOf(int32_t obj);\n");
		builder.append("void executeInOuterArea(void);\n");
		builder.append("void exitMemory(void);\n");
		builder.append("void initAPEH(int32_t apeh, int32_t priority, int32_t backingStoreSpace, int32_t allocAreaSpace, int32_t stackSize);\n");
		builder.append("void initPEH(int32_t peh, int32_t priority, int32_t start, int32_t period, int32_t backingStoreSpace, int32_t allocAreaSpace, int32_t stackSize);\n");
		builder.append("void initOSEHAbs(int32_t oseh, int32_t priority, int32_t startTime, int32_t backingStoreSpace, int32_t allocAreaSpace, int32_t stackSize);\n");
		builder.append("void initOSEHRel(int32_t oseh, int32_t priority, int32_t startTime, int32_t backingStoreSpace, int32_t allocAreaSpace, int32_t stackSize);\n");
		builder.append("void takeLock(int32_t obj);\n");
		builder.append("void releaseLock(int32_t obj);\n");
		builder.append("void releaseAperiodic(int32_t apeh);\n");
		builder.append("int32_t newObject(int32_t classID);\n");
		builder.append("\n");
		
		// output classID constants
		int classIDnum = 0;
		for (ClassID classID : classNameOrdering) {
			builder.append("const int32_t " + classID.toString().replace("\\_", "_") + "ID = " + classIDnum + ";\n");
			classIDnum++;
		}
		builder.append("\n");
		
		HashMap<ClassID,ClassModel> classMap = new HashMap<ClassID,ClassModel>();
		for (ClassModel classModel : classes) {
			classMap.put(classModel.getClassName(), classModel);
		}
		
		// output class structs
		for (ClassModel classInfo : classes) {
			builder.append("typedef struct " + classInfo.getClassName().toString().replace("\\_", "_") + " {\n");
			for (FieldID fieldID : collectFields(classInfo.getClassName(), classMap)) {
				if (fieldID.isLongOrDouble()) {
					builder.append("\tint32_t " + fieldID.toString().replace("\\_", "_") + "_msb;\n");
					builder.append("\tint32_t " + fieldID.toString().replace("\\_", "_") + "_lsb;\n");
				} else {
					builder.append("\tint32_t " + fieldID.toString().replace("\\_", "_") + ";\n");
				}
			}
			builder.append("} " + classInfo.getClassName().toString().replace("\\_", "_") + ";\n\n");
		}

		// output staticClassFields struct
		builder.append("typedef struct staticClassFieldsStruct {\n");
		for (ClassModel classInfo : classes) {
			for (FieldID fieldID : classInfo.getStaticFields()) {
				builder.append("\tint32_t " + classInfo.getClassName().toString().replace("\\_", "_") + "_" + fieldID.toString().replace("\\_", "_") + ";\n");
			}
		}
		builder.append("} staticClassFieldsStruct;\n\n");
		
		return builder.toString();
	}
	
	private Vector<FieldID> collectFields(ClassID classID, HashMap<ClassID,ClassModel> classMap) {
		Vector<FieldID> fields = new Vector<FieldID>();
		
		// get class info
		ClassModel classInfo = classMap.get(classID);
		
		// check if there is a superclass
		if (classInfo.hasSuper()) {
			// add superclass fields first
			fields.addAll(collectFields(classInfo.getSuperName(), classMap));
		} else {
			// add a classID field
			fields.add(new FieldID("classID", ""));
		}
		
		// add the class' fields
		fields.addAll(classInfo.getFields());
		
		return fields;
	}
//	private CircusAction getLastAction(CircusAction[] actions) {
//		CircusAction lastAction = actions[actions.length-1];
//		while (lastAction instanceof CircusAction.VarBlock || lastAction instanceof CircusAction.ChannelComm) {
//			if (lastAction instanceof CircusAction.VarBlock) {
//				actions = ((CircusAction.VarBlock) lastAction).getActions();
//				lastAction = actions[actions.length-1];
//			} else if (lastAction instanceof CircusAction.ChannelComm) {
//				lastAction = ((CircusAction.ChannelComm) lastAction).getAction();
//			}
//		}
//		return lastAction;
//	}
}
