package main;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Vector;

public class ControlFlowGraph {
	private HashMap<Integer, HashSet<Integer>> nodeSuccessors = new HashMap<Integer, HashSet<Integer>>();
	private HashMap<Integer, HashSet<Integer>> nodePredecessors = new HashMap<Integer, HashSet<Integer>>();
	private HashMap<Integer, HashSet<Integer>> nodeLooplessSuccessors = new HashMap<Integer, HashSet<Integer>>();
	private HashMap<Integer, HashSet<Integer>> nodeLooplessPredecessors = new HashMap<Integer, HashSet<Integer>>();
	private HashSet<Integer> nodes = new HashSet<Integer>();
	
	//private static int callCount = 0;
	//private static long callTime = 0;
	
	public ControlFlowGraph(CircusAction[][] nodeActions, HashMap<ClassID, ClassModel> classMap) {
		//callCount++;
		//System.out.println("Start CFG construction (call " + callCount + ")");
		//final long startTime = System.currentTimeMillis();
		// start constructing control flow graph at each method entry point
		for (ClassModel classModel : classMap.values()) {
			for (MethodID methodID : classModel.getMethods()) {
				// renew visited nodes for each method
				HashSet<Integer> visitedNodes = new HashSet<Integer>();
				int methodEntry = classModel.getMethodEntry(methodID);
				// traverse methods to build CFG
				buildCFG(nodeActions, methodEntry, visitedNodes);
			}
		}
		// make sure everything is non-null
		for (int node : nodes) {
			if (!nodeSuccessors.containsKey(node)) {
				nodeSuccessors.put(node, new HashSet<Integer>());
			}
			if (!nodePredecessors.containsKey(node)) {
				nodePredecessors.put(node, new HashSet<Integer>());
			}
			if (!nodeLooplessSuccessors.containsKey(node)) {
				nodeLooplessSuccessors.put(node, new HashSet<Integer>());
			}
			if (!nodeLooplessPredecessors.containsKey(node)) {
				nodeLooplessPredecessors.put(node, new HashSet<Integer>());
			}
		}
		//final long endTime = System.currentTimeMillis();
		//System.out.println("End CFG construction, time elapsed: " + (endTime-startTime) + " ms");
		//callTime += (endTime-startTime);
		//System.out.println("Cumulative time elapsed: " + callTime + " ms");
	}
	
	private void buildCFG(CircusAction[][] nodeActions, int currentNode, HashSet<Integer> visitedNodes) {
		visitedNodes.add(currentNode);
		nodes.add(currentNode);
		
		CircusAction[] actions = nodeActions[currentNode];
		CircusAction lastActionAtCurrentNode = actions[actions.length-1];
		while (lastActionAtCurrentNode instanceof CircusAction.VarBlock || lastActionAtCurrentNode instanceof CircusAction.ChannelComm) {
			if (lastActionAtCurrentNode instanceof CircusAction.VarBlock) {
				actions = ((CircusAction.VarBlock) lastActionAtCurrentNode).getActions();
				lastActionAtCurrentNode = actions[actions.length-1];
			} else if (lastActionAtCurrentNode instanceof CircusAction.ChannelComm) {
				lastActionAtCurrentNode = ((CircusAction.ChannelComm) lastActionAtCurrentNode).getAction();
			}
		}
		
		if (lastActionAtCurrentNode instanceof CircusAction.PCAssignment) {
			int nextNode = ((CircusAction.PCAssignment) lastActionAtCurrentNode).getTarget();
			addCFGEdge(currentNode, nextNode, visitedNodes);
			if (!visitedNodes.contains(nextNode)) {
				buildCFG(nodeActions, nextNode, visitedNodes);
			}
		} else if (lastActionAtCurrentNode instanceof CircusAction.ConditionalPCAssignment) {
			int nextNode1 = ((CircusAction.ConditionalPCAssignment) lastActionAtCurrentNode).getTrueTarget();
			int nextNode2 = ((CircusAction.ConditionalPCAssignment) lastActionAtCurrentNode).getFalseTarget();
			addCFGEdge(currentNode, nextNode1, visitedNodes);
			addCFGEdge(currentNode, nextNode2, visitedNodes);
			@SuppressWarnings("unchecked")
			HashSet<Integer> visitedNodesFork = (HashSet<Integer>) visitedNodes.clone();
			if (!visitedNodes.contains(nextNode1)) {
				buildCFG(nodeActions, nextNode1, visitedNodes);
			}
			if (!visitedNodesFork.contains(nextNode2)) {
				buildCFG(nodeActions, nextNode2, visitedNodesFork);
			}
		} else if (lastActionAtCurrentNode instanceof CircusAction.HandleInvokeEPC) {
			int nextNode = ((CircusAction.PCAssumption) actions[actions.length-2]).getPCValue() + 1;
			addCFGEdge(currentNode, nextNode, visitedNodes);
			if (!visitedNodes.contains(nextNode)) {
				buildCFG(nodeActions, nextNode, visitedNodes);
			}
		}
		
		// anything else should be the end of the control flow graph, so we can return
		
	}
	
	private void addCFGEdge(int currentNode, int nextNode, HashSet<Integer> visitedNodes) {
		if (nodeSuccessors.containsKey(currentNode)) {
			nodeSuccessors.get(currentNode).add(nextNode);
		} else {
			HashSet<Integer> newSet = new HashSet<Integer>();
			newSet.add(nextNode);
			nodeSuccessors.put(currentNode, newSet);
		}
		if (nodePredecessors.containsKey(nextNode)) {
			nodePredecessors.get(nextNode).add(currentNode);
		} else {
			HashSet<Integer> newSet = new HashSet<Integer>();
			newSet.add(currentNode);
			nodePredecessors.put(nextNode, newSet);
		}
		
		if (!visitedNodes.contains(nextNode)) {
			if (nodeLooplessSuccessors.containsKey(currentNode)) {
				nodeLooplessSuccessors.get(currentNode).add(nextNode);
			} else {
				HashSet<Integer> newSet = new HashSet<Integer>();
				newSet.add(nextNode);
				nodeLooplessSuccessors.put(currentNode, newSet);
			}
			if (nodeLooplessPredecessors.containsKey(nextNode)) {
				nodeLooplessPredecessors.get(nextNode).add(currentNode);
			} else {
				HashSet<Integer> newSet = new HashSet<Integer>();
				newSet.add(currentNode);
				nodeLooplessPredecessors.put(nextNode, newSet);
			}
		}
	}
	
	public Integer[] getReverseNodeTraversalOrder(int entryPoint) {
		//System.out.println("Computing reverse node traversal order");
		Vector<Integer> traversalOrder = new Vector<Integer>();
		
		//System.out.println("Start node: " + entryPoint);
		//System.out.println("Start node successors: " + nodeLooplessSuccessors.get(entryPoint));
		
		HashSet<Integer> endNodes = new HashSet<Integer>();
		buildEndNodeSet(endNodes, entryPoint);
		for (int node : endNodes) {
			traversalOrder.add(node);
		}
		//System.out.println("End nodes: " + endNodes);
		
		HashSet<Integer> remainingMethodNodes = new HashSet<Integer>();
		buildSubgraphNodeSet(remainingMethodNodes, entryPoint);
		remainingMethodNodes.removeAll(endNodes);
		//System.out.println("Nodes Remaining: " + remainingMethodNodes);
		
		while (!remainingMethodNodes.isEmpty()) {
			HashSet<Integer> nodesToBeAdded = new HashSet<Integer>();
			for (int node : remainingMethodNodes) {
				if (traversalOrder.containsAll(nodeLooplessSuccessors.get(node))) {
					nodesToBeAdded.add(node);
				}
			}
			
			for (int node : nodesToBeAdded) {
					traversalOrder.add(node);
			}
			remainingMethodNodes.removeAll(nodesToBeAdded);
			//System.out.println("Nodes Added: " + nodesToBeAdded);
			//System.out.println("Nodes Remaining: " + remainingMethodNodes);
		}
		
		return traversalOrder.toArray(new Integer[] {});
	}
	
	private void buildEndNodeSet(HashSet<Integer> endNodes, int currentNode) {
		if (nodeLooplessSuccessors.containsKey(currentNode) && !nodeLooplessSuccessors.get(currentNode).isEmpty()) {
			//System.out.println("Current node: " + currentNode + ", Successors: " + nodeLooplessSuccessors.get(currentNode));
			for (int nextNode : nodeLooplessSuccessors.get(currentNode)) {
				buildEndNodeSet(endNodes, nextNode);
				
			}
		} else {
			endNodes.add(currentNode);
			//System.out.println("Current node: " + currentNode + ", added to end nodes");
		}
	}
	
	private void buildSubgraphNodeSet(HashSet<Integer> subgraphNodes, int currentNode) {
		subgraphNodes.add(currentNode);
		if (nodeLooplessSuccessors.containsKey(currentNode) && !nodeLooplessSuccessors.get(currentNode).isEmpty()) {
			for (int nextNode : nodeLooplessSuccessors.get(currentNode)) {
				buildSubgraphNodeSet(subgraphNodes, nextNode);
			}
		}
	}
	
	public boolean hasSimpleSequence(int node) {
		if (nodeSuccessors.containsKey(node)) {
			if (nodeSuccessors.get(node).size() == 1) {
				int successor = (Integer) nodeSuccessors.get(node).toArray()[0];
				if (nodePredecessors.containsKey(successor)) {
					if (nodePredecessors.get(successor).size() == 1) {
						return true;
					}
				}
			}
		}
		return false;
	}
	
	public boolean hasSimpleConditional(int node) {
		if (nodeSuccessors.containsKey(node)) {
			for (int target : nodeSuccessors.get(node)) {
				if (nodeSuccessors.containsKey(target)) {
					if (nodeSuccessors.get(target).size() != 0) {
						return false;
					}
				}
			}
			return true;
		}
		return false;
	}
	
	public boolean isComplete(int node) {
		if (nodeSuccessors.containsKey(node)) {
			if (nodeSuccessors.get(node).size() == 0) {
				return true;
			}
		}
		return false;
	}
}
