package main;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map.Entry;
import java.util.Set;
import java.util.TreeMap;
import java.util.Vector;

public class ClassModel {
	
	private ClassID className;
	private int thisIndex, superIndex;
	private int interfaceIndices[];
	private TreeMap<Integer,CPEntry> constantPool = new TreeMap<Integer,CPEntry>();
	private HashMap<MethodID,Integer> methodEntry = new HashMap<MethodID,Integer>();
	private HashMap<MethodID,Integer> methodEnd = new HashMap<MethodID,Integer>();
	private HashMap<MethodID,Integer> methodLocals = new HashMap<MethodID,Integer>();
	private HashMap<MethodID,Integer> methodStackSize = new HashMap<MethodID,Integer>();
	private Vector<MethodID> staticMethods = new Vector<MethodID>();
	private Vector<MethodID> synchronizedMethods = new Vector<MethodID>();
	private Vector<FieldID> fields = new Vector<FieldID>();
	private Vector<FieldID> staticFields = new Vector<FieldID>();
	private boolean isAbstract;
	
	public ClassModel(ClassID className, int thisIndex, int superIndex, int interfaceIndices[], boolean isAbstract){
		this.className = className;
		this.thisIndex = thisIndex;
		this.superIndex = superIndex;
		this.interfaceIndices = interfaceIndices;
		this.isAbstract = isAbstract;
	}
	
	public ClassID getClassName() {
		return className;
	}
	
	public ClassID getSuperName() {
		return ((CPEntry.ClassRef) this.constantPool.get(this.superIndex)).getClassID();
	}
	
	public ClassID[] getInterfaceNames() {
		ClassID[] interfaces = new ClassID[interfaceIndices.length];
		for (int i = 0; i < interfaces.length; i++) {
			interfaces[i] = ((CPEntry.ClassRef) this.constantPool.get(interfaceIndices[i])).getClassID();
		}
		return interfaces;
	}
	
	public Vector<FieldID> getFields() {
		return fields;
	}
	
	public void addCPEntry(int index, CPEntry entry){
		this.constantPool.put(index, entry);
	}
	
	public void addMethod(MethodID methodID, int entry, int end, int locals, int stackSize,
			boolean isStatic, boolean isSynchronized) {
		this.methodEntry.put(methodID, entry);
		this.methodEnd.put(methodID, end);
		this.methodLocals.put(methodID, locals);
		this.methodStackSize.put(methodID, stackSize);
		if (isStatic) {
			staticMethods.add(methodID);
		}
		if (isSynchronized) {
			synchronizedMethods.add(methodID);
		}
	}
	
	public void addField(FieldID field) {
		this.fields.add(field);
	}
	
	public void addStaticField(FieldID field) {
		this.staticFields.add(field);
	}
	
	public CPEntry getCPEntry(int cpIndex) {
		return constantPool.get(cpIndex);
	}
	
	public Set<MethodID> getMethods() {
		return methodEntry.keySet();
	}
	
	public int getMethodEntry(MethodID methodID) {
		return methodEntry.get(methodID);
	}
	
	public int getMethodEnd(MethodID methodID) {
		return methodEnd.get(methodID);
	}
	
	public int getMethodLocals(MethodID methodID) {
		return methodLocals.get(methodID);
	}
	
	public int getMethodStackSize(MethodID methodID) {
		return methodStackSize.get(methodID);
	}

	public String toModelString() {
		StringBuilder buffer = new StringBuilder(1024);
		
		buffer.append("\\begin{axdef}\n");
		buffer.append("\t" + className + " : Class\n");
		buffer.append("\\where\n");
		buffer.append("\t" + className + " = \\lblot \\\\\n");
		buffer.append("\t\\t1 constantPool == \\{ \\\\\n");
		Iterator<Entry<Integer, CPEntry>> cpIterator = constantPool.entrySet().iterator();
		while (cpIterator.hasNext()) {
			Entry<Integer, CPEntry> entry = cpIterator.next();
			buffer.append("\t\t\\t2 " + entry.getKey() + " \\mapsto " + entry.getValue().toModelString());
			if (cpIterator.hasNext()) {
				buffer.append(", \\\\\n");
			} else {
				buffer.append(" \\\\\n");
			}
		}
		buffer.append("\t\\t1 \\}, \\\\\n");
		buffer.append("\t\\t1 this == " + thisIndex + ", \\\\\n");
		if (superIndex == 0) {
			buffer.append("\t\\t1 super == nullCPIndex, \\\\\n");
		} else {
			buffer.append("\t\\t1 super == " + superIndex + ", \\\\\n");
		}
		buffer.append("\t\\t1 interfaces == \\{");
		for(int i = 0; i < interfaceIndices.length; i++){
			buffer.append(interfaceIndices[i]);
			if (i < interfaceIndices.length-1) {
				buffer.append(", ");
			}
		}
		buffer.append("\\}, \\\\\n");
		buffer.append("\t\\t1 methodEntry == \\{ \\\\\n");
		Iterator<Entry<MethodID,Integer>> methodIterator = methodEntry.entrySet().iterator();
		while (methodIterator.hasNext()) {
			Entry<MethodID,Integer> entry = methodIterator.next();
			buffer.append("\t\t\\t2 " + entry.getKey() + " \\mapsto " + entry.getValue());
			if (methodIterator.hasNext()) {
				buffer.append(", \\\\\n");
			} else {
				buffer.append(" \\\\\n");
			}
		}
		buffer.append("\t\\t1 \\}, \\\\\n");
		
		buffer.append("\t\\t1 methodEnd == \\{ \\\\\n");
		methodIterator = methodEnd.entrySet().iterator();
		while (methodIterator.hasNext()) {
			Entry<MethodID,Integer> entry = methodIterator.next();
			buffer.append("\t\t\\t2 " + entry.getKey() + " \\mapsto " + entry.getValue());
			if (methodIterator.hasNext()) {
				buffer.append(", \\\\\n");
			} else {
				buffer.append(" \\\\\n");
			}
		}
		buffer.append("\t\\t1 \\}, \\\\\n");
		
		buffer.append("\t\\t1 methodLocals == \\{ \\\\\n");
		methodIterator = methodLocals.entrySet().iterator();
		while (methodIterator.hasNext()) {
			Entry<MethodID,Integer> entry = methodIterator.next();
			buffer.append("\t\t\\t2 " + entry.getKey() + " \\mapsto " + entry.getValue());
			if (methodIterator.hasNext()) {
				buffer.append(", \\\\\n");
			} else {
				buffer.append(" \\\\\n");
			}
		}
		buffer.append("\t\\t1 \\}, \\\\\n");
		
		buffer.append("\t\\t1 methodStackSize == \\{ \\\\\n");
		methodIterator = methodStackSize.entrySet().iterator();
		while (methodIterator.hasNext()) {
			Entry<MethodID,Integer> entry = methodIterator.next();
			buffer.append("\t\t\\t2 " + entry.getKey() + " \\mapsto " + entry.getValue());
			if (methodIterator.hasNext()) {
				buffer.append(", \\\\\n");
			} else {
				buffer.append(" \\\\\n");
			}
		}
		buffer.append("\t\\t1 \\}, \\\\\n");
		
		buffer.append("\t\\t1 fields == \\{ \\\\\n");
		Iterator<FieldID> fieldIterator = fields.iterator();
		while (fieldIterator.hasNext()) {
			FieldID field = fieldIterator.next();
			buffer.append("\t\t\\t2 " + field);
			if (fieldIterator.hasNext()) {
				buffer.append(", \\\\\n");
			} else {
				buffer.append(" \\\\\n");
			}
		}
		buffer.append("\t\\t1 \\}, \\\\\n");
		
		buffer.append("\t\\t1 staticFields == \\{ \\\\\n");
		fieldIterator = staticFields.iterator();
		while (fieldIterator.hasNext()) {
			FieldID field = fieldIterator.next();
			buffer.append("\t\t\\t2 " + field);
			if (fieldIterator.hasNext()) {
				buffer.append(", \\\\\n");
			} else {
				buffer.append(" \\\\\n");
			}
		}
		buffer.append("\t\\t1 \\} \\\\\n");
		buffer.append("\t\\rblot\n");
		buffer.append("\\end{axdef}\n");
		
		return buffer.toString();
	}

	public boolean hasSuper() {
		return this.superIndex != 0;
	}
	
	public boolean isStatic(MethodID methodID) {
		return staticMethods.contains(methodID);
	}
	
	public boolean isSynchronized(MethodID methodID) {
		return synchronizedMethods.contains(methodID);
	}

	public boolean isAbstract() {
		return this.isAbstract;
	}

	public Vector<FieldID> getStaticFields() {
		return staticFields;
	}
 
}
