package main;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.util.HashMap;
import java.util.Vector;

import org.apache.commons.bcel6.classfile.ClassParser;
import org.apache.commons.bcel6.classfile.Constant;
import org.apache.commons.bcel6.classfile.ConstantClass;
import org.apache.commons.bcel6.classfile.ConstantDouble;
import org.apache.commons.bcel6.classfile.ConstantFieldref;
import org.apache.commons.bcel6.classfile.ConstantInterfaceMethodref;
import org.apache.commons.bcel6.classfile.ConstantInteger;
import org.apache.commons.bcel6.classfile.ConstantLong;
import org.apache.commons.bcel6.classfile.ConstantMethodref;
import org.apache.commons.bcel6.classfile.ConstantNameAndType;
import org.apache.commons.bcel6.classfile.ConstantPool;
import org.apache.commons.bcel6.classfile.Field;
import org.apache.commons.bcel6.classfile.JavaClass;
import org.apache.commons.bcel6.classfile.Method;

import main.CPEntry.ClassRef;
import main.CPEntry.FieldRef;
import main.CPEntry.MethodRef;

import org.apache.commons.bcel6.Const;
import org.apache.commons.bcel6.classfile.ClassFormatException;

public class ClassFileModelConverter {

	public static void main(String[] args) throws ClassFormatException, IOException {
		
		Vector<String> inputFiles = new Vector<String>();
		String outputFile = null;
		boolean doEPC = false;
		boolean doEFS = false;
		boolean doC = false;
		boolean includeLibrary = false;
		boolean addPreamble = false;
		boolean expandIINC = false;
		for (int i = 0; i < args.length; i++) {
			if (args[i].equals("-o")) {
				i++;
				if (!(i < args.length)) {
					throw new IllegalArgumentException("Expected output file name");
				}
				if (outputFile != null) {
					throw new IllegalArgumentException("Need one output file, two specified: " + outputFile + " and " + args[i]);
				}
				outputFile = args[i];
			} else if (args[i].equals("--EPC")) {
				doEPC = true;
				includeLibrary = true;
			} else if (args[i].equals("--EFS")) {
				doEFS = true;
				doEPC = true;
				includeLibrary = true;
			} else if (args[i].equals("-C")) {
				doC = true;
				doEFS = true;
				doEPC = true;
				includeLibrary = true;
			} else if (args[i].equals("-l")) {
				includeLibrary = true;
			} else if (args[i].equals("--expand-iinc")) {
				expandIINC = true;
			} else if (args[i].equals("-p")) {
				addPreamble = true;
			} else {
				// we assume this is an input file
				inputFiles.add(args[i]);
			}
		}
		
		if (inputFiles.isEmpty()) {
			// take every class file in the current directory by default
			for (File file : new File(".").listFiles()) {
				String extension = "";

				int i = file.toString().lastIndexOf('.');
				if (i > 0) {
				    extension = file.toString().substring(i+1);
				}
				
				if (extension.equals("class")) {
					inputFiles.add(file.getCanonicalPath());
				}
				
			}
		}
		
		if (outputFile == null) {
			throw new IllegalArgumentException("Need an output file to be specified");
		}
		
		if (inputFiles.isEmpty()) {
			throw new IllegalArgumentException("Need at least one input file");
		}
		
		if (includeLibrary) {
			String librarypath = System.getenv("SCJPATH");
			inputFiles.add(librarypath + "javax/safetycritical/AperiodicEventHandler.class");
			inputFiles.add(librarypath + "javax/safetycritical/ManagedEventHandler.class");
			inputFiles.add(librarypath + "javax/safetycritical/ManagedMemory.class");
			inputFiles.add(librarypath + "javax/safetycritical/ManagedSchedulable.class");
			inputFiles.add(librarypath + "javax/safetycritical/Mission.class");
			inputFiles.add(librarypath + "javax/safetycritical/MissionSequencer.class");
			inputFiles.add(librarypath + "javax/safetycritical/OneShotEventHandler.class");
			inputFiles.add(librarypath + "javax/safetycritical/PeriodicEventHandler.class");
			inputFiles.add(librarypath + "javax/safetycritical/PriorityScheduler.class");
			inputFiles.add(librarypath + "javax/safetycritical/Safelet.class");
			inputFiles.add(librarypath + "javax/safetycritical/Services.class");
			inputFiles.add(librarypath + "javax/safetycritical/io/ConsoleConnection.class");
			inputFiles.add(librarypath + "javax/safetycritical/io/ConsoleInput.class");
			inputFiles.add(librarypath + "javax/safetycritical/io/ConsoleOutput.class");
			inputFiles.add(librarypath + "javax/realtime/AbsoluteTime.class");
			inputFiles.add(librarypath + "javax/realtime/AperiodicParameters.class");
			inputFiles.add(librarypath + "javax/realtime/AsyncBaseEventHandler.class");
			inputFiles.add(librarypath + "javax/realtime/AsyncEventHandler.class");
			inputFiles.add(librarypath + "javax/realtime/BoundAsyncBaseEventHandler.class");
			inputFiles.add(librarypath + "javax/realtime/BoundAsyncEventHandler.class");
			inputFiles.add(librarypath + "javax/realtime/BoundRealtimeExecutor.class");
			inputFiles.add(librarypath + "javax/realtime/BoundSchedulable.class");
			inputFiles.add(librarypath + "javax/realtime/Chronograph.class");
			inputFiles.add(librarypath + "javax/realtime/ConfigurationParameters.class");
			inputFiles.add(librarypath + "javax/realtime/HighResolutionTime.class");
			inputFiles.add(librarypath + "javax/realtime/MemoryArea.class");
			inputFiles.add(librarypath + "javax/realtime/MemoryParameters.class");
			inputFiles.add(librarypath + "javax/realtime/PeriodicParameters.class");
			inputFiles.add(librarypath + "javax/realtime/PriorityParameters.class");
			inputFiles.add(librarypath + "javax/realtime/PriorityScheduler.class");
			inputFiles.add(librarypath + "javax/realtime/RelativeTime.class");
			inputFiles.add(librarypath + "javax/realtime/Releasable.class");
			inputFiles.add(librarypath + "javax/realtime/ReleaseParameters.class");
			inputFiles.add(librarypath + "javax/realtime/Schedulable.class");
			inputFiles.add(librarypath + "javax/realtime/SchedulingParameters.class");
			inputFiles.add(librarypath + "javax/realtime/Scheduler.class");
			inputFiles.add(librarypath + "javax/realtime/Timable.class");
			inputFiles.add(librarypath + "javax/realtime/memory/ScopedMemory.class");
			inputFiles.add(librarypath + "javax/realtime/memory/ScopeParameters.class");
			inputFiles.add(librarypath + "javax/realtime/memory/StackedMemory.class");
			inputFiles.add(librarypath + "javax/microedition/io/Connection.class");
			inputFiles.add(librarypath + "javax/microedition/io/InputConnection.class");
			inputFiles.add(librarypath + "javax/microedition/io/OutputConnection.class");
			inputFiles.add(librarypath + "javax/microedition/io/StreamConnection.class");
			inputFiles.add(librarypath + "java/lang/Array.class");
			inputFiles.add(librarypath + "java/lang/Array1.class");
			inputFiles.add(librarypath + "java/lang/Array2.class");
			inputFiles.add(librarypath + "java/lang/Array3.class");
			inputFiles.add(librarypath + "java/lang/Array4.class");
			inputFiles.add(librarypath + "java/lang/Array5.class");
			inputFiles.add(librarypath + "java/lang/BooleanArray.class");
			inputFiles.add(librarypath + "java/lang/BooleanArray1.class");
			inputFiles.add(librarypath + "java/lang/BooleanArray2.class");
			inputFiles.add(librarypath + "java/lang/BooleanArray3.class");
			inputFiles.add(librarypath + "java/lang/BooleanArray4.class");
			inputFiles.add(librarypath + "java/lang/BooleanArray5.class");
			inputFiles.add(librarypath + "java/lang/Cloneable.class");
			inputFiles.add(librarypath + "java/lang/Comparable.class");
			inputFiles.add(librarypath + "java/lang/IntArray.class");
			inputFiles.add(librarypath + "java/lang/IntArray1.class");
			inputFiles.add(librarypath + "java/lang/LongArray.class");
			inputFiles.add(librarypath + "java/lang/LongArray1.class");
			inputFiles.add(librarypath + "java/lang/Object.class");
			inputFiles.add(librarypath + "java/lang/String.class");
			inputFiles.add(librarypath + "java/io/Closeable.class");
			inputFiles.add(librarypath + "java/io/DataInputStream.class");
			inputFiles.add(librarypath + "java/io/DataOutputStream.class");
			inputFiles.add(librarypath + "java/io/Flushable.class");
			inputFiles.add(librarypath + "java/io/InputStream.class");
			inputFiles.add(librarypath + "java/io/OutputStream.class");
			inputFiles.add(librarypath + "devices/Console.class");
			inputFiles.add(librarypath + "javax/scj/util/Const.class");
		}
		
		Model model = new Model();
		
		for (String inputFile : inputFiles) {
			ClassParser parser = new ClassParser(inputFile);
			JavaClass classFile = parser.parse();
			
			ClassID classID = new ClassID(classFile.getClassName());
			model.addClassName(classID);
			ClassModel classModel = new ClassModel(
				classID,
				classFile.getClassNameIndex(), 
				classFile.getSuperclassNameIndex(), 
				classFile.getInterfaceIndices(),
				classFile.isAbstract());
			
			ConstantPool constantPoolObject = classFile.getConstantPool();
			Constant[] constantPool = constantPoolObject.getConstantPool();
			for (int i = 1; i < constantPool.length; i++) {
				if (constantPool[i] instanceof ConstantClass) {
					ClassID className = new ClassID(((ConstantClass) constantPool[i]).getBytes(constantPoolObject));
					model.addClassName(className);
					classModel.addCPEntry(i, new ClassRef(className));
					
				} else if (constantPool[i] instanceof ConstantFieldref) {
					int classIndex = ((ConstantFieldref) constantPool[i]).getClassIndex();
					int nameAndTypeIndex = ((ConstantFieldref) constantPool[i]).getNameAndTypeIndex();
					ClassID className = new ClassID(((ConstantClass) constantPool[classIndex]).getBytes(constantPoolObject));
					String fieldName = ((ConstantNameAndType) constantPool[nameAndTypeIndex]).getName(constantPoolObject);
					String signature = ((ConstantNameAndType) constantPool[nameAndTypeIndex]).getSignature(constantPoolObject);
					FieldID fieldID = new FieldID(fieldName, signature);
					model.addClassName(className);
					model.addFieldName(fieldID);
					classModel.addCPEntry(i, new FieldRef(className, fieldID));
					
				} else if (constantPool[i] instanceof ConstantInterfaceMethodref) {
					int classIndex = ((ConstantInterfaceMethodref) constantPool[i]).getClassIndex();
					int nameAndTypeIndex = ((ConstantInterfaceMethodref) constantPool[i]).getNameAndTypeIndex();
					ClassID className = new ClassID(((ConstantClass) constantPool[classIndex]).getBytes(constantPoolObject));
					String methodName = ((ConstantNameAndType) constantPool[nameAndTypeIndex]).getName(constantPoolObject);
					String signature = ((ConstantNameAndType) constantPool[nameAndTypeIndex]).getSignature(constantPoolObject);
					MethodID methodID = new MethodID(methodName, signature);
					model.addClassName(className);
					model.addMethodName(methodID);
					classModel.addCPEntry(i, new MethodRef(className, methodID));
					
				} else if (constantPool[i] instanceof ConstantMethodref) {
					int classIndex = ((ConstantMethodref) constantPool[i]).getClassIndex();
					int nameAndTypeIndex = ((ConstantMethodref) constantPool[i]).getNameAndTypeIndex();
					ClassID className = new ClassID(((ConstantClass) constantPool[classIndex]).getBytes(constantPoolObject));
					String methodName = ((ConstantNameAndType) constantPool[nameAndTypeIndex]).getName(constantPoolObject);
					String signature = ((ConstantNameAndType) constantPool[nameAndTypeIndex]).getSignature(constantPoolObject);
					MethodID methodID = new MethodID(methodName, signature);
					model.addClassName(className);
					model.addMethodName(methodID);
					classModel.addCPEntry(i, new MethodRef(className, methodID));
					
				} else if (constantPool[i] instanceof ConstantDouble) {
					i++; // double constants take up two entries
				} else if (constantPool[i] instanceof ConstantLong) {
					i++; // long constants take up two entries
				}
			}
			
			for (Method method : classFile.getMethods()) {
				MethodID methodID = new MethodID(method.getName(), method.getSignature());
				if (method.isAbstract() || method.isNative()){
					continue;
				}
				model.addMethodName(methodID);
				int methodStart = model.getNextBytecodeIndex();
				
				byte[] code = method.getCode().getCode();
				
				// Determine how the addresses in the bytecode match up to addresses in the model
				// - this is needed to handle goto and if_icmple instructions correctly
				HashMap<Integer,Integer> bytecodeAddressMap = new HashMap<Integer,Integer>(code.length);
				int modelAddress = 0;
				int codeAddress = 0;
				while (codeAddress < code.length){
					switch (code[codeAddress]){
					case Const.ALOAD:
					case Const.ASTORE:
					case Const.BIPUSH:
					case Const.DLOAD:
					case Const.DSTORE:
					case Const.FLOAD:
					case Const.FSTORE:
					case Const.ILOAD:
					case Const.ISTORE:
					case Const.LDC:
					case Const.LLOAD:
					case Const.LSTORE:
					case (byte) Const.NEWARRAY:
					case (byte) Const.RET:
						// 1 parameter instructions
						bytecodeAddressMap.put(codeAddress, modelAddress);
						codeAddress += 2;
						modelAddress += 1;
						break;
					case (byte) Const.ANEWARRAY:
					case (byte) Const.GETFIELD:
					case (byte) Const.GETSTATIC:
					case (byte) Const.GOTO:
					case (byte) Const.IF_ACMPEQ:
					case (byte) Const.IF_ACMPNE:
					case (byte) Const.IF_ICMPEQ:
					case (byte) Const.IF_ICMPGE:
					case (byte) Const.IF_ICMPGT:
					case (byte) Const.IF_ICMPLE:
					case (byte) Const.IF_ICMPLT:
					case (byte) Const.IF_ICMPNE:
					case (byte) Const.IFEQ:
					case (byte) Const.IFGE:
					case (byte) Const.IFGT:
					case (byte) Const.IFLE:
					case (byte) Const.IFLT:
					case (byte) Const.IFNE:
					case (byte) Const.IFNONNULL:
					case (byte) Const.IFNULL:
					case (byte) Const.INSTANCEOF:
					case (byte) Const.INVOKESPECIAL:
					case (byte) Const.INVOKESTATIC:
					case (byte) Const.INVOKEVIRTUAL:
					case (byte) Const.JSR:
					case Const.LDC_W:
					case Const.LDC2_W:
					case (byte) Const.NEW:
					case (byte) Const.PUTFIELD:
					case (byte) Const.PUTSTATIC:
					case Const.SIPUSH:
						// 2 parameter instructions
						bytecodeAddressMap.put(codeAddress, modelAddress);
						codeAddress += 3;
						modelAddress += 1;
						break;
					case (byte) Const.MULTIANEWARRAY:
						// 3 parameter instructions
						bytecodeAddressMap.put(codeAddress, modelAddress);
						codeAddress += 4;
						modelAddress += 1;
						break;
					case (byte) Const.GOTO_W:
					case (byte) Const.INVOKEDYNAMIC:
					case (byte) Const.INVOKEINTERFACE:
					case (byte) Const.JSR_W:
						// 4 parameter instructions
						bytecodeAddressMap.put(codeAddress, modelAddress);
						codeAddress += 4;
						modelAddress += 1;
						break;
					case (byte) Const.IINC:
						bytecodeAddressMap.put(codeAddress, modelAddress);	
						codeAddress += 3;
						if (expandIINC) {
							modelAddress += 4; // we generate 3 extra instructions for iinc
						} else {
							modelAddress += 1;
						}
						break;
					case (byte) Const.CHECKCAST:
						bytecodeAddressMap.put(codeAddress, modelAddress);
						codeAddress += 3;
						// we don't generate a model instruction for checkcast
						break;
					case (byte) Const.L2I:
					case (byte) Const.I2L:
						bytecodeAddressMap.put(codeAddress, modelAddress);
						codeAddress += 1;
						modelAddress += 1;
						break;						
					default:
						// 0 parameter instructions
						bytecodeAddressMap.put(codeAddress, modelAddress);
						codeAddress += 1;
						modelAddress += 1;
						//TODO: need to handle lookupswitch and tableswitch
					}
				}
				
				for (int i = 0; i < code.length; i++) {
					switch (code[i]) {
					case Const.NOP:
						continue;
					case Const.ACONST_NULL:
						model.addBytecode(new BytecodeModel.ACONST_NULL());
						break;
					case Const.ALOAD:
						model.addBytecode(new BytecodeModel.ALOAD(code[i+1]));
						i++;
						break;
					case Const.ILOAD:
						model.addBytecode(new BytecodeModel.ALOAD(code[i+1]));
						i++;
						break;
					case Const.LLOAD:
						model.addBytecode(new BytecodeModel.LLOAD(code[i+1]));
						i++;
						break;
					case Const.ALOAD_0:
					case Const.ALOAD_1:
					case Const.ALOAD_2:
					case Const.ALOAD_3:
						model.addBytecode(new BytecodeModel.ALOAD(code[i]-Const.ALOAD_0));
						break;
					case Const.ILOAD_0:
					case Const.ILOAD_1:
					case Const.ILOAD_2:
					case Const.ILOAD_3:
						model.addBytecode(new BytecodeModel.ALOAD(code[i]-Const.ILOAD_0));
						break;
					case Const.LLOAD_0:
					case Const.LLOAD_1:
					case Const.LLOAD_2:
					case Const.LLOAD_3:
						model.addBytecode(new BytecodeModel.LLOAD(code[i]-Const.LLOAD_0));
						break;
					case Const.ASTORE:
						model.addBytecode(new BytecodeModel.ASTORE(code[i+1]));
						i++;
						break;
					case Const.ISTORE:
						model.addBytecode(new BytecodeModel.ASTORE(code[i+1]));
						i++;
						break;
					case Const.LSTORE:
						model.addBytecode(new BytecodeModel.LSTORE(code[i+1]));
						i++;
						break;
					case Const.ASTORE_0:
					case Const.ASTORE_1:
					case Const.ASTORE_2:
					case Const.ASTORE_3:
						model.addBytecode(new BytecodeModel.ASTORE(code[i]-Const.ASTORE_0));
						break;
					case Const.ISTORE_0:
					case Const.ISTORE_1:
					case Const.ISTORE_2:
					case Const.ISTORE_3:
						model.addBytecode(new BytecodeModel.ASTORE(code[i]-Const.ISTORE_0));
						break;
					case Const.LSTORE_0:
					case Const.LSTORE_1:
					case Const.LSTORE_2:
					case Const.LSTORE_3:
						model.addBytecode(new BytecodeModel.LSTORE(code[i]-Const.LSTORE_0));
						break;
					case (byte) Const.ARETURN:
					case (byte) Const.IRETURN:
						model.addBytecode(new BytecodeModel.ARETURN());
						break;
					case (byte) Const.LRETURN:
						model.addBytecode(new BytecodeModel.LRETURN());
						break;
					case Const.DUP:
						model.addBytecode(new BytecodeModel.DUP());
							break;
					case (byte) Const.GETFIELD:
						model.addBytecode(new BytecodeModel.GETFIELD((int) ((code[i+1] & 0xFF) << 8) | (code[i+2] & 0xFF)));
						i += 2;
						break;
					case (byte) Const.GETSTATIC:
						model.addBytecode(new BytecodeModel.GETSTATIC((int) ((code[i+1] & 0xFF) << 8) | (code[i+2] & 0xFF)));
						i += 2;
						break;
					case (byte) Const.GOTO:
						int offset = (int) ((code[i+1] & 0xFF) << 8) | (code[i+2] & 0xFF);
						int current = bytecodeAddressMap.get(i);
						//System.out.println(bytecodeAddressMap);
						//System.out.println(i+offset);
						int target = bytecodeAddressMap.get(i+offset);
						model.addBytecode(new BytecodeModel.GOTO(target-current));
						i += 2;
						break;
					case Const.IADD:
						model.addBytecode(new BytecodeModel.IADD());
						break;
					case Const.LADD:
						model.addBytecode(new BytecodeModel.LADD());
						break;
					case Const.IREM:
						model.addBytecode(new BytecodeModel.IREM());
						break;
					case Const.ISUB:
						model.addBytecode(new BytecodeModel.ISUB());
						break;
					case (byte) Const.IINC:
						int index = code[i+1];
						int constval = code[i+2];
						if (expandIINC) {
							model.addBytecode(new BytecodeModel.ALOAD(index));
							model.addBytecode(new BytecodeModel.ICONST(constval));
							model.addBytecode(new BytecodeModel.IADD());
							model.addBytecode(new BytecodeModel.ASTORE(index));
						} else {
							model.addBytecode(new BytecodeModel.IINC(index, constval));
						}
						i += 2;
						break;
					case Const.ICONST_0:
					case Const.ICONST_1:
					case Const.ICONST_2:
					case Const.ICONST_3:
					case Const.ICONST_4:
					case Const.ICONST_5:
					case Const.ICONST_M1:
						model.addBytecode(new BytecodeModel.ICONST(code[i]-Const.ICONST_0));
						break;
					case Const.BIPUSH:
						model.addBytecode(new BytecodeModel.ICONST(code[i+1]));
						i++;
						break;
					case Const.SIPUSH:
						model.addBytecode(new BytecodeModel.ICONST((int) ((code[i+1] & 0xFF) << 8) | (code[i+2] & 0xFF)));
						i += 2;
						break;
					case Const.LCONST_0:
					case Const.LCONST_1:
						model.addBytecode(new BytecodeModel.LCONST(code[i]-Const.LCONST_0));
						break;
					case Const.LDC:
						if (constantPool[code[i+1]] instanceof ConstantInteger) {
							model.addBytecode(new BytecodeModel.ICONST(((ConstantInteger) constantPool[code[i+1]]).getBytes()));
						} else {
							throw new UnsupportedOperationException("Non-integer constant pool access:" + inputFile  + ", " + method + ", " + i);
						}
						i++;
						break;
					case Const.LDC_W:
						model.addBytecode(new BytecodeModel.ICONST(((ConstantInteger) constantPool[((code[i+1] & 0xFF) << 8) | (code[i+2] & 0xFF)]).getBytes()));
						i += 2;
						break;
					case Const.LDC2_W:
						model.addBytecode(new BytecodeModel.LCONST(((ConstantLong) constantPool[((code[i+1] & 0xFF) << 8) | (code[i+2] & 0xFF)]).getBytes()));
						i += 2;
						break;
					case (byte) Const.IF_ICMPLE:
						offset = (code[i+1] << 8) | code[i+2];
						current = bytecodeAddressMap.get(i);
						target = bytecodeAddressMap.get(i+offset);
						model.addBytecode(new BytecodeModel.IF_ICMPLE(target-current));
						i += 2;
						break;
					case (byte) Const.IF_ICMPLT:
						offset = (code[i+1] << 8) | code[i+2];
						current = bytecodeAddressMap.get(i);
						target = bytecodeAddressMap.get(i+offset);
						model.addBytecode(new BytecodeModel.IF_ICMPLT(target-current));
						i += 2;
						break;
					case (byte) Const.IF_ICMPGT:
						offset = (code[i+1] << 8) | code[i+2];
						current = bytecodeAddressMap.get(i);
						target = bytecodeAddressMap.get(i+offset);
						model.addBytecode(new BytecodeModel.IF_ICMPGT(target-current));
						i += 2;
						break;
					case (byte) Const.IF_ICMPNE:
						offset = (code[i+1] << 8) | code[i+2];
						current = bytecodeAddressMap.get(i);
						target = bytecodeAddressMap.get(i+offset);
						model.addBytecode(new BytecodeModel.IF_ICMPNE(target-current));
						i += 2;
						break;
					case (byte) Const.IFEQ:
					case (byte) Const.IFNULL:
						offset = (code[i+1] << 8) | code[i+2];
						current = bytecodeAddressMap.get(i);
						target = bytecodeAddressMap.get(i+offset);
						model.addBytecode(new BytecodeModel.IFEQ(target-current));
						i += 2;
						break;
					case (byte) Const.IFNE:
					case (byte) Const.IFNONNULL:
						offset = (code[i+1] << 8) | code[i+2];
						current = bytecodeAddressMap.get(i);
						target = bytecodeAddressMap.get(i+offset);
						model.addBytecode(new BytecodeModel.IFNE(target-current));
						i += 2;
						break;
					case Const.INEG:
						model.addBytecode(new BytecodeModel.INEG());
						break;
					case (byte) Const.INVOKESPECIAL:
						model.addBytecode(new BytecodeModel.INVOKESPECIAL((int) ((code[i+1] & 0xFF) << 8) | (code[i+2] & 0xFF)));
						i += 2;
						break;
					case (byte) Const.INVOKESTATIC:
						model.addBytecode(new BytecodeModel.INVOKESTATIC((int) ((code[i+1] & 0xFF) << 8) | (code[i+2] & 0xFF)));
						i += 2;
						break;
					case (byte) Const.INVOKEVIRTUAL:
					case (byte) Const.INVOKEINTERFACE:
						model.addBytecode(new BytecodeModel.INVOKEVIRTUAL((int) ((code[i+1] & 0xFF) << 8) | (code[i+2] & 0xFF)));
						if (code[i] == (byte) Const.INVOKEINTERFACE){
							i += 4;
						} else {
							i += 2;
						}
						break;
					case (byte) Const.NEW:
						model.addBytecode(new BytecodeModel.NEW((int) ((code[i+1] & 0xFF) << 8) | (code[i+2] & 0xFF)));
						i += 2;
						break;
					case (byte) Const.PUTFIELD:
						model.addBytecode(new BytecodeModel.PUTFIELD((int) ((code[i+1] & 0xFF) << 8) | (code[i+2] & 0xFF)));
						i += 2;
						break;
					case (byte) Const.PUTSTATIC:
						model.addBytecode(new BytecodeModel.PUTSTATIC((int) ((code[i+1] & 0xFF) << 8) | (code[i+2] & 0xFF)));
						i += 2;
						break;
					case (byte) Const.RETURN:
						model.addBytecode(new BytecodeModel.RETURN());
						break;
					case (byte) Const.CHECKCAST:
						i +=2;
						break;
					case (byte) Const.L2I:
						model.addBytecode(new BytecodeModel.L2I());
						break;
						// throw new UnsupportedOperationException("unhandled long to integer conversion in file " + inputFile);
					case (byte) Const.I2L:
						model.addBytecode(new BytecodeModel.I2L());
						break;
						//throw new UnsupportedOperationException("unhandled integer to long conversion in file " + inputFile);
						// don't generate a model bytecode so don't need to do anything
						//break;
					default:
						throw new UnsupportedOperationException("Bytecode not supported by subset: " + Byte.toUnsignedInt(code[i]) + ", in file " + inputFile);
					}
				}
				
				classModel.addMethod(methodID, 
						methodStart, 
						model.getNextBytecodeIndex()-1,
						method.getCode().getMaxLocals(),
						method.getCode().getMaxStack(),
						method.isStatic(),
						method.isSynchronized());
			}
				
			for (Field field : classFile.getFields()) {
				FieldID fieldID = new FieldID(field.getName(), field.getSignature());
				if (field.isStatic()) {
					classModel.addStaticField(fieldID);
				} else {
					classModel.addField(fieldID);
				}
			}
				
			model.addClass(classModel);
		}
		
		FileWriter writer = new FileWriter(outputFile);
		
		if (addPreamble && !doC) {
			writer.write("\\documentclass[a4paper,10pt]{article}\n\n\\usepackage[cm]{fullpage}\n\\usepackage{circus}\n\n\\begin{document}\n\\tiny\n\n");
		}
		
		String sectionName = new File(outputFile).getName().substring(0, new File(outputFile).getName().indexOf('.'));
		if (!doC) { 
			if (doEPC || doEFS) {
				writer.write("\\begin{zsection}\n\t\\SECTION " + sectionName + " \\parents stack\\_frames, LIchans, memory\\_chans\n\\end{zsection}\n\n");
			} else {
				writer.write("\\begin{zsection}\n\t\\SECTION " + sectionName + " \\parents complete\\_cee\n\\end{zsection}\n\n");
			}
		}
		
		if (doC) {
			writer.write("#include \"" + outputFile.replace(".c", ".h") + "\"\n");
			writer.write(model.doEliminationOfProgramCounter().doEliminationOfFrameStack().toCCode());
			
			// output header file
			FileWriter headerWriter = new FileWriter(outputFile.replace(".c", ".h"));
			headerWriter.write(model.generateHeader(outputFile.replace(".c", ".h")));
			headerWriter.close();
		} else if (doEFS) {
			 writer.write(model.doEliminationOfProgramCounter().doEliminationOfFrameStack().toModelString());
		} else if (doEPC) {
			writer.write(model.doEliminationOfProgramCounter().toModelString());
		} else {
			writer.write(model.toModelString());
		} 
		
		if (addPreamble && !doC) {
			writer.write("\\end{document}\n");
		}
		
		writer.close();
		System.out.println("Output written to " + outputFile);
	}

}
