package main;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Vector;

import main.CPEntry.ClassRef;
import main.CPEntry.FieldRef;

public abstract class CircusAction {

	static String actionsToString(CircusAction[] actions, int indentLevel) {
		StringBuilder builder = new StringBuilder();
		
		for (int i = 0; i < actions.length; i++) {
			if (indentLevel > 0 && i > 0) {
				builder.append("\t\t");
				builder.append("\\t" + indentLevel + " ");
			}
			builder.append(actions[i].toModelString(indentLevel));
			if (i < actions.length-1) {
				builder.append(" \\circseq \\\\\n");
			}
		}
		
		// make sure unary negation is correctly handled
		return builder.toString().replaceAll("-(\\S)", "\\\\negate $1");
	}
	
	public static Object actionsToCCode(CircusAction[] actions, int indentLevel) {
		StringBuilder builder = new StringBuilder();
		
		boolean firstIndent = true;
		for (int i = 0; i < actions.length; i++) {
			if (!actions[i].toCCode(indentLevel).equals("")) {
				if (indentLevel > 0 && !firstIndent) {
					for (int j = 0; j < indentLevel; j++) {
						builder.append("\t");
					}
				}
				firstIndent = false;
			
				builder.append(actions[i].toCCode(indentLevel));
				if (i < actions.length-1) {
					builder.append("\n");
				}
			}
		}
		
		return builder.toString();
	}
	
	private CircusAction() {} // prevent further extension of this class
	
	public abstract String toModelString(int indentLevel);
	
	public abstract String toCCode(int indentLevel);
	
	public abstract CircusAction[] expandWithClassInfo(ClassModel classInfo);
	
	public abstract CircusAction[] doEFSDataRefinement(int stackDepth);
	
	public abstract int stackDepthChange();
	
	public static final class HandleBinOpEPC extends CircusAction {

		String operator;
		
		public HandleBinOpEPC(String operator) {
			this.operator = operator;
		}
		
		@Override
		public String toModelString(int indentLevel) {
			return "HandleBinOpEPC";
		}

		@Override
		public CircusAction[] expandWithClassInfo(ClassModel classInfo) {
			return new CircusAction[] {this};
		}

		@Override
		public CircusAction[] doEFSDataRefinement(int stackDepth) {
			return new CircusAction[] {new CircusAction.Assignment("stack" + (stackDepth-1), "stack" + stackDepth + " " + operator + " stack" + (stackDepth-1))};
		}

		@Override
		public int stackDepthChange() {
			return -1;
		}

		@Override
		public String toCCode(int indentLevel) {
			throw new UnsupportedOperationException("This should be eliminated before C code is produced");
		}
		
	}
	
	public static final class PoppedArgsAssignment extends CircusAction {
		private String[] args;

		public PoppedArgsAssignment(String[] args) {
			this.args = args;
		}
		
		public String[] getArgs() {
			return args;
		}

		@Override
		public String toModelString(int indentLevel) {
			StringBuilder buffer = new StringBuilder();
			buffer.append("poppedArgs := \\langle ");
			for (int i = 0; i < args.length; i++) {
				buffer.append(args[i]);
				if (i < args.length-1) {
					buffer.append(", ");
				}
			}
			buffer.append(" \\rangle");
			return buffer.toString();
		}

		@Override
		public CircusAction[] expandWithClassInfo(ClassModel classInfo) {
			return new CircusAction[] {this};
		}

		@Override
		public CircusAction[] doEFSDataRefinement(int stackDepth) {
			return new CircusAction[] {this};
		}

		@Override
		public int stackDepthChange() {
			return -args.length;
		}

		@Override
		public String toCCode(int indentLevel) {
			throw new UnsupportedOperationException("This should be eliminated before C code is produced");
		}

	}
	
	public static final class PCAssignment extends CircusAction {
		private int target;
		public PCAssignment(int target) {
			this.target = target;
		}
		
		public int getTarget() {
			return target;
		}

		@Override
		public String toModelString(int indentLevel) {
			return "pc := " + target;
		}

		@Override
		public CircusAction[] doEFSDataRefinement(int stackDepth) {
			throw new UnsupportedOperationException("Can't perform frame stack elimination on a pc assignment");
		}

		@Override
		public int stackDepthChange() {
			return 0;
		}

		@Override
		public CircusAction[] expandWithClassInfo(ClassModel classInfo) {
			throw new UnsupportedOperationException("Can't perform frame stack elimination on a pc assignment");
		}
		
		@Override
		public String toCCode(int indentLevel) {
			throw new UnsupportedOperationException("This should be eliminated before C code is produced");
		}
	}
	
	public static final class ConditionalPCAssignment extends CircusAction {
		private final String condition;
		private final int target1, target2;
		public ConditionalPCAssignment(String condition, int target1, int target2) {
			this.condition = condition;
			this.target1 = target1;
			this.target2 = target2;
		}
		
		public String getCondition() {
			return condition;
		}
		
		public int getTrueTarget() {
			return target1;
		}
		
		public int getFalseTarget() {
			return target2;
		}

		@Override
		public String toModelString(int indentLevel) {
			return "pc := \\IF " + condition + " \\THEN " + target1 + " \\ELSE " + target2;
		}

		@Override
		public CircusAction[] doEFSDataRefinement(int stackDepth) {
			throw new UnsupportedOperationException("Can't perform frame stack elimination on a conditional pc assignment");
		}

		@Override
		public int stackDepthChange() {
			return 0;
		}

		@Override
		public CircusAction[] expandWithClassInfo(ClassModel classInfo) {
			throw new UnsupportedOperationException("Can't perform frame stack elimination on a conditional pc assignment");
		}
		
		@Override
		public String toCCode(int indentLevel) {
			throw new UnsupportedOperationException("This should be eliminated before C code is produced");
		}
	}
	
	public static final class Assignment extends CircusAction {
		private String var;
		private String expr;
		public Assignment(String var, String expr) {
			this.var = var;
			this.expr = expr;
		}
		public String getVar() {
			return var;
		}
		public String getExpr() {
			return expr;
		}
		@Override
		public String toModelString(int indentLevel) {
			return var + " := " + expr;
		}
		@Override
		public CircusAction[] doEFSDataRefinement(int stackDepth) {
			// assignments are generated by EFS so they shouldn't need data refining
			return new CircusAction[] {this};
		}
		@Override
		public int stackDepthChange() {
			return 0;
		}
		@Override
		public CircusAction[] expandWithClassInfo(ClassModel classInfo) {
			return new CircusAction[] {this};
		}
		
		@Override
		public String toCCode(int indentLevel) {
			if (var.equals("retVal") || var.equals("retVal\\_msb") || var.equals("retVal\\_lsb")) {
				// return values are really pointers
				return "*" + var + " = " + (expr.replace("\\mod", "%").replace("null", "0")) + ";";
			} else {
				return var + " = " + (expr.replace("\\mod", "%").replace("null", "0")) + ";";
			}
		}
	}
	
	public static final class Assumption extends CircusAction {
		private String condition;
		public Assumption(String condition) {
			this.condition = condition;
		}
		
		public String getPredicate() {
			return condition;
		}

		@Override
		public String toModelString(int indentLevel) {
			return "\\{ " + condition + " \\}";
		}

		@Override
		public CircusAction[] doEFSDataRefinement(int stackDepth) {
			// assumptions are generated by EFS so they shouldn't need data refining
			return new CircusAction[] {this};
		}

		@Override
		public int stackDepthChange() {
			return 0;
		}

		@Override
		public CircusAction[] expandWithClassInfo(ClassModel classInfo) {
			return new CircusAction[] {this};
		}
		
		@Override
		public String toCCode(int indentLevel) {
			throw new UnsupportedOperationException("This should be eliminated before C code is produced");
		}
		
	}
	
	public static final class PCAssumption extends CircusAction {
		private int pcValue;
		public PCAssumption(int pcValue) {
			this.pcValue = pcValue;
		}
		
		public int getPCValue() {
			return pcValue;
		}

		@Override
		public String toModelString(int indentLevel) {
			return "\\{ pc = " + pcValue + " \\}";
		}

		@Override
		public CircusAction[] doEFSDataRefinement(int stackDepth) {
			throw new UnsupportedOperationException("Can't perform frame stack elimination on a pc assumption");
		}

		@Override
		public int stackDepthChange() {
			return 0;
		}

		@Override
		public CircusAction[] expandWithClassInfo(ClassModel classInfo) {
			throw new UnsupportedOperationException("Can't perform frame stack elimination on a pc assumption");
		}
		
		@Override
		public String toCCode(int indentLevel) {
			throw new UnsupportedOperationException("This should be eliminated before C code is produced");
		}
	}
	
	public static final class HandleAconst_nullEPC extends CircusAction {
		public HandleAconst_nullEPC() {
			
		}

		@Override
		public String toModelString(int indentLevel) {
			return "HandleAconst\\_nullEPC";
		}

		@Override
		public CircusAction[] doEFSDataRefinement(int stackDepth) {
			return new CircusAction[] {new CircusAction.Assignment("stack" + (stackDepth+1), "null")};
		}

		@Override
		public int stackDepthChange() {
			return 1;
		}

		@Override
		public CircusAction[] expandWithClassInfo(ClassModel classInfo) {
			return new CircusAction[] {this};
		}
		
		@Override
		public String toCCode(int indentLevel) {
			throw new UnsupportedOperationException("This should be eliminated before C code is produced");
		}
		
	}
	
	public static final class HandleDupEPC extends CircusAction {
		public HandleDupEPC() {
			
		}

		@Override
		public String toModelString(int indentLevel) {
			return "HandleDupEPC";
		}

		@Override
		public CircusAction[] doEFSDataRefinement(int stackDepth) {
			return new CircusAction[] {new CircusAction.Assignment("stack" + (stackDepth+1), "stack" + stackDepth)};
		}

		@Override
		public int stackDepthChange() {
			return 1;
		}

		@Override
		public CircusAction[] expandWithClassInfo(ClassModel classInfo) {
			return new CircusAction[] {this};
		}
		
		@Override
		public String toCCode(int indentLevel) {
			throw new UnsupportedOperationException("This should be eliminated before C code is produced");
		}
	}
	
	public static final class HandleAloadEPC extends CircusAction {
		private int lvi;
		public HandleAloadEPC(int lvi) {
			this.lvi = lvi;
		}
		
		public int getLocalVarIndex() {
			return lvi;
		}
		
		@Override
		public String toModelString(int indentLevel) {
			return "HandleAloadEPC(" + lvi + ")";
		}

		@Override
		public CircusAction[] doEFSDataRefinement(int stackDepth) {
			return new CircusAction[] {new CircusAction.Assignment("stack" + (stackDepth+1), "var" + (lvi+1))};
		}

		@Override
		public int stackDepthChange() {
			return 1;
		}

		@Override
		public CircusAction[] expandWithClassInfo(ClassModel classInfo) {
			return new CircusAction[] {this};
		}
		
		@Override
		public String toCCode(int indentLevel) {
			throw new UnsupportedOperationException("This should be eliminated before C code is produced");
		}
	}
	
	public static final class HandleLloadEPC extends CircusAction {
		private int lvi;
		public HandleLloadEPC(int lvi) {
			this.lvi = lvi;
		}
		
		public int getLocalVarIndex() {
			return lvi;
		}
		
		@Override
		public String toModelString(int indentLevel) {
			return "HandleLloadEPC(" + lvi + ")";
		}

		@Override
		public CircusAction[] doEFSDataRefinement(int stackDepth) {
			return new CircusAction[] {
					new CircusAction.Assignment("stack" + (stackDepth+1), "var" + (lvi+1)),
					new CircusAction.Assignment("stack" + (stackDepth+2), "var" + (lvi+2))
				};
		}

		@Override
		public int stackDepthChange() {
			return 2;
		}

		@Override
		public CircusAction[] expandWithClassInfo(ClassModel classInfo) {
			return new CircusAction[]{this};
		}
		
		@Override
		public String toCCode(int indentLevel) {
			throw new UnsupportedOperationException("This should be eliminated before C code is produced");
		}
	}
	
	public static final class HandleAstoreEPC extends CircusAction {
		private int lvi;
		public HandleAstoreEPC(int lvi) {
			this.lvi = lvi;
		}
		
		public int getLocalVarIndex() {
			return lvi;
		}
		
		@Override
		public String toModelString(int indentLevel) {
			return "HandleAstoreEPC(" + lvi + ")";
		}

		@Override
		public CircusAction[] doEFSDataRefinement(int stackDepth) {
			return new CircusAction[] {new CircusAction.Assignment("var" + (lvi+1), "stack" + (stackDepth))};
		}

		@Override
		public int stackDepthChange() {
			return -1;
		}

		@Override
		public CircusAction[] expandWithClassInfo(ClassModel classInfo) {
			return new CircusAction[] {this};
		}
		
		@Override
		public String toCCode(int indentLevel) {
			throw new UnsupportedOperationException("This should be eliminated before C code is produced");
		}
	}
	
	public static final class HandleLstoreEPC extends CircusAction {
		private int lvi;
		public HandleLstoreEPC(int lvi) {
			this.lvi = lvi;
		}
		
		public int getLocalVarIndex() {
			return lvi;
		}
		
		@Override
		public String toModelString(int indentLevel) {
			return "HandleLstoreEPC(" + lvi + ")";
		}

		@Override
		public CircusAction[] doEFSDataRefinement(int stackDepth) {
			return new CircusAction[] {
					new CircusAction.Assignment("var" + (lvi+1), "stack" + (stackDepth-1)),
					new CircusAction.Assignment("var" + (lvi+2), "stack" + (stackDepth))
				};
		}

		@Override
		public int stackDepthChange() {
			return -1;
		}

		@Override
		public CircusAction[] expandWithClassInfo(ClassModel classInfo) {
			return new CircusAction[]{this};
		}
		
		@Override
		public String toCCode(int indentLevel) {
			throw new UnsupportedOperationException("This should be eliminated before C code is produced");
		}
	}
	
	public static final class HandleIaddEPC extends CircusAction {
		public HandleIaddEPC() {
			
		}

		@Override
		public String toModelString(int indentLevel) {
			return "HandleIaddEPC";
		}

		@Override
		public CircusAction[] doEFSDataRefinement(int stackDepth) {
			return new CircusAction[] {new CircusAction.Assignment("stack" + (stackDepth-1), "stack" + stackDepth + " + stack" + (stackDepth-1))};
		}

		@Override
		public int stackDepthChange() {
			return -1;
		}

		@Override
		public CircusAction[] expandWithClassInfo(ClassModel classInfo) {
			return new CircusAction[] {this};
		}
		
		@Override
		public String toCCode(int indentLevel) {
			throw new UnsupportedOperationException("This should be eliminated before C code is produced");
		}
	}
	
	public static final class HandleLaddEPC extends CircusAction {
		public HandleLaddEPC() {
			
		}

		@Override
		public String toModelString(int indentLevel) {
			return "HandleLaddEPC";
		}

		@Override
		public CircusAction[] doEFSDataRefinement(int stackDepth) {
			return new CircusAction[] {new CircusAction.AddLong("stack" + (stackDepth-3), "stack" + (stackDepth-2), " + stack" + (stackDepth-1), " + stack" + stackDepth)};
		}

		@Override
		public int stackDepthChange() {
			return -2;
		}

		@Override
		public CircusAction[] expandWithClassInfo(ClassModel classInfo) {
			return new CircusAction[]{this};
		}
		
		@Override
		public String toCCode(int indentLevel) {
			throw new UnsupportedOperationException("This should be eliminated before C code is produced");
		}
	}
	
	public static class HandleLtoIEPC extends CircusAction {

		public HandleLtoIEPC() {
		}
		
		@Override
		public String toModelString(int indentLevel) {
			return "HandleLtoIEPC";
		}

		@Override
		public String toCCode(int indentLevel) {
			throw new UnsupportedOperationException("This should be eliminated before C code is produced");
		}

		@Override
		public CircusAction[] expandWithClassInfo(ClassModel classInfo) {
			return new CircusAction[]{this};
		}

		@Override
		public CircusAction[] doEFSDataRefinement(int stackDepth) {
			return new CircusAction[]{new CircusAction.Assignment("stack" + (stackDepth-1), "stack" + stackDepth)};
		}

		@Override
		public int stackDepthChange() {
			return -1;
		}

	}
	
	public static final class AddLong extends CircusAction {
		private String msb1;
		private String lsb1;
		private String msb2;
		private String lsb2;
		public AddLong(String msb1, String lsb1, String msb2, String lsb2) {
			this.msb1 = msb1;
			this.lsb1 = lsb1;
			this.msb2 = msb2;
			this.lsb2 = lsb2;
		}

		@Override
		public String toModelString(int indentLevel) {
			return "AddLong(" + msb1 + ", " + lsb1 + ", " + msb2 + ", " + lsb2 + ")";
		}

		@Override
		public CircusAction[] doEFSDataRefinement(int stackDepth) {
			return new CircusAction[]{this};
		}

		@Override
		public int stackDepthChange() {
			return -2;
		}

		@Override
		public CircusAction[] expandWithClassInfo(ClassModel classInfo) {
			return new CircusAction[]{this};
		}
		
		@Override
		public String toCCode(int indentLevel) {
			return "addLong(&" + msb1 + ", &" + lsb1 + ", " + msb2 + ", " + lsb2 + ");";
		}
	}
	
	public static final class HandleInegEPC extends CircusAction {
		public HandleInegEPC() {
			
		}

		@Override
		public String toModelString(int indentLevel) {
			return "HandleInegEPC";
		}

		@Override
		public CircusAction[] doEFSDataRefinement(int stackDepth) {
			return new CircusAction[] {new CircusAction.Assignment("stack" + stackDepth, "\\negate stack" + stackDepth)};
		}

		@Override
		public int stackDepthChange() {
			return 0;
		}

		@Override
		public CircusAction[] expandWithClassInfo(ClassModel classInfo) {
			return new CircusAction[] {this};
		}
		
		@Override
		public String toCCode(int indentLevel) {
			throw new UnsupportedOperationException("This should be eliminated before C code is produced");
		}
	}

	public static final class HandleIincEPC extends CircusAction {
		private int variableIndex;
		private int constantValue;
		
		public HandleIincEPC(int variableIndex, int constantValue) {
			this.variableIndex = variableIndex;
			this.constantValue = constantValue;
		}

		@Override
		public String toModelString(int indentLevel) {
			return "HandleIincEPC(" + variableIndex + ", " + constantValue + ")";
		}

		@Override
		public CircusAction[] doEFSDataRefinement(int stackDepth) {
			return new CircusAction[] {new CircusAction.Assignment("var" + (variableIndex+1), "var" + (variableIndex+1) + " + " + constantValue)};
		}

		@Override
		public int stackDepthChange() {
			return 0;
		}

		@Override
		public CircusAction[] expandWithClassInfo(ClassModel classInfo) {
			return new CircusAction[] {this};
		}
		
		@Override
		public String toCCode(int indentLevel) {
			throw new UnsupportedOperationException("This should be eliminated before C code is produced");
		}
	}
	
	public static final class HandleIconstEPC extends CircusAction {
		private int n;
		public HandleIconstEPC(int n) {
			this.n = n;
		}
		
		public int getNum() {
			return n;
		}
		
		@Override
		public String toModelString(int indentLevel) {
			return "HandleIconstEPC(" + n + ")";
		}

		@Override
		public CircusAction[] doEFSDataRefinement(int stackDepth) {
			return new CircusAction[] {new CircusAction.Assignment("stack" + (stackDepth+1), Integer.toString(n))};
		}

		@Override
		public int stackDepthChange() {
			return 1;
		}

		@Override
		public CircusAction[] expandWithClassInfo(ClassModel classInfo) {
			return new CircusAction[] {this};
		}
		
		@Override
		public String toCCode(int indentLevel) {
			throw new UnsupportedOperationException("This should be eliminated before C code is produced");
		}
	}
	
	public static final class HandleLconstEPC extends CircusAction {
		private long n;
		public HandleLconstEPC(long n) {
			this.n = n;
		}
		
		public long getNum() {
			return n;
		}
		
		@Override
		public String toModelString(int indentLevel) {
			return "HandleLconstEPC(" + n + ")";
		}

		@Override
		public CircusAction[] doEFSDataRefinement(int stackDepth) {
			int msb = (int) (n >> 32);
			int lsb;
			if (n > Integer.MAX_VALUE) {
				lsb = Integer.MAX_VALUE; 
			} else if (n < Integer.MIN_VALUE) {
				lsb = Integer.MIN_VALUE; 
			} else {
				lsb = (int) n;
			}
			return new CircusAction[] {
					new CircusAction.Assignment("stack" + (stackDepth+1), Integer.toString(msb)),
					new CircusAction.Assignment("stack" + (stackDepth+2), Integer.toString(lsb))
					};
		}

		@Override
		public int stackDepthChange() {
			return 2;
		}

		@Override
		public CircusAction[] expandWithClassInfo(ClassModel classInfo) {
			return new CircusAction[] {this};
		}
		
		@Override
		public String toCCode(int indentLevel) {
			throw new UnsupportedOperationException("This should be eliminated before C code is produced");
		}
	}
	
	public static final class InterpreterPopEPC extends CircusAction {
		private String var_name;
		public InterpreterPopEPC(String var_name) {
			this.var_name = var_name;
		}
		
		public String getVarName() {
			return var_name;
		}
		
		@Override
		public String toModelString(int indentLevel) {
			if (var_name == null) {
				return "\\lschexpract InterpreterPopEPC \\rschexpract";
			} else {
				return "\\lschexpract InterpreterPopEPC[" + var_name + "/value!] \\rschexpract";
			}
		}

		@Override
		public CircusAction[] doEFSDataRefinement(int stackDepth) {
			if (var_name == null) {
				return new CircusAction[] {new CircusAction.Assignment("value", "stack" + stackDepth)};
			} else {
				return new CircusAction[] {new CircusAction.Assignment(var_name, "stack" + stackDepth)};
			}
		}

		@Override
		public int stackDepthChange() {
			return -1;
		}

		@Override
		public CircusAction[] expandWithClassInfo(ClassModel classInfo) {
			return new CircusAction[] {this};
		}
		
		@Override
		public String toCCode(int indentLevel) {
			throw new UnsupportedOperationException("This should be eliminated before C code is produced");
		}
	}
	
	public static final class InterpreterPushEPC extends CircusAction {
		private String var_name;
		public InterpreterPushEPC(String var_name) {
			this.var_name = var_name;
		}
		
		public String getVarName() {
			return var_name;
		}
		
		@Override
		public String toModelString(int indentLevel) {
			if (var_name == null) {
				return "\\lschexpract InterpreterPushEPC \\rschexpract";
			} else {
				return "\\lschexpract InterpreterPushEPC[" + var_name + "/value?] \\rschexpract";
			}
		}

		@Override
		public CircusAction[] doEFSDataRefinement(int stackDepth) {
			if (var_name == null) {
				return new CircusAction[] {new CircusAction.Assignment("stack" + (stackDepth+1), "value")};
			} else {
				return new CircusAction[] {new CircusAction.Assignment("stack" + (stackDepth+1), var_name)};
			}
		}

		@Override
		public int stackDepthChange() {
			return 1;
		}

		@Override
		public CircusAction[] expandWithClassInfo(ClassModel classInfo) {
			return new CircusAction[] {this};
		}
		
		@Override
		public String toCCode(int indentLevel) {
			throw new UnsupportedOperationException("This should be eliminated before C code is produced");
		}
	}
	
	public static final class InterpreterStackFrameInvokeEPC extends CircusAction {
		private int num_args;
		public InterpreterStackFrameInvokeEPC(int num_args) {
			this.num_args = num_args;
		}
		
		public int getNumArgs() {
			return num_args;
		}
		
		@Override
		public String toModelString(int indentLevel) {
			return "\\lschexpract \\exists argsToPop? == " + num_args + " @ InterpreterStackFrameInvokeEPC \\rschexpract";
		}

		@Override
		public CircusAction[] doEFSDataRefinement(int stackDepth) {
			String[] poppedArgsArray = new String[num_args]; 
			//poppedArgsArray.append("\\langle ");
			int x = 0;
			for (int i = -num_args+1; i <= 0; i++) {
				poppedArgsArray[x] = "stack" + (stackDepth+i);
				x++;
				//poppedArgsArray.append("stack" + (stackDepth+i));
				//if (i != 0) {
				//	poppedArgsArray.append(", ");
				//}
			}
			//poppedArgsArray.append(" \\rangle");
			
			return new CircusAction[] {new CircusAction.PoppedArgsAssignment(poppedArgsArray)};
		}

		@Override
		public int stackDepthChange() {
			return -num_args;
		}

		@Override
		public CircusAction[] expandWithClassInfo(ClassModel classInfo) {
			return new CircusAction[] {this};
		}
		
		@Override
		public String toCCode(int indentLevel) {
			throw new UnsupportedOperationException("This should be eliminated before C code is produced");
		}
	}
	
	public static final class HandleAreturnEPC extends CircusAction {
		public HandleAreturnEPC() {
			
		}

		@Override
		public String toModelString(int indentLevel) {
			return "HandleAreturnEPC";
		}

		@Override
		public CircusAction[] doEFSDataRefinement(int stackDepth) {
			return new CircusAction[] {new CircusAction.Assignment("retVal", "stack" + stackDepth)};
		}

		@Override
		public int stackDepthChange() {
			return -1;
		}

		@Override
		public CircusAction[] expandWithClassInfo(ClassModel classInfo) {
			throw new UnsupportedOperationException("HandleAreturnEPC should have been eliminated by this point");
		}
		
		@Override
		public String toCCode(int indentLevel) {
			throw new UnsupportedOperationException("This should be eliminated before C code is produced");
		}
	}
	
	public static final class HandleLreturnEPC extends CircusAction {
		public HandleLreturnEPC() {
			
		}

		@Override
		public String toModelString(int indentLevel) {
			return "HandleLreturnEPC";
		}

		@Override
		public CircusAction[] doEFSDataRefinement(int stackDepth) {
			return new CircusAction[] {
					new CircusAction.Assignment("retVal\\_lsb", "stack" + stackDepth),
					new CircusAction.Assignment("retVal\\_msb", "stack" + (stackDepth-1))
					};
		}

		@Override
		public int stackDepthChange() {
			return -2;
		}

		@Override
		public CircusAction[] expandWithClassInfo(ClassModel classInfo) {
			throw new UnsupportedOperationException("HandleLreturnEPC should have been eliminated by this point");
		}
		
		@Override
		public String toCCode(int indentLevel) {
			throw new UnsupportedOperationException("This should be eliminated before C code is produced");
		}
	}
	
	public static final class HandleReturnEPC extends CircusAction {
		public HandleReturnEPC() {
			
		}

		@Override
		public String toModelString(int indentLevel) {
			return "HandleReturnEPC";
		}

		@Override
		public CircusAction[] doEFSDataRefinement(int stackDepth) {
			return new CircusAction[] {new CircusAction.Skip()};
		}

		@Override
		public int stackDepthChange() {
			return 0;
		}

		@Override
		public CircusAction[] expandWithClassInfo(ClassModel classInfo) {
			throw new UnsupportedOperationException("HandleReturnEPC should have been eliminated by this point");
		}
		
		@Override
		public String toCCode(int indentLevel) {
			throw new UnsupportedOperationException("This should be eliminated before C code is produced");
		}
	}
	
	public static final class HandleNewEPC extends CircusAction {
		private int cpi;
		public HandleNewEPC(int cpi) {
			this.cpi = cpi;
		}
		
		public int getConstantPoolIndex() {
			return cpi;
		}
		
		@Override
		public String toModelString(int indentLevel) {
			return "HandleNewEPC(" + cpi + ")";
		}

		@Override
		public CircusAction[] doEFSDataRefinement(int stackDepth) {
			throw new UnsupportedOperationException("HandleNewEPC must be refined with class information before applying EFS data refinement");
		}

		@Override
		public int stackDepthChange() {
			return 1;
		}

		@Override
		public CircusAction[] expandWithClassInfo(ClassModel classInfo) {
			ClassRef classRef = (ClassRef) classInfo.getCPEntry(cpi);
			String cid = classRef.getClassID().toString() + "ID";
			
			return new CircusAction[] {
				new CircusAction.ChannelComm("newObject", new String[] {"thread", cid}, new boolean[] {false, false} , 
					new CircusAction.ChannelComm("newObjectRet", new String[] {"oid"}, new boolean[] {true}, 
							new CircusAction.InterpreterPushEPC("oid")))
				};
		}
		
		@Override
		public String toCCode(int indentLevel) {
			throw new UnsupportedOperationException("This should be eliminated before C code is produced");
		}
	}
	
	public static final class HandleGetfieldEPC extends CircusAction {
		private int cpi;
		public HandleGetfieldEPC(int cpi) {
			this.cpi = cpi;
		}
		
		public int getConstantPoolIndex() {
			return cpi;
		}
		
		@Override
		public String toModelString(int indentLevel) {
			return "HandleGetfieldEPC(" + cpi + ")";
		}

		@Override
		public CircusAction[] doEFSDataRefinement(int stackDepth) {
			throw new UnsupportedOperationException("HandleGetfieldEPC must be refined with class information before applying EFS data refinement");
		}

		@Override
		public int stackDepthChange() {
			return 0;
		}

		@Override
		public CircusAction[] expandWithClassInfo(ClassModel classInfo) {
			FieldRef fieldRef = (FieldRef) classInfo.getCPEntry(cpi);
			String cid = fieldRef.getClassID().toString() + "ID";
			String fid = fieldRef.getFieldID().toString();
			
			if (fieldRef.getFieldID().isLongOrDouble()){
				// need to get the values from two separate fields for a long or double
				// - treat it as two getfields but with one var block
				return new CircusAction[]{
						new CircusAction.VarBlock("oid", new CircusAction[] {
						new CircusAction.InterpreterPopEPC("oid"),
						new CircusAction.ChannelComm("getField", new String[] {"oid", cid, fid + "_msb"}, new boolean[] {false, false, false}, 
								new CircusAction.ChannelComm("getFieldRet", new String[] {"value"}, new boolean[] {true}, 
										new CircusAction.InterpreterPushEPC(null))),
						new CircusAction.ChannelComm("getField", new String[] {"oid", cid, fid + "_lsb"}, new boolean[] {false, false, false}, 
								new CircusAction.ChannelComm("getFieldRet", new String[] {"value"}, new boolean[] {true}, 
									new CircusAction.InterpreterPushEPC(null)))
						})
					};
			}
			
			return new CircusAction[]{
					new CircusAction.VarBlock("oid", new CircusAction[] {
					new CircusAction.InterpreterPopEPC("oid"),
					new CircusAction.ChannelComm("getField", new String[] {"oid", cid, fid}, new boolean[] {false, false, false}, 
							new CircusAction.ChannelComm("getFieldRet", new String[] {"value"}, new boolean[] {true}, 
									new CircusAction.InterpreterPushEPC(null)))
			})};
		}
		
		@Override
		public String toCCode(int indentLevel) {
			throw new UnsupportedOperationException("This should be eliminated before C code is produced");
		}
	}
	
	public static final class HandlePutfieldEPC extends CircusAction {
		private int cpi;
		public HandlePutfieldEPC(int cpi) {
			this.cpi = cpi;
		}
		
		public int getConstantPoolIndex() {
			return cpi;
		}
		
		@Override
		public String toModelString(int indentLevel) {
			return "HandlePutfieldEPC(" + cpi + ")";
		}

		@Override
		public CircusAction[] doEFSDataRefinement(int stackDepth) {
			throw new UnsupportedOperationException("HandlePutfieldEPC must be refined with class information before applying EFS data refinement");
		}

		@Override
		public int stackDepthChange() {
			return -2;
		}

		@Override
		public CircusAction[] expandWithClassInfo(ClassModel classInfo) {
			FieldRef fieldRef = (FieldRef) classInfo.getCPEntry(cpi);
			String cid = fieldRef.getClassID().toString() + "ID";
			String fid = fieldRef.getFieldID().toString();
			
			if (fieldRef.getFieldID().isLongOrDouble()){
				return new CircusAction[]{
						new CircusAction.VarBlock("oid, lsb, msb", new CircusAction[] {
						new CircusAction.InterpreterPopEPC("lsb"),
						new CircusAction.InterpreterPopEPC("msb"),
						new CircusAction.InterpreterPopEPC("oid"),
						new CircusAction.ChannelComm("putField", new String[] {"oid", cid, fid + "_lsb", "lsb"}, new boolean[] {false, false, false, false}, 
								new CircusAction.Skip()),
						new CircusAction.ChannelComm("putField", new String[] {"oid", cid, fid + "_msb", "msb"}, new boolean[] {false, false, false, false}, 
								new CircusAction.Skip())
				})};
			} else {
				return new CircusAction[]{
						new CircusAction.VarBlock("oid, value", new CircusAction[] {
						new CircusAction.InterpreterPopEPC(null),
						new CircusAction.InterpreterPopEPC("oid"),
						new CircusAction.ChannelComm("putField", new String[] {"oid", cid, fid, "value"}, new boolean[] {false, false, false, false}, 
								new CircusAction.Skip())
				})};
			}
		}
		
		@Override
		public String toCCode(int indentLevel) {
			throw new UnsupportedOperationException("This should be eliminated before C code is produced");
		}
	}
	
	public static final class HandleGetstaticEPC extends CircusAction {
		private int cpi;
		public HandleGetstaticEPC(int cpi) {
			this.cpi = cpi;
		}

		public int getConstantPoolIndex() {
			return cpi;
		}
		
		@Override
		public String toModelString(int indentLevel) {
			return "HandleGetstaticEPC(" + cpi + ")";
		}

		@Override
		public CircusAction[] doEFSDataRefinement(int stackDepth) {
			throw new UnsupportedOperationException("HandleGetstaticEPC must be refined with class information before applying EFS data refinement");
		}

		@Override
		public int stackDepthChange() {
			return 1;
		}

		@Override
		public CircusAction[] expandWithClassInfo(ClassModel classInfo) {
			FieldRef fieldRef = (FieldRef) classInfo.getCPEntry(cpi);
			String cid = fieldRef.getClassID().toString() + "ID";
			String fid = fieldRef.getFieldID().toString();
			
			if (fieldRef.getFieldID().isLongOrDouble()){
				return new CircusAction[]{
						new CircusAction.ChannelComm("getStatic", new String[] {cid, fid + "_msb"}, new boolean[] {false, false}, 
								new CircusAction.ChannelComm("getStaticRet", new String[] {"value"}, new boolean[] {true}, 
										new CircusAction.InterpreterPushEPC(null))),
						new CircusAction.ChannelComm("getStatic", new String[] {cid, fid + "_lsb"}, new boolean[] {false, false}, 
								new CircusAction.ChannelComm("getStaticRet", new String[] {"value"}, new boolean[] {true}, 
										new CircusAction.InterpreterPushEPC(null)))
						};
			} else {
				return new CircusAction[]{
						new CircusAction.ChannelComm("getStatic", new String[] {cid, fid}, new boolean[] {false, false}, 
								new CircusAction.ChannelComm("getStaticRet", new String[] {"value"}, new boolean[] {true}, 
										new CircusAction.InterpreterPushEPC(null))),
						};
			}
		}
		
		@Override
		public String toCCode(int indentLevel) {
			throw new UnsupportedOperationException("This should be eliminated before C code is produced");
		}
	}
	
	public static final class HandlePutstaticEPC extends CircusAction {
		private int cpi;
		public HandlePutstaticEPC(int cpi) {
			this.cpi = cpi;
		}
		
		public int getConstantPoolIndex() {
			return cpi;
		}
		
		@Override
		public String toModelString(int indentLevel) {
			return "HandlePutstaticEPC(" + cpi + ")";
		}

		@Override
		public CircusAction[] doEFSDataRefinement(int stackDepth) {
			throw new UnsupportedOperationException("HandlePutstaticEPC must be refined with class information before applying EFS data refinement");
		}

		@Override
		public int stackDepthChange() {
			return 1;
		}

		@Override
		public CircusAction[] expandWithClassInfo(ClassModel classInfo) {
			FieldRef fieldRef = (FieldRef) classInfo.getCPEntry(cpi);
			String cid = fieldRef.getClassID().toString() + "ID";
			String fid = fieldRef.getFieldID().toString();
			
			if (fieldRef.getFieldID().isLongOrDouble()){
				return new CircusAction[] {
						new CircusAction.VarBlock("lsb, msb", new CircusAction[] {
							new CircusAction.InterpreterPopEPC("lsb"),
							new CircusAction.InterpreterPopEPC("msb"),
							new CircusAction.ChannelComm("putStatic", new String[] {cid, fid + "_lsb", "lsb"}, new boolean[] {false, false, false},
									new CircusAction.Skip()),
							new CircusAction.ChannelComm("putStatic", new String[] {cid, fid + "_msb", "msb"}, new boolean[] {false, false, false},
									new CircusAction.Skip())
						
						})
					};
			} else {
				return new CircusAction[] {
						new CircusAction.VarBlock("value", new CircusAction[] {
							new CircusAction.InterpreterPopEPC(null),
							new CircusAction.ChannelComm("putStatic", new String[] {cid, fid, "value"}, new boolean[] {false, false, false},
									new CircusAction.Skip())
						
						})
					};
			}
		}
		
		@Override
		public String toCCode(int indentLevel) {
			throw new UnsupportedOperationException("This should be eliminated before C code is produced");
		}
	}
	
	public static abstract class HandleInvokeEPC extends CircusAction {
		private HandleInvokeEPC() {}
		
		public abstract int getConstantPoolIndex();
	}
	
	public static final class HandleInvokevirtualEPC extends HandleInvokeEPC {
		private int cpi;
		public HandleInvokevirtualEPC(int cpi) {
			this.cpi = cpi;
		}
		
		public int getConstantPoolIndex() {
			return cpi;
		}
		
		@Override
		public String toModelString(int indentLevel) {
			return "HandleInvokevirtualEPC(" + cpi + ")";
		}

		@Override
		public CircusAction[] doEFSDataRefinement(int stackDepth) {
			throw new UnsupportedOperationException("Can't perform frame stack elimination on HandleInvokevirtualEPC");
		}

		@Override
		public int stackDepthChange() {
			return 0;
		}

		@Override
		public CircusAction[] expandWithClassInfo(ClassModel classInfo) {
			throw new UnsupportedOperationException("Can't perform frame stack elimination on HandleInvokevirtualEPC");
		}
		
		@Override
		public String toCCode(int indentLevel) {
			throw new UnsupportedOperationException("This should be eliminated before C code is produced");
		}
	}
	
	public static final class HandleInvokespecialEPC extends HandleInvokeEPC {
		private int cpi;
		public HandleInvokespecialEPC(int cpi) {
			this.cpi = cpi;
		}
		
		public int getConstantPoolIndex() {
			return cpi;
		}
		
		@Override
		public String toModelString(int indentLevel) {
			return "HandleInvokespecialEPC(" + cpi + ")";
		}

		@Override
		public CircusAction[] doEFSDataRefinement(int stackDepth) {
			throw new UnsupportedOperationException("Can't perform frame stack elimination on HandleInvokespecialEPC");
		}

		@Override
		public int stackDepthChange() {
			return 0;
		}

		@Override
		public CircusAction[] expandWithClassInfo(ClassModel classInfo) {
			throw new UnsupportedOperationException("Can't perform frame stack elimination on HandleInvokespecialEPC");
		}
		
		@Override
		public String toCCode(int indentLevel) {
			throw new UnsupportedOperationException("This should be eliminated before C code is produced");
		}
	}
	
	public static final class HandleInvokestaticEPC extends HandleInvokeEPC {
		private int cpi;
		public HandleInvokestaticEPC(int cpi) {
			this.cpi = cpi;
		}
		
		public int getConstantPoolIndex() {
			return cpi;
		}
		
		@Override
		public String toModelString(int indentLevel) {
			return "HandleInvokestaticEPC(" + cpi + ")";
		}

		@Override
		public CircusAction[] doEFSDataRefinement(int stackDepth) {
			throw new UnsupportedOperationException("Can't perform frame stack elimination on HandleInvokestaticEPC");
		}

		@Override
		public int stackDepthChange() {
			return 0;
		}

		@Override
		public CircusAction[] expandWithClassInfo(ClassModel classInfo) {
			throw new UnsupportedOperationException("Can't perform frame stack elimination on HandleInvokestaticEPC");
		}
		
		@Override
		public String toCCode(int indentLevel) {
			throw new UnsupportedOperationException("This should be eliminated before C code is produced");
		}
	}
	
	public static final class InterpreterNewStackFrame extends CircusAction {
		private ClassID classInfo;
		private MethodID methodID;
		public InterpreterNewStackFrame(ClassID resolvedClassName, MethodID methodName) {
			this.classInfo = resolvedClassName;
			this.methodID = methodName;
		}
		
		public ClassID getClassName() {
			return classInfo;
		}
		
		public MethodID getMethodID() {
			return methodID;
		}
		
		@Override
		public String toModelString(int indentLevel) {
			if (classInfo == null && methodID == null) {
				return "\\lschexpract InterpreterNewStackFrame \\rschexpract";
			} else if (classInfo != null && methodID != null) {
				return "\\lschexpract InterpreterNewStackFrame["
						+ classInfo + "/class?, "
						+ methodID + "/methodID?] \\rschexpract";
			} else if (classInfo != null) {
				return "\\lschexpract InterpreterNewStackFrame["
						+ classInfo + "/class?] \\rschexpract";
			} else {
				return "\\lschexpract InterpreterNewStackFrame["
						+ methodID + "/methodID?] \\rschexpract";
			}
		}

		@Override
		public CircusAction[] doEFSDataRefinement(int stackDepth) {
			throw new UnsupportedOperationException("InterpreterNewStackFrame must refined with class information before applying EFS data refinement");
		}

		@Override
		public int stackDepthChange() {
			return 0;
		}

		@Override
		public CircusAction[] expandWithClassInfo(ClassModel classInfo) {
			throw new UnsupportedOperationException("InterpreterNewStackFrame should be elimated by this point");
		}
		
		@Override
		public String toCCode(int indentLevel) {
			throw new UnsupportedOperationException("This should be eliminated before C code is produced");
		}
	}

	public static final class VarBlock extends CircusAction {
		private String varName;
		private CircusAction[] actions;
		public VarBlock(String varName, CircusAction[] actions) {
			if (varName == null || actions == null) {
				throw new IllegalArgumentException("varName and actions must be non-null");
			}
			this.varName = varName;
			this.actions = actions;
		}
		
		public String getVarName() {
			return varName;
		}
		
		public CircusAction[] getActions() {
			return actions;
		}
		
		@Override
		public String toModelString(int indentLevel) {
			StringBuilder builder = new StringBuilder();
			builder = builder.append("(\\circvar " + varName + " : Word \\circspot \\\\\n");
			
			for (int i = 0; i < indentLevel; i++) {
				builder.append("\t");
			}
			builder.append("\t");
			builder.append("\\t" + (indentLevel+1) + " ");
			
			for (int i = 0; i < actions.length; i++) {
				builder.append(actions[i].toModelString(indentLevel+1));
				if (i < actions.length-1) {
					builder.append(" \\circseq \\\\\n\t\t\\t" + (indentLevel+1) + " ");
				} else {
					builder.append(")");
				}
			}
			
			return builder.toString();
		}

		@Override
		public CircusAction[] doEFSDataRefinement(int stackDepth) {
			Vector<CircusAction> newActions = new Vector<CircusAction>();
			int currentStackDepth = stackDepth;
			
			for (CircusAction action : actions) {
				newActions.addAll(Arrays.asList(action.doEFSDataRefinement(currentStackDepth)));
				currentStackDepth += action.stackDepthChange();
			}
			
			return new CircusAction[] {new CircusAction.VarBlock(varName, newActions.toArray(new CircusAction[] {}))};
		}

		@Override
		public int stackDepthChange() {
			int totalStackDepthChange = 0;
			for (CircusAction action : actions) {
				totalStackDepthChange += action.stackDepthChange();
			}
			return totalStackDepthChange;
		}

		@Override
		public CircusAction[] expandWithClassInfo(ClassModel classInfo) {
			Vector<CircusAction> newActions = new Vector<CircusAction>();
			
			for (CircusAction action : actions) {
				newActions.addAll(Arrays.asList(action.expandWithClassInfo(classInfo)));
			}
			
			return new CircusAction[] {new CircusAction.VarBlock(varName, newActions.toArray(new CircusAction[] {}))};
		}
		
		@Override
		public String toCCode(int indentLevel) {
			throw new UnsupportedOperationException("This should be eliminated before C code is produced");
//			StringBuilder builder = new StringBuilder();
//			builder = builder.append("{int32_t" + varName + ";\n");
//			
//			for (int i = 0; i < indentLevel+1; i++) {
//				builder.append("\t");
//			}
//			
//			builder.append(CircusAction.actionsToCCode(actions, indentLevel+1));
//			
//			builder.append("\n");
//			for (int i = 0; i < indentLevel; i++) {
//				builder.append("\t");
//			}
//			builder.append("}");
//			
//			return builder.toString();
		}
	}
	
	public static final class StackFrameVarBlock extends CircusAction {
		private CircusAction[] actions;
		private int numVars;
		private int stackSize;
		
		public StackFrameVarBlock(CircusAction[] actions, int numVars, int stackSize) {
			this.actions = actions;
			this.numVars = numVars;
			this.stackSize = stackSize;
		}
		
		public CircusAction[] getActions() {
			return actions;
		}
		
		public int getNumVars() {
			return numVars;
		}
		
		public int getStackSize() {
			return stackSize;
		}
		
		@Override
		public String toModelString(int indentLevel) {
			StringBuilder builder = new StringBuilder();
			builder.append("\\circvar stackFrame : StackFrameEPC \\circspot \\\\\n");
	
			builder.append("\t\t\\\t" + (indentLevel+1) + " ");
			
			for (int i = 0; i < actions.length; i++) {
				builder.append(actions[i].toModelString(indentLevel+1));
				if (i < actions.length-1) {
					builder.append(" \\circseq \\\\\n\t\t\\t" + (indentLevel+1) + " ");
				} else {
					builder.append(")");
				}
			}
			
			return builder.toString();
		}

		@Override
		public CircusAction[] expandWithClassInfo(ClassModel classInfo) {
			Vector<CircusAction> newActions = new Vector<CircusAction>();
			
			for (CircusAction action : actions) {
				newActions.addAll(Arrays.asList(action.expandWithClassInfo(classInfo)));
			}
			
			return new CircusAction[] {new CircusAction.StackFrameVarBlock(newActions.toArray(new CircusAction[] {}), numVars, stackSize)};
		}

		@Override
		public CircusAction[] doEFSDataRefinement(int stackDepth) {
			Vector<CircusAction> newActions = new Vector<CircusAction>();
			
			int currentStackDepth = stackDepth;
			for (CircusAction action : actions) {
				newActions.addAll(Arrays.asList(action.doEFSDataRefinement(currentStackDepth)));
				currentStackDepth += action.stackDepthChange();
			}
			
			return new CircusAction[] {
					new CircusAction.MethodVarBlock(newActions.toArray(new CircusAction[] {}), numVars, stackSize, 1)
			};
		}

		@Override
		public int stackDepthChange() {
			int totalStackDepthChange = 0;
			for (CircusAction action : actions) {
				totalStackDepthChange += action.stackDepthChange();
			}
			return totalStackDepthChange;
		}
		
		@Override
		public String toCCode(int indentLevel) {
			throw new UnsupportedOperationException("This should be eliminated before C code is produced");
		}
	}
	
	public static final class MethodVarBlock extends CircusAction {
		private CircusAction[] actions;
		private int numVars;
		private int stackSize;
		private int initVar;
		
		public MethodVarBlock(CircusAction[] actions, int numVars, int stackSize, int initVar) {
			this.actions = actions;
			this.numVars = numVars;
			this.stackSize = stackSize;
			this.initVar = initVar;
		}

		public CircusAction[] getActions() {
			return actions;
		}
		
		public int getNumVars() {
			return numVars;
		}
		
		public int getStackSize() {
			return stackSize;
		}
		
		@Override
		public String toModelString(int indentLevel) {
			StringBuilder builder = new StringBuilder();
			
			if (numVars > 0 && initVar <= numVars) {
				builder.append("\\circvar ");
				for (int i = initVar; i <= numVars; i++) {
					builder.append("var" + i);
					if (i < numVars) {
						builder.append(", ");
					}
				}
				builder.append(" : Word \\circspot \\\\\n");
				builder.append("\t\t\\t" + indentLevel + " ");
			}
			
			if (stackSize > 0) {
				builder.append("\\circvar ");
				for (int i = 1; i <= stackSize; i++) {
					builder.append("stack" + i);
					if (i < stackSize) {
						builder.append(", ");
					}
				}
				builder.append(" : Word \\circspot \\\\\n");
				builder.append("\t\t\\t" + (indentLevel) + " ");
			}
			builder.append(CircusAction.actionsToString(actions, indentLevel));
			
//			for (int i = 0; i < actions.length; i++) {
//				builder.append(actions[i].toModelString(indentLevel+1));
//				if (i < actions.length-1) {
//					builder.append(" \\circseq \\\\\n\t\t\\t" + (indentLevel+1) + " ");
//				} else {
//					builder.append(")");
//				}
//			}
			
			return builder.toString();
		}

		@Override
		public CircusAction[] expandWithClassInfo(ClassModel classInfo) {
			Vector<CircusAction> newActions = new Vector<CircusAction>();
			
			for (CircusAction action : actions) {
				newActions.addAll(Arrays.asList(action.expandWithClassInfo(classInfo)));
			}
			
			return new CircusAction[] {new CircusAction.MethodVarBlock(newActions.toArray(new CircusAction[] {}), numVars, stackSize, initVar)};
		}

		@Override
		public CircusAction[] doEFSDataRefinement(int stackDepth) {
			Vector<CircusAction> newActions = new Vector<CircusAction>();
			
			int currentStackDepth = stackDepth;
			for (CircusAction action : actions) {
				newActions.addAll(Arrays.asList(action.doEFSDataRefinement(currentStackDepth)));
				currentStackDepth += action.stackDepthChange();
			}
			
			return new CircusAction[] {
					new CircusAction.MethodVarBlock(newActions.toArray(new CircusAction[] {}), numVars, stackSize, initVar)
			};
		}

		@Override
		public int stackDepthChange() {
			int totalStackDepthChange = 0;
			for (CircusAction action : actions) {
				totalStackDepthChange += action.stackDepthChange();
			}
			return totalStackDepthChange;
		}
		
		@Override
		public String toCCode(int indentLevel) {
			StringBuilder builder = new StringBuilder();
			
			if (numVars > 0 && initVar <= numVars) {
				builder.append("int32_t ");
				for (int i = initVar; i <= numVars; i++) {
					builder.append("var" + i);
					if (i < numVars) {
						builder.append(", ");
					}
				}
				builder.append(";\n");
				for (int i = 0; i < indentLevel; i++) {
					builder.append("\t");
				}
			}
			
			if (stackSize > 0) {
				builder.append("int32_t ");
				for (int i = 1; i <= stackSize; i++) {
					builder.append("stack" + i);
					if (i < stackSize) {
						builder.append(", ");
					}
				}
				builder.append(";\n");
				for (int i = 0; i < indentLevel; i++) {
					builder.append("\t");
				}
			}
			builder.append(CircusAction.actionsToCCode(actions, indentLevel));
			
//			for (int i = 0; i < actions.length; i++) {
//				builder.append(actions[i].toModelString(indentLevel+1));
//				if (i < actions.length-1) {
//					builder.append(" \\circseq \\\\\n\t\t\\t" + (indentLevel+1) + " ");
//				} else {
//					builder.append(")");
//				}
//			}
			
			return builder.toString();
		}
		
	}
	
	public static final class ChannelComm extends CircusAction {
		private String channelName;
		private String[] channelArgs;
		private boolean[] argIsInput;
		private CircusAction nextAction;
		public ChannelComm(
				String channelName, 
				String[] channelArgs, 
				boolean[] argIsInput, 
				CircusAction nextAction) {
			if (channelArgs != null && argIsInput != null) {
				if (channelArgs.length != argIsInput.length) {
					throw new IllegalArgumentException("channelArgs and argIsInput must have the same length");
				}
			}
			this.channelName = channelName;
			this.channelArgs = channelArgs;
			this.argIsInput = argIsInput;
			this.nextAction = nextAction;
		}
		
		public String getChannelName() {
			return channelName;
		}
		
		public String[] getChannelArgs() {
			return channelArgs;
		}
		
		public boolean[] getArgIsInputArray() {
			return argIsInput;
		}
		
		public CircusAction getAction() {
			return nextAction;
		}
		
		@Override
		public String toModelString(int indentLevel) {
			StringBuilder builder = new StringBuilder();
			builder.append(channelName);
			
			if (channelArgs != null && argIsInput != null) {
				for (int i = 0; i < channelArgs.length; i++) {
					if (argIsInput[i]) {
						builder.append("?" + channelArgs[i]);
					} else {
						builder.append("!" + channelArgs[i]);
					}
				}
			}
			
			builder.append(" \\then ");
			builder.append(nextAction.toModelString(indentLevel));
			
			return builder.toString();
		}

		@Override
		public CircusAction[] doEFSDataRefinement(int stackDepth) {
			// the only communication that directly mentions part of the state is releaseLock
			if (channelName.equals("releaseLock")) {
				return new CircusAction[] {
						new CircusAction.ChannelComm("releaseLock", new String[] {"var1"}, 
								new boolean[] {false}, nextAction.doEFSDataRefinement(stackDepth)[0])
				};
			} else {
				return new CircusAction[] {
						new CircusAction.ChannelComm(channelName, channelArgs, 
								argIsInput, nextAction.doEFSDataRefinement(stackDepth)[0])
						};
			}
		}

		@Override
		public int stackDepthChange() {
			return nextAction.stackDepthChange();
		}

		@Override
		public CircusAction[] expandWithClassInfo(ClassModel classInfo) {
			return new CircusAction[] {new CircusAction.ChannelComm(channelName, channelArgs, argIsInput, nextAction.expandWithClassInfo(classInfo)[0])};
		}

		@Override
		public String toCCode(int indentLevel) {
			StringBuilder builder = new StringBuilder();
			
			// the handling of this depends on the channel name
			if (channelName.equals("getClassIDOf")) {
				// there should be a conditional here as it is a choice over classID
				if (nextAction instanceof CircusAction.Conditional) {
					Conditional conditional = (CircusAction.Conditional) nextAction;
					
					for (int branch = 0; branch < conditional.conditions.length; branch++) {
						if (branch > 0) {
							for (int i = 0; i < indentLevel; i++) {
								builder.append("\t");
							}
							builder.append("} else ");
						}
						
						builder.append("if (" + conditional.conditions[branch].replace("=", "==").replace("cid", "((java_lang_Object*)  ((uintptr_t)" + channelArgs[0]+"))->classID") + ") {\n");
						for (int i = 0; i < indentLevel+1; i++) {
							builder.append("\t");
						}
						builder.append(CircusAction.actionsToCCode(conditional.actions[branch], indentLevel+1));
						builder.append("\n");
					}
					for (int i = 0; i < indentLevel; i++) {
						builder.append("\t");
					}
					builder.append("}");
				}
			} else if (channelName.equals("putField")) {
				builder.append("((" + channelArgs[1].replaceAll("ID\\z", "") + " *) ((uintptr_t)" + channelArgs[0] + "))->" + channelArgs[2] + " = " + channelArgs[3] + ";");
			} else if (channelName.equals("getField")) {
				CircusAction.ChannelComm innerComm = (CircusAction.ChannelComm) nextAction;
				CircusAction.Assignment innerAssign = (CircusAction.Assignment) innerComm.nextAction;

				builder.append(innerAssign.getVar() + " = " + "((" + channelArgs[1].replaceAll("ID\\z", "") + " *)  ((uintptr_t)" + channelArgs[0] + "))->" + channelArgs[2] + ";");
			} else if (channelName.equals("putStatic")) {
				builder.append("staticClassFields->" + channelArgs[0].replaceAll("ID\\z", "") + "_" + channelArgs[1] + " = " + channelArgs[2] + ";");
			} else if (channelName.equals("getStatic")) {
				CircusAction.ChannelComm innerComm = (CircusAction.ChannelComm) nextAction;
				CircusAction.Assignment innerAssign = (CircusAction.Assignment) innerComm.nextAction;

				builder.append(innerAssign.getVar() + " = " + "staticClassFields->" + channelArgs[0].replaceAll("ID\\z", "") + "_" + channelArgs[1] + ";");
			} else if (channelName.equals("newObject")) {
				CircusAction.ChannelComm innerComm = (CircusAction.ChannelComm) nextAction;
				CircusAction.Assignment innerAssign = (CircusAction.Assignment) innerComm.nextAction;
				builder.append(innerAssign.getVar() + " = newObject(" + channelArgs[1] + ");");
			} else if (channelName.equals("input")) {
				CircusAction.Assignment innerAssign = (CircusAction.Assignment) nextAction;
				builder.append(innerAssign.getVar() + " = input();");
			} else if (channelName.equals("output")) {
				builder.append("output(" + channelArgs[0] + ");");	
			} else if (channelName.equals("setPriorityCeiling")) {
				builder.append("setPriorityCeiling(" + channelArgs[0] + ", " + channelArgs[1] + ");");	
			} else if (channelName.equals("takeLock")) {
				builder.append("takeLock(" + channelArgs[0] + ");");	
			} else if (channelName.equals("releaseLock")) {
				builder.append("releaseLock(" + channelArgs[0] + ");");	
			} else if (channelName.equals("register")) {
				// register is a keyword in C, so needs changing
				builder.append("registerSchedulable(" + channelArgs[1] + ");");	
			} else {
				// other things are just native methods that return nothing and start with a thread identifier
				builder.append(channelName + "(");
				for (int i = 1; i < channelArgs.length; i++) {
					builder.append(channelArgs[i]);
					if (i < channelArgs.length-1) {
						builder.append(", ");
					}
				}
				builder.append(");");
			}
			
			return builder.toString();
		}
	}
	
	public static final class Skip extends CircusAction {
		public Skip() {
			
		}

		@Override
		public String toModelString(int indentLevel) {
			return "\\Skip";
		}

		@Override
		public CircusAction[] doEFSDataRefinement(int stackDepth) {
			return new CircusAction[] {this};
		}

		@Override
		public int stackDepthChange() {
			return 0;
		}

		@Override
		public CircusAction[] expandWithClassInfo(ClassModel classInfo) {
			return new CircusAction[] {this};
		}

		@Override
		public String toCCode(int indentLevel) {
			return "";
		}
	}
	
	public static final class Chaos extends CircusAction {
		public Chaos() {
			
		}

		@Override
		public String toModelString(int indentLevel) {
			return "\\Chaos";
		}

		@Override
		public CircusAction[] doEFSDataRefinement(int stackDepth) {
			return new CircusAction[] {this};
		}

		@Override
		public int stackDepthChange() {
			return 0;
		}

		@Override
		public CircusAction[] expandWithClassInfo(ClassModel classInfo) {
			return new CircusAction[] {this};
		}
		
		@Override
		public String toCCode(int indentLevel) {
			throw new UnsupportedOperationException("This should be eliminated before C code is produced");
		}
	}
	
	public static final class Poll extends CircusAction {
		public Poll() {
			
		}

		@Override
		public String toModelString(int indentLevel) {
			return "Poll";
		}

		@Override
		public CircusAction[] doEFSDataRefinement(int stackDepth) {
			return new CircusAction[] {this};
		}

		@Override
		public int stackDepthChange() {
			return 0;
		}

		@Override
		public CircusAction[] expandWithClassInfo(ClassModel classInfo) {
			return new CircusAction[] {this};
		}

		@Override
		public String toCCode(int indentLevel) {
			// Polls are removed in the generated C code
			return "";
		}
	}
	
	public static final class Conditional extends CircusAction {
		private String[] conditions;
		private CircusAction[][] actions;
		public Conditional(String[] conditions, CircusAction[][] actions) {
			if (conditions != null && actions != null) {
				if (conditions.length != actions.length) {
					throw new IllegalArgumentException("conditions and actions must have the same length");
				}
			}
			this.conditions = conditions;
			this.actions = actions;
		}
		
		public String[] getConditions() {
			return conditions;
		}
		
		public CircusAction[][] getConditionActions() {
			return actions;
		}
		
		@Override
		public String toModelString(int indentLevel) {
			StringBuilder builder = new StringBuilder();
			
			if(conditions != null && actions != null && conditions.length >= 1) {
				builder.append("\\circif " + conditions[0] + " \\circthen {} \\\\\n");
				
				for (int i = 0; i < actions[0].length; i++) {
					builder.append("\t\t\\t" + (indentLevel+1) + " ");
					builder.append(actions[0][i].toModelString(indentLevel+1));
					if (i < actions[0].length-1) {
						builder.append(" \\circseq \\\\\n");
					}
				}
				builder.append(" \\\\\n");
				
				for (int i = 1; i < conditions.length; i++) {
					builder.append("\t\t\\t" + indentLevel + " ");
					builder.append("{} \\circelse " + conditions[i] + " \\circthen {} \\\\\n");
					
					for (int j = 0; j < actions[i].length; j++) {
						builder.append("\t\t\\t" + (indentLevel+1) + " ");
						builder.append(actions[i][j].toModelString(indentLevel+1));
						if (j < actions[i].length-1) {
							builder.append(" \\circseq ");
						}
						builder.append(" \\\\\n");
					}
				}
				
				builder.append("\t\t\\t" + indentLevel + " \\circfi");
			} else {
				builder.append("\\Chaos");
			}
			
			
			return builder.toString();
		}

		@Override
		public CircusAction[] doEFSDataRefinement(int stackDepth) {
			CircusAction[][] newActions = new CircusAction[actions.length][];
			
			for (int i = 0; i < actions.length; i++) {
				CircusAction[] branchActions = actions[i];
				Vector<CircusAction> newBranchActions = new Vector<CircusAction>();
				int currentStackDepth = stackDepth;
				for (CircusAction action : branchActions) {
					newBranchActions.addAll(Arrays.asList(action.doEFSDataRefinement(currentStackDepth)));
					currentStackDepth += action.stackDepthChange();
				}
				newActions[i] = newBranchActions.toArray(new CircusAction[] {});
			}
			
			return new CircusAction[] {new CircusAction.Conditional(conditions, newActions)};
		}

		@Override
		public int stackDepthChange() {
			if (actions != null && actions.length != 0) {
				int totalStackDepthChange = 0;
				for (CircusAction action : actions[0]) {
					totalStackDepthChange += action.stackDepthChange();
				}
				return totalStackDepthChange;
			} else {
				return 0;
			}
		}

		@Override
		public CircusAction[] expandWithClassInfo(ClassModel classInfo) {
			CircusAction[][] newActions = new CircusAction[actions.length][];
			
			for (int i = 0; i < actions.length; i++) {
				CircusAction[] branchActions = actions[i];
				Vector<CircusAction> newBranchActions = new Vector<CircusAction>();
				for (CircusAction action : branchActions) {
					newBranchActions.addAll(Arrays.asList(action.expandWithClassInfo(classInfo)));
				}
				newActions[i] = newBranchActions.toArray(new CircusAction[] {});
			}
			
			return new CircusAction[] {new CircusAction.Conditional(conditions, newActions)};
		}

		@Override
		public String toCCode(int indentLevel) {
			StringBuilder builder = new StringBuilder();
			
			if (actions[0][0] instanceof CircusAction.Skip) {
				// we only need to have one branch, and it is the second
				builder.append("if (" + conditions[1].replace("=", "==").replace("\\neq", "!=").replace("\\leq", "<=").replace("\\geq", ">=").replaceAll("\\\\lnot (.*)", "!\\($1\\)") + ") {\n");
				for (int i = 0; i < indentLevel+1; i++) {
					builder.append("\t");
				}
				builder.append(CircusAction.actionsToCCode(actions[1], indentLevel+1));
				builder.append("\n");
				for (int i = 0; i < indentLevel; i++) {
					builder.append("\t");
				}
				builder.append("}");
				
			} else {
				builder.append("if (" + conditions[0].replace("=", "==").replace("\\neq", "!=").replace("\\leq", "<=").replace("\\geq", ">=").replaceAll("\\\\lnot (\\S)", "!\\($1\\)") + ") {\n");
				for (int i = 0; i < indentLevel+1; i++) {
					builder.append("\t");
				}
				builder.append(CircusAction.actionsToCCode(actions[0], indentLevel+1));
				builder.append("\n");
				for (int i = 0; i < indentLevel; i++) {
					builder.append("\t");
				}
				builder.append("} else {\n");
				for (int i = 0; i < indentLevel+1; i++) {
					builder.append("\t");
				}
				builder.append(CircusAction.actionsToCCode(actions[1], indentLevel+1));
				builder.append("\n");
				for (int i = 0; i < indentLevel; i++) {
					builder.append("\t");
				}
				builder.append("}");
			}
			
			return builder.toString();
		}
	}
	
	public static final class Recursion extends CircusAction {
		private String recVar;
		private CircusAction[] actions;
		public Recursion(String recVar, CircusAction[] actions) {
			this.recVar = recVar;
			this.actions = actions;
		}
		
		public String getRecVarName() {
			return recVar;
		}
		
		public CircusAction[] getActions() {
			return actions;
		}

		@Override
		public String toModelString(int indentLevel) {
			StringBuilder builder = new StringBuilder();
			builder.append("\\circmu " + recVar + " \\circspot \\\\\n");
			
//			for (int i = 0; i < indentLevel; i++) {
//				builder.append("\t");
//			}
			
			for (int i = 0; i < actions.length; i++) {
				builder.append("\t\t\\t" + (indentLevel+1) + " ");
				builder.append(actions[i].toModelString(indentLevel+1));
				if (i < actions.length-1) {
					builder.append(" \\circseq \\\\\n");
				}
			}
			
			
			return builder.toString();
		}

		@Override
		public CircusAction[] doEFSDataRefinement(int stackDepth) {
			ArrayList<CircusAction> newActions = new ArrayList<CircusAction>();
			int currentStackDepth = stackDepth;
			
			for (CircusAction action : actions) {
				newActions.addAll(Arrays.asList(action.doEFSDataRefinement(currentStackDepth)));
				currentStackDepth += action.stackDepthChange();
			}
			
			return new CircusAction[] {new CircusAction.Recursion(recVar, newActions.toArray(new CircusAction[] {}))};
		}

		@Override
		public int stackDepthChange() {
			int totalStackDepthChange = 0;
			for (CircusAction action : actions) {
				totalStackDepthChange += action.stackDepthChange();
			}
			return totalStackDepthChange;
		}

		@Override
		public CircusAction[] expandWithClassInfo(ClassModel classInfo) {
			ArrayList<CircusAction> newActions = new ArrayList<CircusAction>();
			
			for (CircusAction action : actions) {
				newActions.addAll(Arrays.asList(action.expandWithClassInfo(classInfo)));
			}
			
			return new CircusAction[] {new CircusAction.Recursion(recVar, newActions.toArray(new CircusAction[] {}))};
		}

		@Override
		public String toCCode(int indentLevel) {
			StringBuilder builder = new StringBuilder();
			
			// first, determine if we are dealing with an infinite loop or a conditional loop
			CircusAction lastAction = actions[actions.length-1];
			if (lastAction instanceof CircusAction.ActionRef){
				// ends with an (unconditional) recursion, so must be infinite
				
				CircusAction[] bodyActions = new CircusAction[actions.length-1];
				for (int i = 0; i < bodyActions.length; i++) {
					bodyActions[i] = actions[i];
				}
				
				builder.append("while(1) {\n");
				for (int i = 0; i < indentLevel+1; i++) {
					builder.append("\t");
				}
				builder.append(CircusAction.actionsToCCode(bodyActions, indentLevel+1));
				builder.append("\n");
				for (int i = 0; i < indentLevel; i++) {
					builder.append("\t");
				}
				builder.append("}");
			} else if (lastAction instanceof CircusAction.Conditional) {
				CircusAction.Conditional conditional = (CircusAction.Conditional) lastAction;
				// next, we need to determine if we are dealing with a while or do-while
				if (actions.length == 1) {
					// we have a simple while loop with no actions before the loop
					// check which branch is Skip
					if (conditional.actions[0][0] instanceof CircusAction.Skip) {
						// loop body is in second branch
						builder.append("while (" + conditional.conditions[1].replace("=", "==").replace("\\neq", "!=").replace("\\leq", "<=").replace("\\geq", ">=").replaceAll("\\\\lnot (\\S)", "!\\($1\\)") + ") {\n");
						for (int i = 0; i < indentLevel+1; i++) {
							builder.append("\t");
						}
						
						CircusAction[] bodyActions = new CircusAction[conditional.actions[1].length-1];
						for (int i = 0; i < bodyActions.length; i++) {
							bodyActions[i] = conditional.actions[1][i];
						}
						builder.append(CircusAction.actionsToCCode(bodyActions, indentLevel+1));
						builder.append("\n");
						for (int i = 0; i < indentLevel; i++) {
							builder.append("\t");
						}
						builder.append("}");
					} else {
						// loop body is in first branch
						builder.append("while (" + conditional.conditions[0].replace("=", "==").replace("\\neq", "!=").replace("\\leq", "<=").replace("\\geq", ">=").replaceAll("\\\\lnot (\\S)", "!\\($1\\)") + ") {\n");
						for (int i = 0; i < indentLevel+1; i++) {
							builder.append("\t");
						}
						
						CircusAction[] bodyActions = new CircusAction[conditional.actions[0].length-1];
						for (int i = 0; i < bodyActions.length; i++) {
							bodyActions[i] = conditional.actions[0][i];
						}
						builder.append(CircusAction.actionsToCCode(bodyActions, indentLevel+1));
						builder.append("\n");
						for (int i = 0; i < indentLevel; i++) {
							builder.append("\t");
						}
						builder.append("}");
					}
				} else {
					// do-while loop: everything before the conditional is the loop body, first branch is condition 
					builder.append("while (" + conditional.conditions[0].replace("=", "==").replace("\\neq", "!=").replace("\\leq", "<=").replace("\\geq", ">=").replaceAll("\\\\lnot (\\S)", "!\\($1\\)") + ") {\n");
					for (int i = 0; i < indentLevel+1; i++) {
						builder.append("\t");
					}
					
					CircusAction[] bodyActions = new CircusAction[actions.length-1];
					for (int i = 0; i < bodyActions.length; i++) {
						bodyActions[i] = actions[i];
					}
					builder.append(CircusAction.actionsToCCode(bodyActions, indentLevel+1));
					builder.append("\n");
					for (int i = 0; i < indentLevel; i++) {
						builder.append("\t");
					}
					builder.append("}");
				}
			}
			
			return builder.toString();
		}
	}
	
	public static final class ActionRef extends CircusAction {
		private String actionName;
		public ActionRef(String actionName) {
			this.actionName = actionName;
		}
		
		public String getActionName() {
			return this.toModelString(0);
		}

		@Override
		public String toModelString(int indentLevel) {
			if (actionName != null) {
				return actionName;
			} else {
				return "Skip";
			}
		}

		@Override
		public CircusAction[] doEFSDataRefinement(int stackDepth) {
			return new CircusAction[] {this};
		}

		@Override
		public int stackDepthChange() {
			return 0;
		}

		@Override
		public CircusAction[] expandWithClassInfo(ClassModel classInfo) {
			return new CircusAction[] {this};
		}
		
		@Override
		public String toCCode(int indentLevel) {
			throw new UnsupportedOperationException("This should be eliminated before C code is produced");
		}
	}
	
	public static final class MethodActionRef extends CircusAction {
		private FullMethodID methodName;
		private String[] args;
		private boolean[] isReturn;
		public MethodActionRef(FullMethodID methodName, String[] args) {
			this.methodName = methodName;
			this.args = args;
			if (args != null) {
				this.isReturn = new boolean[args.length];
				int i = 0;
				for (String arg : args) {
					this.isReturn[i] = arg.equals("retVal") || arg.equals("retVal\\_msb") || arg.equals("retVal\\_lsb"); 
					i++;
				}
			} else {
				this.isReturn = null;
			}
		}
		
		public MethodActionRef(FullMethodID methodName, String[] args, boolean[] isReturn) {
			this.methodName = methodName;
			this.args = args;
			this.isReturn = isReturn;
		}

		public FullMethodID getMethodName() {
			return methodName;
		}
		
		public String[] getMethodArgs() {
			return args;
		}
		
		public boolean[] getIsReturnArray() {
			return isReturn;
		}
		
		@Override
		public String toModelString(int indentLevel) {
			StringBuilder buffer = new StringBuilder();
			
			buffer.append(methodName.toModelString());
			if (args != null && args.length != 0) {
				buffer.append("(");
				for (int i = 0; i < args.length; i++) {
					buffer.append(args[i]);
					if (i < args.length-1) {
						buffer.append(", ");
					}
				}
				buffer.append(")");
			}
			
			return buffer.toString();
		}

		@Override
		public CircusAction[] doEFSDataRefinement(int stackDepth) {
			if (args != null && args.length > 0 && args[args.length-1].equals("retVal")) {
				String[] newArgs = new String[args.length];
				for (int i = 0; i < args.length-1; i++) {
					newArgs[i] = args[i];
				}
				newArgs[args.length-1] = "stack" + (stackDepth+1);
				return new CircusAction[] {new CircusAction.MethodActionRef(methodName, newArgs, isReturn)};
			} else if (args != null && args.length > 1 && args[args.length-1].equals("retVal\\_lsb") && args[args.length-2].equals("retVal\\_msb")) {
				String[] newArgs = new String[args.length];
				for (int i = 0; i < args.length-1; i++) {
					newArgs[i] = args[i];
				}
				newArgs[args.length-2] = "stack" + (stackDepth+1);
				newArgs[args.length-1] = "stack" + (stackDepth+2);
				return new CircusAction[] {new CircusAction.MethodActionRef(methodName, newArgs, isReturn)};
			} else {
				return new CircusAction[] {this};
			}
		}

		@Override
		public int stackDepthChange() {
			if (args != null && args.length > 0 && args[args.length-1].equals("retVal")) {
				return 1;
			} else if (args != null && args.length > 0 && args[args.length-1].equals("retVal\\_lsb") && args[args.length-2].equals("retVal\\_msb")) {
				return 2;
			} else {
				return 0;
			}
		}

		@Override
		public CircusAction[] expandWithClassInfo(ClassModel classInfo) {
			return new CircusAction[] {this};
		}

		@Override
		public String toCCode(int indentLevel) {
			StringBuilder builder = new StringBuilder();
			
			int numArgs = args.length;
			
//			if (methodName.methodID.hasReturnValue()) {
//				builder.append(args[args.length-1] + " = ");
//				numArgs--;
//			}
			
			builder.append(methodName);
			builder.append("(");
			for (int i = 0; i < numArgs; i++) {
				if (isReturn[i]) {
					builder.append("&");
				}
				builder.append(args[i]);
				if (i < numArgs-1) {
					builder.append(", ");
				}
			}
			builder.append(");");
			
			return builder.toString();
		}
		
	}
	
	public static final class InitSF extends CircusAction {

		private ClassID classID;
		private MethodID methodID;
		private int numArgs;
		
		public InitSF(FullMethodID method, int numArgs) {
			this.classID = method.classID;
			this.methodID = method.methodID;
			this.numArgs = numArgs;
		}
		
		@Override
		public String toModelString(int indentLevel) {
			return "Invoke" + classID + "\\_" + methodID + "SF";
		}

		@Override
		public CircusAction[] doEFSDataRefinement(int stackDepth) {
			CircusAction[] actions = new CircusAction[numArgs];
			
			for (int i = 0; i < numArgs; i++) {
				actions[i] = new CircusAction.Assignment("var" + (i+1), "arg" + (i+1));
			}
			
			return actions;
		}

		@Override
		public int stackDepthChange() {
			return 0;
		}

		@Override
		public CircusAction[] expandWithClassInfo(ClassModel classInfo) {
			return new CircusAction[] {this};
		}
		
		@Override
		public String toCCode(int indentLevel) {
			throw new UnsupportedOperationException("This should be eliminated before C code is produced");
		}
		
	}
	
	public static final class ParametrisedBlock extends CircusAction {
		private String[] parameterNames;
		private boolean[] parameterIsRet;
		private CircusAction[] actions;
		
		public ParametrisedBlock(String[] parameterNames, boolean[] parameterIsRet, CircusAction[] actions) {
			if (parameterNames != null && parameterIsRet != null) {
				if (parameterNames.length != parameterIsRet.length) {
					throw new IllegalArgumentException("parameterNames and parameterIsRet must have the same length");
				}
			}
			this.parameterNames = parameterNames;
			this.parameterIsRet = parameterIsRet;
			this.actions = actions;
		}
		
		public String[] getParameterNames() {
			return this.parameterNames;
		}
		
		public boolean[] getParameterIsRet() {
			return parameterIsRet;
		}
		
		public CircusAction[] getActions() {
			return actions;
		}

		@Override
		public String toModelString(int indentLevel) {
			StringBuilder buffer = new StringBuilder();
			
			if (parameterNames != null && parameterNames.length != 0) {
				for (int i = 0; i < parameterNames.length; i++) {
					if (parameterIsRet[i]) {
						buffer.append("\\circres ");
					} else {
						buffer.append("\\circval ");
					}
					buffer.append(parameterNames[i]);
					if (i < parameterNames.length-1) {
						buffer.append(" : Word; ");
					} else {
						buffer.append(" : Word");
					}
				}
				buffer.append(" \\circspot \\\\\n");
				buffer.append("\t\t\\t" + indentLevel + " ");
			}
			buffer.append(CircusAction.actionsToString(actions, indentLevel));
			
			return buffer.toString();
		}
		
		@Override
		public CircusAction[] doEFSDataRefinement(int stackDepth) {
			ArrayList<CircusAction> newActions = new ArrayList<CircusAction>();
			int currentStackDepth = stackDepth;
			
			for (CircusAction action : actions) {
				newActions.addAll(Arrays.asList(action.doEFSDataRefinement(currentStackDepth)));
				currentStackDepth += action.stackDepthChange();
			}
			
			return new CircusAction[] {new ParametrisedBlock(parameterNames, parameterIsRet, newActions.toArray(new CircusAction[] {}))};
		}
		
		@Override
		public int stackDepthChange() {
			int totalStackDepthChange = 0;
			for (CircusAction action : actions) {
				totalStackDepthChange += action.stackDepthChange();
			}
			return totalStackDepthChange;
		}

		@Override
		public CircusAction[] expandWithClassInfo(ClassModel classInfo) {
			ArrayList<CircusAction> newActions = new ArrayList<CircusAction>();
			
			for (CircusAction action : actions) {
				newActions.addAll(Arrays.asList(action.expandWithClassInfo(classInfo)));
			}
			
			return new CircusAction[] {new CircusAction.ParametrisedBlock(parameterNames, parameterIsRet, newActions.toArray(new CircusAction[] {}))};
		}

		@Override
		public String toCCode(int indentLevel) {
			throw new UnsupportedOperationException("This needs to be performed separately");
		}
	}
}


