package main;

public abstract class BytecodeModel {
	
	private BytecodeModel() {} // prevent further extension of this class
	
	public abstract String toModelString();
	
	public static final class ACONST_NULL extends BytecodeModel {
		@Override
		public String toModelString() {
			return "aconst\\_null";
		}
	}
	public static final class DUP extends BytecodeModel {
		@Override
		public String toModelString() {
			return "dup";
		}
	}
	public static final class ARETURN extends BytecodeModel {
		@Override
		public String toModelString() {
			return "areturn";
		}
	}
	public static final class RETURN extends BytecodeModel {
		@Override
		public String toModelString() {
			return "return";
		}
	}
	public static final class IADD extends BytecodeModel {
		@Override
		public String toModelString() {
			return "iadd";
		}
	}
	public static final class INEG extends BytecodeModel {
		@Override
		public String toModelString() {
			return "ineg";
		}
	}
	public static final class NEW extends BytecodeModel {
		private int classIndex;
		
		public NEW(int classIndex) {
			this.classIndex	= classIndex;
		}
		
		@Override
		public String toModelString() {
			return "new~" + classIndex;
		}

		public int getClassIndex() {
			return classIndex;
		}
	}
	public static final class ICONST extends BytecodeModel {
		private int constantValue;
		
		public ICONST(int constantValue) {
			this.constantValue = constantValue;
		}
		
		@Override
		public String toModelString() {
			return "iconst~" + constantValue;
		}

		public int getConstantValue() {
			return constantValue;
		}
	}
	public static final class ALOAD extends BytecodeModel {
		private int variableIndex;
		
		public ALOAD(int variableIndex) {
			this.variableIndex = variableIndex;
		}
		
		@Override
		public String toModelString() {
			return "aload~" + variableIndex;
		}

		public int getVariableIndex() {
			return variableIndex;
		}
	}
	public static final class ASTORE extends BytecodeModel {
		private int variableIndex;
		
		public ASTORE(int variableIndex) {
			this.variableIndex = variableIndex;
		}
		
		@Override
		public String toModelString() {
			return "astore~" + variableIndex;
		}

		public int getVariableIndex() {
			return variableIndex;
		}
	}

	public static final class GETFIELD extends BytecodeModel {
		private int fieldIndex;
		
		public GETFIELD(int fieldIndex) {
			this.fieldIndex = fieldIndex;
		}
		
		@Override
		public String toModelString() {
			return "getfield~" + fieldIndex;
		}

		public int getFieldIndex() {
			return fieldIndex;
		}
	}
	public static final class PUTFIELD extends BytecodeModel {
		private int fieldIndex;
		
		public PUTFIELD(int fieldIndex) {
			this.fieldIndex = fieldIndex;
		}
		
		@Override
		public String toModelString() {
			return "putfield~" + fieldIndex;
		}

		public int getFieldIndex() {
			return fieldIndex;
		}
	}
	public static final class GETSTATIC extends BytecodeModel {
		private int fieldIndex;
		
		public GETSTATIC(int fieldIndex) {
			this.fieldIndex = fieldIndex;
		}
		
		@Override
		public String toModelString() {
			return "getstatic~" + fieldIndex;
		}

		public int getFieldIndex() {
			return fieldIndex;
		}
	}
	public static final class PUTSTATIC extends BytecodeModel {
		private int fieldIndex;
		
		public PUTSTATIC(int fieldIndex) {
			this.fieldIndex = fieldIndex;
		}
		
		@Override
		public String toModelString() {
			return "putstatic~" + fieldIndex;
		}

		public int getFieldIndex() {
			return fieldIndex;
		}
	}
	public static final class INVOKESPECIAL extends BytecodeModel {
		private int methodIndex;
		
		public INVOKESPECIAL(int methodIndex) {
			this.methodIndex = methodIndex;
		}
		
		@Override
		public String toModelString() {
			return "invokespecial~" + methodIndex;
		}

		public int getMethodIndex() {
			return methodIndex;
		}
	}
	public static final class INVOKESTATIC extends BytecodeModel {
		private int methodIndex;
		
		public INVOKESTATIC(int methodIndex) {
			this.methodIndex = methodIndex;
		}
		
		@Override
		public String toModelString() {
			return "invokestatic~" + methodIndex;
		}

		public int getMethodIndex() {
			return methodIndex;
		}
	}
	public static final class INVOKEVIRTUAL extends BytecodeModel {
		private int methodIndex;
		
		public INVOKEVIRTUAL(int methodIndex) {
			this.methodIndex = methodIndex;
		}
		
		@Override
		public String toModelString() {
			return "invokevirtual~" + methodIndex;
		}

		public int getMethodIndex() {
			return methodIndex;
		}
	}
	public static final class IF_ICMPLE extends BytecodeModel {
		private int offset;
		
		public IF_ICMPLE(int programAddress) {
			this.offset = programAddress;
		}
		
		public int getOffset() {
			return offset;
		}
		
		@Override
		public String toModelString() {
			return "if\\_icmple~" + offset;
		}
	}
	public static final class GOTO extends BytecodeModel {
		private int offset;
		
		public GOTO(int offset) {
			this.offset = offset;
		}
		
		public int getOffset() {
			return offset;
		}
		
		@Override
		public String toModelString() {
			return "goto~" + offset;
		}
	}
	
	public static abstract class NonSubsetInstruction extends BytecodeModel {
		
	}
	
	public static final class LLOAD extends NonSubsetInstruction {
		private int variableIndex;
		
		public LLOAD(int variableIndex) {
			this.variableIndex = variableIndex;
		}
		
		@Override
		public String toModelString() {
			return "lload~" + variableIndex;
		}

		public int getVariableIndex() {
			return variableIndex;
		}
	}
	public static final class LSTORE extends NonSubsetInstruction {
		private int variableIndex;
		
		public LSTORE(int variableIndex) {
			this.variableIndex = variableIndex;
		}
		
		@Override
		public String toModelString() {
			return "lstore~" + variableIndex;
		}

		public int getVariableIndex() {
			return variableIndex;
		}
	}
	
	public static final class LCONST extends NonSubsetInstruction {
		private long constantValue;
		
		public LCONST(long constantValue) {
			this.constantValue = constantValue;
		}
		
		@Override
		public String toModelString() {
			return "lconst~" + constantValue;
		}

		public long getConstantValue() {
			return constantValue;
		}
	}
	
	public static final class IINC extends NonSubsetInstruction {
		private int variableIndex;
		private int constantValue;
		
		public IINC(int variableIndex, int constantValue) {
			this.variableIndex = variableIndex;
			this.constantValue = constantValue;
		}
		
		@Override
		public String toModelString() {
			return "iinc~" + variableIndex + "~" + constantValue;
		}

		public int getVariableIndex() {
			return variableIndex;
		}
		
		public int getConstantValue() {
			return constantValue;
		}
	}
	
	public static final class LADD extends NonSubsetInstruction {
		@Override
		public String toModelString() {
			return "ladd";
		}
	}
	
	public static final class L2I extends NonSubsetInstruction {
		@Override
		public String toModelString() {
			return "ladd";
		}
	}
	
	public static final class I2L extends NonSubsetInstruction {
		@Override
		public String toModelString() {
			return "ladd";
		}
	}

	public static final class LRETURN extends NonSubsetInstruction {
		@Override
		public String toModelString() {
			return "lreturn";
		}
	}
	
	public static abstract class BinOpInstruction extends NonSubsetInstruction {
		public abstract String operator();
	}
	
	public static final class IREM extends BinOpInstruction {

		@Override
		public String operator() {
			return "\\mod";
		}

		@Override
		public String toModelString() {
			return "irem";
		}
		
	}
	
	public static final class ISUB extends BinOpInstruction {

		@Override
		public String operator() {
			return "-";
		}

		@Override
		public String toModelString() {
			return "isub";
		}
		
	}
	
	public static abstract class IfInstruction extends NonSubsetInstruction {
		public abstract String comparisonOperator();
		
		public abstract int getOffset();
	}
	
	public static final class IFEQ extends IfInstruction {
		private int offset;
		
		public IFEQ(int offset) {
			this.offset = offset;
		}
		
		@Override
		public String comparisonOperator() {
			return "=";
		}

		@Override
		public int getOffset() {
			return offset;
		}

		@Override
		public String toModelString() {
			return "ifeq~" + offset;
		}
		
	}
	
	public static final class IFNE extends IfInstruction {
		private int offset;
		
		public IFNE(int offset) {
			this.offset = offset;
		}
		
		@Override
		public String comparisonOperator() {
			return "\\neq";
		}

		@Override
		public int getOffset() {
			return offset;
		}

		@Override
		public String toModelString() {
			return "ifne~" + offset;
		}
		
	}
	
	public static abstract class If_icmpInstruction extends NonSubsetInstruction {
		public abstract String comparisonOperator();
		
		public abstract int getOffset();
	}
	
	public static final class IF_ICMPLT extends If_icmpInstruction {
		private int offset;
		
		public IF_ICMPLT(int offset) {
			this.offset = offset;
		}
		
		@Override
		public String comparisonOperator() {
			return "<";
		}

		@Override
		public int getOffset() {
			return offset;
		}

		@Override
		public String toModelString() {
			return "if\\_icmplt~" + offset;
		}
		
	}
	
	public static final class IF_ICMPGT extends If_icmpInstruction {
		private int offset;
		
		public IF_ICMPGT(int offset) {
			this.offset = offset;
		}
		
		@Override
		public String comparisonOperator() {
			return ">";
		}

		@Override
		public int getOffset() {
			return offset;
		}

		@Override
		public String toModelString() {
			return "if\\_icmpgt~" + offset;
		}
		
	}
	
	public static final class IF_ICMPNE extends If_icmpInstruction {
		private int offset;
		
		public IF_ICMPNE(int offset) {
			this.offset = offset;
		}
		
		@Override
		public String comparisonOperator() {
			return "\\neq";
		}

		@Override
		public int getOffset() {
			return offset;
		}

		@Override
		public String toModelString() {
			return "if\\_icmpne~" + offset;
		}
		
	}
}
