package main;

import javax.safetycritical.AperiodicEventHandler;
import javax.safetycritical.PriorityScheduler;
import javax.safetycritical.Services;

/**
 * Controls the synchronisation between several handlers
 * Each handler must trigger the barrier with its own unique id
 * When all have done so, the Aperiodic Event 
 * passed during initialisation is fired
 * @author ish503
 *
 */
public class Barrier {
	private boolean[] flag;
	private AperiodicEventHandler e;

	/**
	 * Creates a Barrier
	 * @param size the number of handlers of interest
	 * @param launch the event to be called when the barrier is released
	 */
	public Barrier(int size, AperiodicEventHandler launch) {
		/*
		 * Set the ceiling priority for this shared object
		 * used by Priority Ceiling Emulation protocol
		 * FireHandler is at max priority
		 */
		Services.setCeiling(this, 
				PriorityScheduler.instance().getMaxPriority());
		
		this.flag = new boolean[size];
		this.e = launch;
	}

	/**
	 * Checks if all handlers have triggered the barrier
	 * @return true if all handlers have triggered the barrier
	 */
	public synchronized boolean isOkToFire() {
		boolean okToFire = true;
		
		for (int i = 0; i < this.flag.length; i++) {
			if (this.flag[i] == false) {
				okToFire = false;
			}
		}
		
		return okToFire;
	}
	
	/**
	 * Triggers the barrier for the specified handler id
	 * @param id
	 */
	public synchronized void trigger(int id) {
		this.flag[id] = true;
		
		if (isOkToFire()) 
			{
				this.e.release();
				this.reset();
			}
	}
	
	/**
	 * Checks if the handler has already triggered the barrier
	 * @param id the unique handler id
	 * @return true if the handler has already triggered the barrier
	 */
	public synchronized boolean isAlreadyTriggered(int id) {
		return this.flag[id];
	}

	/**
	 * Resets the barrier.
	 * The event to be fired during the next barrier release
	 * is not changed.
	 */
	private synchronized void reset() {
		for (int i = 0; i < this.flag.length; i++)
		{
			this.flag[i] = false; 
		}		
	}
}
