with ada.interrupts.names; use ada.interrupts;
with system; use system;
with system.storage_elements; use system.storage_elements;
package body adc_device_driver is
  bits_in_word : constant := 16;
  word : constant := 2; -- bytes in word
  type flag is (down, set);

  type control_register is
  record
    ad_start : flag;
    ienable  : flag;
    done     : flag;
    ch       : channel;
    error    : flag;
  end record;

  for control_register use   
    -- specifies the layout of the control register
    record
      ad_start at 0*word range 0..0;
      ienable  at 0*word range 6..6;
      done     at 0*word range 7..7;
      ch       at 0*word range 8..13;
      error    at 0*word range 15..15;
  end record;

  for control_register'size use bits_in_word; 
    -- the register is 16-bits long
  for control_register'alignment use word; 
    -- on a word boundary
  for control_register'bit_order use low_order_first;

  type data_register is range 0 .. max_measure;
  for data_register'size use bits_in_word;  
    -- the register is 16-bits long

  contr_reg_addr : constant address := to_address(8#150002#);
  data_reg_addr : constant address  := to_address(8#150000#);
  adc_priority : constant interrupt_priority := interrupt_priority'first;
  control_reg : aliased control_register;
      -- aliased indicates that pointers are used to access it
  for control_reg'address use contr_reg_addr; 
      -- specifies the address of the control register
  data_reg : aliased data_register;
  for data_reg'address use data_reg_addr; 
      -- specifies the address of the data register

  protected type interrupt_interface(int_id : interrupt_id;
                 cr : access control_register;
                 dr : access data_register) is
    entry read(chan : channel; m : out measurement);
  private
    entry done(chan : channel; m : out measurement);
    procedure handler;
    pragma attach_handler(handler, int_id);
    pragma interrupt_priority(adc_priority);
      -- see section \ref{ceiling} for discussion on priorities
    interrupt_occurred : boolean := false;
    next_request : boolean := true;
  end interrupt_interface;

  adc_interface : interrupt_interface(names.adc,
                  control_reg'access, 
                  data_reg'access);
    -- this assumes that `adc' is registered as an 
    -- interrupt_id in ada.interrupts.names
    -- 'access gives the address of the object

  protected body interrupt_interface is

    entry read(chan : channel; m : out measurement) 
          when next_request is
      shadow_register : control_register;
    begin
      shadow_register := (ad_start => set, ienable => set, 
            done => down, ch   => chan, error => down);
      cr.all := shadow_register;
      interrupt_occurred := false;
      next_request := false;
      requeue done;
    end read;

    procedure handler is
    begin
      interrupt_occurred := true;
    end handler;

    entry done(chan : channel; m : out measurement) 
                             when interrupt_occurred is
    begin
      next_request := true;
      if cr.done = set and cr. error = down then
            m := measurement(dr.all);
      else
        raise conversion_error; 
     end if;
    end done;
  end interrupt_interface;

  procedure read(ch : channel; m : out measurement) is
  begin
    for i in 1..3 loop
      begin
        adc_interface.read(ch,m);
        return;
      exception
        when conversion_error => null;
      end;
    end loop;
    raise conversion_error;
  end read;

end adc_device_driver;
