package uk.ac.york.tokeneer;

import java.util.concurrent.atomic.AtomicReference;

import javax.realtime.AperiodicParameters;
import javax.realtime.HighResolutionTime;
import javax.realtime.PriorityParameters;
import javax.safetycritical.PriorityScheduler;
import javax.safetycritical.StorageConfigurationParameters;

import uk.ac.york.tokeneer.realworld.TokenReader;

/**
 * {@link AbstractTimeout} implementation for specifying the window in which a 
 * Token can be removed for a successfully validated user. If the token is not 
 * removed in this window, then the user is not allowed entry into the enclave.
 * 
 * @author Jon Co
 * 
 */
public class TokenRemovalTimeout extends AbstractTimeout {

	private static final String TAG = "TokenRemovalTimeout";

	// Current Stage of the UserEntryOperation
	private final AtomicReference<UserEntryState> state;
	// Token Reader peripheral
	private final TokenReader tokenReader;

	/**
	 * Default Constructor
	 * 
	 * @param timeout
	 *            Duration when the token can be removed.
	 * @param state
	 *            The stage of the user entry operation.
	 */
	// @formatter:off
	public TokenRemovalTimeout(
			final HighResolutionTime timeout,
			final AtomicReference<UserEntryState> state,
			final TokenReader tokenReader) {
		super(
				timeout,
				new PriorityParameters(PriorityScheduler.instance().getMaxPriority()), 
				new AperiodicParameters(),
				new StorageConfigurationParameters(4096, 4096, 4096), 
				4096);
		this.state = state;
		this.tokenReader = tokenReader;
	}
	// @formatter:on

	@Override
	public void handleEvent() {
		Log.d(TAG, "Token Removal Timeout Expired");
		final boolean failed = this.state.compareAndSet(
				UserEntryState.WAITING_TOKEN_REMOVAL_SUCCESS,
				UserEntryState.WAITING_TOKEN_REMOVAL_FAILURE);
		if (failed) {
			Log.i(TAG, "Failed to remove token in time, entry revoked");
		}
	}

	/**
	 * Start the timeout for Token Removal.
	 */
	public synchronized void startTimeout() {
		Log.d(TAG, "Token Removal Timeout Started");
		super.startTimeout();
	}

	@Override
	public synchronized boolean deschedule() {
		Log.d(TAG, "Token Removal Timeout Stopped");
		return super.deschedule();
	}
	
}
