package uk.ac.york.tokeneer;

import java.util.concurrent.atomic.AtomicReference;

import javax.realtime.AperiodicParameters;
import javax.realtime.PriorityParameters;
import javax.safetycritical.AperiodicEventHandler;
import javax.safetycritical.PriorityScheduler;
import javax.safetycritical.StorageConfigurationParameters;

import uk.ac.york.tokeneer.realworld.Latch;
import uk.ac.york.tokeneer.realworld.TokenReader;

/**
 * Handler for processing Token Removal.
 * 
 * @author Jon Co
 * 
 */
public class TokenRemovalHandler extends AperiodicEventHandler {

	private static final String TAG = "TokenRemoval";

	private final AtomicReference<UserEntryState> state;
	private final Latch latch;
	private final TokenReader tokenReader;
	
	private final LatchTimeout latchTimeout;
	private final AlarmTimeout alarmTimeout;

	// @formatter:off
	public TokenRemovalHandler(
			final AtomicReference<UserEntryState> state,
			final Latch latch,
			final TokenReader tokenReader,
			final LatchTimeout latchTimeout,
			final AlarmTimeout alarmTimeout) {
		super(new PriorityParameters(PriorityScheduler.instance().getNormPriority()), 
				new AperiodicParameters(),
				new StorageConfigurationParameters(4096, 4096, 4096), 
				4096);
		// @formatter:on
		
		this.state = state;
		this.latch = latch;
		this.tokenReader = tokenReader;
		this.latchTimeout = latchTimeout;
		this.alarmTimeout = alarmTimeout;
	}

	@Override
	public void handleEvent() {
		// Token not actually removed, event fired prematurely
		if (!isTokenRemoved()) {
			return;
		}
		
		final UserEntryState currentState = this.state.get();
		
		if (currentState == UserEntryState.WAITING_TOKEN_REMOVAL_SUCCESS) {
			// User authed
			Log.i(TAG, "Authentication successful, user allowed entry");
			this.latch.unlock();
			this.latchTimeout.startTimeout();
			this.alarmTimeout.startTimeout();
			this.state.set(UserEntryState.IDLE);
		} else if (currentState == UserEntryState.WAITING_TOKEN_REMOVAL_FAILURE) {
			// User was not authenticated, block entry
			Log.i(TAG, "Authentication unsuccessful, user denined entry");
			this.state.set(UserEntryState.IDLE);
		} else {
			// Premature token teaar
			Log.i(TAG, "Token removed prematurely, cancel entry operation");
			this.state.set(UserEntryState.IDLE);
		}
	}

	private boolean isTokenRemoved() {
		return this.tokenReader.pollToken() == null;
	}
}
