package uk.ac.york.tokeneer;

import javax.realtime.PeriodicParameters;
import javax.realtime.PriorityParameters;
import javax.realtime.RelativeTime;
import javax.safetycritical.PeriodicEventHandler;
import javax.safetycritical.PriorityScheduler;
import javax.safetycritical.StorageConfigurationParameters;

import uk.ac.york.tokeneer.realworld.Alarm;
import uk.ac.york.tokeneer.realworld.AlarmState;
import uk.ac.york.tokeneer.realworld.Door;
import uk.ac.york.tokeneer.realworld.DoorState;
import uk.ac.york.tokeneer.realworld.Latch;
import uk.ac.york.tokeneer.realworld.LatchState;

/**
 * {@link PeriodicEventHandler} implementation that polls the state of the 
 * Door, Latch and Alarm from the real world.
 * 
 * If the state of the system is found to be unsecure, then the alarm timeout
 * will be started.
 * 
 * @author Jon Co
 * 
 */
public class DoorLatchAlarmUpdateHandler extends PeriodicEventHandler {

	private static final String TAG ="DoorLatchAlarmUpdate";

	// Controllers for various real world entities
	private final Alarm alarm;
	private final Door door;
	private final Latch latch;

	/**
	 * Default Constructor.
	 * 
	 * @param updateInterval
	 *            The update interval, must be specified.
	 * @param alarmTimeout
	 *            Handler of the alarm timeout mechanism.
	 * @param controller
	 *            Interface to real world controllers.
	 * @param state
	 *            Internal state to be updated.
	 */
	
	// TODO Probably make this into a reverse-builder constructor
	// @formatter:off
	public DoorLatchAlarmUpdateHandler(
			final RelativeTime updateInterval,
			final Alarm alarm,
			final Door door,
			final Latch latch) {
		
		super(
				new PriorityParameters(PriorityScheduler.instance().getNormPriority()), 
				new PeriodicParameters(updateInterval),
				new StorageConfigurationParameters(1024, 1024, 1024), 65523);
		// @formatter:on
		
		this.alarm = alarm;
		this.door = door;
		this.latch = latch;
	}

	@Override
	public void handleEvent() {
		Log.d(TAG, "Reading Door Security...");

		// Update internal state representation
		final AlarmState alarmState = this.alarm.pollState();
		final DoorState doorState = this.door.pollState();
		final LatchState latchState = this.latch.pollState();
		
		Log.i(TAG, 
				"Alarm: " + alarmState
				+ ", Door: " + doorState
				+ ", Latch: " + latchState);

		// If alarm is still sounding check if system has been resecured.
		if (alarmState == AlarmState.ON) {
			if (doorState == DoorState.CLOSED 
					&& latchState == LatchState.LOCKED) {
				this.alarm.deactivate();
			} 
		}
	}

}
